package gnu.crypto.sasl.srp;

// ----------------------------------------------------------------------------
// $Id: SRP.java,v 1.2 2003/05/30 13:05:57 raif Exp $
//
// Copyright (C) 2003 Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to the
//
//    Free Software Foundation Inc.,
//    59 Temple Place - Suite 330,
//    Boston, MA 02111-1307
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
// ----------------------------------------------------------------------------

import gnu.crypto.hash.HashFactory;
import gnu.crypto.hash.IMessageDigest;
import gnu.crypto.key.IKeyPairGenerator;
import gnu.crypto.key.srp6.SRPKeyPairGenerator;
import gnu.crypto.util.Util;

import java.math.BigInteger;
import java.util.HashMap;

/**
 * <p>A Factory class that returns SRP Singletons that know all SRP-related
 * mathematical computations and protocol-related operations for both the
 * client- and server-sides.</p>
 *
 * @version $Revision: 1.2 $
 */
public final class SRP {

   // Constants and variables
   // --------------------------------------------------------------------------

   private static final BigInteger THREE = BigInteger.valueOf(3L);

   /** The map of already instantiated SRP algorithm instances. */
   private static final HashMap algorithms = new HashMap();

   /** The underlying message digest algorithm used for all SRP calculations. */
   private IMessageDigest mda;

   private IKeyPairGenerator kpg;

   // Constructor(s)
   // --------------------------------------------------------------------------

   /** Trivial private constructor to enforce Singleton pattern. */
   private SRP(IMessageDigest mda) {
      super();

      this.mda = mda;
      kpg = new SRPKeyPairGenerator();
   }

   // Class methods
   // -------------------------------------------------------------------------

   /**
    * Returns an instance of this object that uses the designated message
    * digest algorithm as its digest function.
    *
    * @return an instance of this object for the designated digest name.
    */
   public static synchronized SRP instance(String mdName) {
      if (mdName != null) {
         mdName = mdName.trim().toLowerCase();
      }
      if (mdName == null || mdName.equals("")) {
         mdName = SRPRegistry.SRP_DEFAULT_DIGEST_NAME;
      }
      SRP result = (SRP) algorithms.get(mdName);
      if (result == null) {
         IMessageDigest mda = HashFactory.getInstance(mdName);
         result = new SRP(mda);
         algorithms.put(mdName, result);
      }
      return result;
   }

   private static final byte[] xor(byte[] b1, byte[] b2, int length) {
      byte[] result = new byte[length];
      for(int i = 0; i < length; ++i) {
         result[i] = (byte)(b1[i] ^ b2[i]);
      }
      return result;
   }

   // Instance methods
   // -------------------------------------------------------------------------

   /** @return the message digest algorithm name used by this instance. */
   public String getAlgorithm() {
      return mda.name();
   }

//   /**
//    * <p>Generates the server-side private and public key-pair, given the shared
//    * public modulus <code>n</code> and the generator <code>g</code>.</p>
//    *
//    * @param N the shared public modulus.
//    * @param g the field generator.
//    * @param v the user's verifier.
//    * @return a {@link KeyPair} that contains both the private (b) as well as
//    * the public (B) keys of the server in the SRP exchange.
//    */
//   public KeyPair generateServerKeyPair(BigInteger N, BigInteger g, BigInteger v) {
////      SRPAlgorithm.checkParams(N, g);
////
////      byte[] bBytes = new byte[(N.bitLength() + 7) / 8];
////      BigInteger b, B;
////      do {
////         do {
////            PRNG.nextBytes(bBytes);
////            b = new BigInteger(1, bBytes);
////         } while (b.compareTo(ONE) <= 0);
////         B = THREE.multiply(v).add(g.modPow(b, N)).mod(N);
////      } while (B.compareTo(ZERO) == 0 || B.compareTo(N) >= 0);
////
////      KeyPair result = new KeyPair(
////            new SRPPublicKey( new BigInteger[] { N, g, B }),
////            new SRPPrivateKey(new BigInteger[] { N, g, b }));
////      return result;
//      HashMap attributes = new HashMap();
////      attributes.put(SRPKeyPairGenerator.HASH_ALGORITHM, mda);
//      attributes.put(SRPKeyPairGenerator.SHARED_MODULUS, N);
//      attributes.put(SRPKeyPairGenerator.GENERATOR, g);
//      attributes.put(SRPKeyPairGenerator.USER_VERIFIER, v);
//      kpg.setup(attributes);
//
//      return kpg.generate();
//   }

//   /**
//    * <p>Generates the client-side private and public key-pair, given the shared
//    * public modulus <code>n</code> and the generator <code>g</code>.</p>
//    *
//    * @param N the shared public modulus.
//    * @param g the field generator.
//    * @return a {@link KeyPair} that contains both the private (a) as well as
//    * the public (A) keys of the client in the SRP exchange.
//    */
//   public KeyPair
////   generateClientKeyPair(BigInteger N, BigInteger g, BigInteger B, BigInteger x) {
//   generateClientKeyPair(BigInteger N, BigInteger g) {
////      SRPAlgorithm.checkParams(N, g);
////
////      byte[] aBytes = new byte[(N.bitLength() + 7) / 8];
////      BigInteger v = g.modPow(x, N);
////      BigInteger a, A, avu;
////      do {
////         do {
////            PRNG.nextBytes(aBytes);
////            a = new BigInteger(1, aBytes);
////         } while (a.compareTo(ONE) <= 0);
////         A = g.modPow(a, N);
////         avu = A.multiply(v.modPow(uValue(A, B), N)).mod(N);
////      } while (avu.compareTo(ONE) <= 0 || avu.compareTo(N.subtract(ONE)) == 0);
////
////      KeyPair result = new KeyPair(
////            new SRPPublicKey( new BigInteger[] { N, g, A } ),
////            new SRPPrivateKey(new BigInteger[] { N, g, a } ));
////      return result;
//      HashMap attributes = new HashMap();
////      attributes.put(SRPKeyPairGenerator.HASH_ALGORITHM, mda);
//      attributes.put(SRPKeyPairGenerator.SHARED_MODULUS, N);
//      attributes.put(SRPKeyPairGenerator.GENERATOR, g);
////      attributes.put(SRPKeyPairGenerator.SERVER_PUBLIC_KEY, B);
////      attributes.put(SRPKeyPairGenerator.HASHED_USER_NAME_AND_PASSWORD, x);
//      kpg.setup(attributes);
//
//      return kpg.generate();
//   }

//   /**
//    * Used by the server-side in the SRP exchange. This method generates a
//    * session key suitable for symmetric encryption of the exchange between the
//    * client and server after authentication is successfully concluded.
//    *
//    * @param kp the server's key-pair.
//    * @param A the client's public key.
//    * @return the session key <code>K</code>.
//    */
////   public byte[] generateServerK(KeyPair kp, BigInteger A, BigInteger v) {
//   public byte[] generateServerK(KeyPair kp, BigInteger A) {
//      BigInteger B = ((SRPPublicKey) kp.getPublic()).getY();
//      BigInteger N = ((SRPPublicKey) kp.getPublic()).getN();
//      BigInteger b = ((SRPPrivateKey) kp.getPrivate()).getX();
//      BigInteger v = ((SRPPrivateKey) kp.getPrivate()).getV();
//
//      // compute u = H(A | B)
//      BigInteger u = uValue(A, B);
//
//      // compute S = ((A * (v ** u)) ** b) % N
//      BigInteger S = A.multiply(v.modPow(u, N)).modPow(b, N);
//
//      // compute K = H(S) (as of rev 08)
//      IMessageDigest hash = (IMessageDigest) mda.clone();
//      byte[] sBytes = Util.trim(S);
//      hash.update(sBytes, 0, sBytes.length);
//      return hash.digest();
//   }

//   /**
//    * Used by the client-side in the SRP exchange. This method generates a
//    * session key suitable for symmetric encryption of the exchange between the
//    * client and server after authentication is successfully concluded.
//    *
//    * @param kp the client's key-pair.
//    * @return the session key <code>K</code>.
//    */
//   public byte[] generateClientK(KeyPair kp, BigInteger B, BigInteger x) {
////   public byte[] generateClientK(KeyPair kp) {
//      BigInteger A = ((SRPPublicKey) kp.getPublic()).getY();
//      BigInteger N = ((SRPPublicKey) kp.getPublic()).getN();
//      BigInteger g = ((SRPPublicKey) kp.getPublic()).getG();
//      BigInteger a = ((SRPPrivateKey) kp.getPrivate()).getX();
////      BigInteger B = ((SRPPrivateKey) kp.getPrivate()).getB();
////      BigInteger x = ((SRPPrivateKey) kp.getPrivate()).getx();
//
//      BigInteger v = g.modPow(x, N);
//
//      // compute u = H(A | B)
//      BigInteger u = uValue(A, B);
//
//      // compute S = ((B - (3 * (g ** x))) ** (a + (u * x))) % N
//      // compute S = ((B - (3 * v)) ** (a + (u * x))) % N
//      BigInteger S = B.subtract(THREE.multiply(v)).modPow(a.add(u.multiply(x)), N);
//
//      // compute K = H(S) (as of rev 08)
//      IMessageDigest hash = (IMessageDigest) mda.clone();
//      byte[] sBytes = Util.trim(S);
//      hash.update(sBytes, 0, sBytes.length);
//      return hash.digest();
//   }

   // Message Digest algorithm related methods --------------------------------

   /**
    * Returns a new instance of the SRP message digest algorithm --which is
    * SHA-160 by default, but could be anything else provided the proper
    * conditions as specified in the SRP specifications.
    *
    * @return a new instance of the underlying SRP message digest algorithm.
    * @throws RuntimeException if the implementation of the message digest
    * algorithm does not support cloning.
    */
   public IMessageDigest newDigest() {
      return (IMessageDigest) mda.clone();
   }

   /**
    * Convenience method to return the result of digesting the designated input
    * with a new instance of the SRP message digest algorithm.
    *
    * @param src some bytes to digest.
    * @return the bytes constituting the result of digesting the designated
    * input with a new instance of the SRP message digest algorithm.
    */
   public byte[] digest(byte[] src) {
      IMessageDigest hash = (IMessageDigest) mda.clone();
      hash.update(src, 0, src.length);
      return hash.digest();
   }

   /**
    * Convenience method to return the result of digesting the designated input
    * with a new instance of the SRP message digest algorithm.
    *
    * @param src a String whose bytes (using the default platform encoding) are
    * to be digested.
    * @return the bytes constituting the result of digesting the designated
    * input with a new instance of the SRP message digest algorithm.
    */
   public byte[] digest(String src) {
      return digest(src.getBytes());
   }

   // Other methods -----------------------------------------------------------

   /**
    * Convenience method to XOR N bytes from two arrays; N being the output size
    * of the SRP message digest algorithm.
    *
    * @param a the first byte array.
    * @param b the second one.
    * @return N bytes which are the result of the XOR operations on the first N
    * bytes from the designated arrays. N is the size of the SRP message digest
    * algorithm; eg. 20 for SHA-160.
    */
   public byte[] xor(byte[] a, byte[] b) {
      return xor(a, b, mda.hashSize());
   }

//   public byte[] userHash(String U, String p) {
//      return digest(U + ":" + p);
//   }

   public byte[] generateM2(BigInteger A, byte[] M, byte[] K, String U, String I,
                            String o, String sid, int ttl, byte[] cIV, byte[] sIV) {
      IMessageDigest hash = (IMessageDigest) mda.clone();
      byte[] b;
      b = Util.trim(A);
      hash.update(b, 0, b.length);
      hash.update(M, 0, M.length);
      hash.update(K, 0, K.length);
      b = digest(U);
      hash.update(b, 0, b.length);
      b = digest(I);
      hash.update(b, 0, b.length);
      b = digest(o);
      hash.update(b, 0, b.length);
      b = sid.getBytes();
      hash.update(b, 0, b.length);
      hash.update((byte)(ttl >>> 24));
      hash.update((byte)(ttl >>> 16));
      hash.update((byte)(ttl >>>  8));
      hash.update((byte) ttl);
      hash.update(cIV, 0, cIV.length);
      hash.update(sIV, 0, sIV.length);

      return hash.digest();
   }

   public byte[] generateM1(BigInteger N, BigInteger g, String U, byte[] s,
                            BigInteger A, BigInteger B, byte[] K,
                            String I, String L) {
      IMessageDigest hash = (IMessageDigest) mda.clone();
      byte[] b;
      b = xor(digest(Util.trim(N)), digest(Util.trim(g)));
      hash.update(b, 0, b.length);
      b = digest(U);
      hash.update(b, 0, b.length);
      hash.update(s, 0, s.length);
      b = Util.trim(A);
      hash.update(b, 0, b.length);
      b = Util.trim(B);
      hash.update(b, 0, b.length);
      hash.update(K, 0, K.length);
      b = digest(I);
      hash.update(b, 0, b.length);
      b = digest(L);
      hash.update(b, 0, b.length);

      return hash.digest();
   }

   public byte[] generateKn(byte[] K, byte[] cn, byte[] sn) {
      IMessageDigest hash = (IMessageDigest) mda.clone();
      hash.update(K, 0, K.length);
      hash.update(cn, 0, cn.length);
      hash.update(sn, 0, sn.length);

      return hash.digest();
   }

   // helper methods ----------------------------------------------------------

//   private BigInteger uValue(BigInteger A, BigInteger B) {
//      IMessageDigest hash = (IMessageDigest) mda.clone();
//      byte[] b;
//      b = Util.trim(A);
//      hash.update(b, 0, b.length);
//      b = Util.trim(B);
//      hash.update(b, 0, b.length);
//
//      return new BigInteger(1, hash.digest());
//   }
}
