/*
 * gaia - opensource 3D interface to the planet
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "gefetch_internal.h"

/**
 * Internal function used in curl
 *
 * Is called when chunk of data is received. Thus, appends it to
 * buffer storing curently downloaded file (handle->currentdata)
 *
 * See libCURL documentation for description.
 */
size_t gefetch_curl_writefunction(char *ptr, size_t size, size_t nmemb, void *stream) {
	gefetch *handle = (gefetch *)stream;

	size_t newsize = handle->currentsize + size*nmemb;

	unsigned char *newdata = (unsigned char*)realloc((void*)handle->currentdata, newsize);

	if (newdata == 0)
		return 0;

	handle->currentdata = newdata;

	memcpy(&handle->currentdata[handle->currentsize], ptr, size*nmemb);

	handle->currentsize = newsize;

	return size * nmemb;
}

/**
 * Internal function to form and execute single HTTP request
 *
 * @param method request method (METHOD_POST | METHOD_GET)
 * @param url full URL for request
 * @param postdata (POST only) pointer to data to send to server
 * @param postsize (POST only) length of postdata
 * @param code address of long to store HTTP response code to
 */
gefetch_error gefetch_do_http_request(gefetch *handle, int method, const char *url, void *postdata, size_t postsize, long *code) {
	CURLcode res;

	/* free last downloaded file, if any */
	if (handle->currentdata)
		free(handle->currentdata);

	handle->currentdata = 0;
	handle->currentsize = 0;

	/* drop all curl headers */
	if (handle->curlheaders) {
		curl_slist_free_all(handle->curlheaders);
		handle->curlheaders = 0;
	}

	/* reset curl */
	curl_easy_reset(handle->curl);

	/* init request */
	if (method == METHOD_POST) {
		if ((res = curl_easy_setopt(handle->curl, CURLOPT_POSTFIELDS, postdata)) != CURLE_OK)
			return GEFETCH_CURL_INIT_FAILED;
		if ((res = curl_easy_setopt(handle->curl, CURLOPT_POSTFIELDSIZE, postsize)) != CURLE_OK)
			return GEFETCH_CURL_INIT_FAILED;
	} else {
		if ((res = curl_easy_setopt(handle->curl, CURLOPT_HTTPGET, 1)) != CURLE_OK)
			return GEFETCH_CURL_INIT_FAILED;
	}               

	/* init headers */
	struct curl_slist *tmplist;

	/* Accept: */
	if ((tmplist = curl_slist_append(handle->curlheaders, "Accept: text/plain, text/html, text/xml, text/xml-external-parsed-entity, application/octet-stream, application/vnd.google-earth.kml+xml, application/vnd.google-earth.kmz, image/*")) == 0)
		return GEFETCH_CURL_INIT_FAILED;

	handle->curlheaders = tmplist;

	/* Cache-Control: */
	if ((tmplist = curl_slist_append(handle->curlheaders, "Cache-Control: no-store")) == 0)
		return GEFETCH_CURL_INIT_FAILED;

	handle->curlheaders = tmplist;

	/* Content-Type: */
	if ((tmplist = curl_slist_append(handle->curlheaders, "Content-Type: application/octet-stream")) == 0)
		return GEFETCH_CURL_INIT_FAILED;

	handle->curlheaders = tmplist;

	/* Connection: */
	if ((tmplist = curl_slist_append(handle->curlheaders, "Connection: Keep-Alive")) == 0)
		return GEFETCH_CURL_INIT_FAILED;

	handle->curlheaders = tmplist;

	/* Connection: */
	if (handle->sid != 0) {
		char buf[1024];

		if (snprintf(buf, sizeof(buf), "Cookie: $Version=\"0\"; SessionId=\"%s\"; State=\"1\"", handle->sid) >= sizeof(buf))
			return GEFETCH_SMALL_BUFFER;

		if ((tmplist = curl_slist_append(handle->curlheaders, buf)) == 0)
			return GEFETCH_CURL_INIT_FAILED;

		handle->curlheaders = tmplist;
	}

	/* options */
	if ((res = curl_easy_setopt(handle->curl, CURLOPT_HTTPHEADER, handle->curlheaders)) != CURLE_OK)
		return GEFETCH_CURL_INIT_FAILED;

	if ((res = curl_easy_setopt(handle->curl, CURLOPT_USERAGENT, "kh_lt/LT3.0.0762")) != CURLE_OK)
		return GEFETCH_CURL_INIT_FAILED;

	if ((res = curl_easy_setopt(handle->curl, CURLOPT_URL, url)) != CURLE_OK)
		return GEFETCH_CURL_INIT_FAILED;

	if ((res = curl_easy_setopt(handle->curl, CURLOPT_TIMEOUT, 30)) != CURLE_OK)
		return GEFETCH_CURL_INIT_FAILED;

	if ((res = curl_easy_setopt(handle->curl, CURLOPT_NOSIGNAL, 1)) != CURLE_OK)
		return GEFETCH_CURL_INIT_FAILED;

	if ((res = curl_easy_setopt(handle->curl, CURLOPT_WRITEFUNCTION, gefetch_curl_writefunction)) != CURLE_OK)
		return GEFETCH_CURL_INIT_FAILED;

	if ((res = curl_easy_setopt(handle->curl, CURLOPT_WRITEDATA, handle)) != CURLE_OK)
		return GEFETCH_CURL_INIT_FAILED;

	if ((res = curl_easy_perform(handle->curl)) != CURLE_OK)
		return GEFETCH_CURL_PERFORM_FAILED;

	if ((res = curl_easy_getinfo(handle->curl, CURLINFO_RESPONSE_CODE, code)) != CURLE_OK)
		return GEFETCH_CURL_PERFORM_FAILED;

	return GEFETCH_OK;
}
