/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef __CHEROKEE_SERVER_H__
#define __CHEROKEE_SERVER_H__

#include "common.h"

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#include <unistd.h>
#include <sys/types.h>

#include "list.h"
#include "fdpoll.h"
#include "handler_table.h"
#include "virtual_server.h"
#include "encoder_table.h"
#include "logger_table.h"
#include "thread.h"
#include "module_loader.h"
#include "icons.h"


typedef struct {
	/* Current time
	 */
	time_t             start_time;
	time_t             bogo_now;
	cherokee_buffer_t *bogo_now_string;

	int    wanna_exit;
	char  *config_filename;
	
	/* Virtual servers
	 */
	struct list_head           vservers;
	cherokee_table_t          *vservers_ref;
	cherokee_virtual_server_t *vserver_default;
	
	/* Threads
	 */
	cherokee_thread_t *main_thread;
	struct list_head   thread_list;

	/* Modules
	 */
	cherokee_module_loader_t *loader;
	cherokee_encoder_table_t *encoders;
	cherokee_logger_table_t  *loggers;

	/* Icons
	 */
	cherokee_icons_t *icons;

	/* Default index files
	 */
	list_t index_list;

	/* Logging
	 */
	int    log_flush_elapse;
	time_t log_flush_next;

	/* Main socket
	 */
	int   socket;
	int   socket_tls;
#ifdef HAVE_PTHREAD
	pthread_mutex_t accept_mutex;
#endif

	/* Configuration
	 */
	unsigned short port;
	unsigned short port_tls;

	int   ipv6;
	char *listen_to;	
	int   fdwatch_msecs;

	int   ncpus;
	int   thread_num;

	int                timeout;
	cherokee_buffer_t *timeout_header;

	int   hideversion;
	int   hideservername;

	int   keepalive;
	int   keepalive_max;

	uid_t  user;
	uid_t  user_orig;
	gid_t  group;
	gid_t  group_orig;

	char  *chroot;
	int    chrooted;

	char  *mimetypes_file;
	char  *icons_file;

	char                           *userdir;              /* Eg: public_html */
	cherokee_handler_table_entry_t *userdir_handler;     

} cherokee_server_t;


#define SRV(x) ((cherokee_server_t *)(x))


ret_t cherokee_server_new   (cherokee_server_t **srv);
ret_t cherokee_server_init  (cherokee_server_t  *srv);
ret_t cherokee_server_clean (cherokee_server_t  *srv);
ret_t cherokee_server_free  (cherokee_server_t  *srv);

void  cherokee_server_step            (cherokee_server_t *srv);
void  cherokee_server_set_min_latency (cherokee_server_t *srv, int msecs);
ret_t cherokee_server_daemonize       (cherokee_server_t *srv);

ret_t cherokee_server_read_config_file    (cherokee_server_t *srv, char *filename);
ret_t cherokee_server_read_config_string  (cherokee_server_t *srv, char *string);

ret_t cherokee_server_get_active_conns   (cherokee_server_t *srv, int *num);
ret_t cherokee_server_get_reusable_conns (cherokee_server_t *srv, int *num);
ret_t cherokee_server_get_total_traffic  (cherokee_server_t *srv, unsigned long int *rx, unsigned long int *tx);

/* System signal callback
 */
ret_t cherokee_server_handle_HUP (cherokee_server_t *srv);


#endif /* __CHEROKEE_SERVER_H__ */
