/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/*
** Copyright (c) 1999,2000 by Jef Poskanzer <jef@acme.com>
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
*/

/* Useful documentation:
 *
 * http://www.opengroup.org/onlinepubs/007904975/functions/poll.html
 */

#include "common.h"
#include "fdpoll.h"

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#ifdef HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#else
#include <resource.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif


#ifdef HAVE_PTHREAD
# define LOCK_WRITER(fdp) pthread_rwlock_wrlock (&fdp->rwlock)
# define LOCK_READER(fdp) pthread_rwlock_rdlock (&fdp->rwlock)
# define UNLOCK(fdp)      pthread_rwlock_unlock (&fdp->rwlock)
#else
# define LOCK_WRITER(fdp)
# define LOCK_READER(fdp)
# define UNLOCK(fdp)
#endif

#define POLL_ERROR  (POLLHUP | POLLERR)
#define POLL_READ   (POLLIN | POLLPRI | POLL_ERROR)
#define POLL_WRITE  (POLLOUT | POLL_ERROR)


static inline int
get_fd_limit (void)
{
	   int nfiles;
	   struct rlimit rl;

	   if (getrlimit (RLIMIT_NOFILE, &rl) == 0)
	   {
			 nfiles = rl.rlim_cur;

			 if (rl.rlim_max == RLIM_INFINITY) {
				 /* arbitrary */
				 rl.rlim_cur = 8192;  
			 } else if (rl.rlim_max > rl.rlim_cur) {
				 rl.rlim_cur = rl.rlim_max;
			 }
			 
			 if (setrlimit( RLIMIT_NOFILE, &rl ) == 0) {
				    nfiles = rl.rlim_cur;
			 }

			 return nfiles;
	   }

	   
#ifdef USE_SELECT
	   /* If we use select(), then we must limit ourselves to FD_SETSIZE. 
	    */
	   if (nfiles > FD_SETSIZE) {
		   nfiles = FD_SETSIZE;
	   }
#endif

	   return getdtablesize();
}



/***********************************************************************/
/* epoll:                                                              */
/*                                                                     */
/* #include <sys/epoll.h>                                              */
/*                                                                     */
/* int epoll_ctl(int epfd, int op, int fd, struct epoll_event *event); */
/*                                                                     */
/* Info:                                                               */
/* http://www.xmailserver.org/linux-patches/nio-improve.html           */
/*                                                                     */
/* As you can see in /usr/src/linux/fs/eventpoll.c, the epoll          */
/* implementation is thread-safe, so we don't need to use mutex        */
/* as using poll()                                                     */
/*                                                                     */
/***********************************************************************/
#ifdef USE_EPOLL


ret_t 
fdpoll_epoll_get_method (cherokee_fdpoll_t  *fdp, char **name)
{
	static char *method = "epoll";

	*name = method;
	return ret_ok;
}

ret_t
fdpoll_epoll_new (cherokee_fdpoll_t **fdp)
{
	   CHEROKEE_NEW_STRUCT (n, fdpoll);

	   /* Look for max fd limit
	    */
	   n->nfiles       = get_fd_limit();
	   n->ep_readyfds  = 0;
	   n->ep_events    = (struct epoll_event *) malloc (sizeof(struct epoll_event) * n->nfiles);
	   n->epoll_rs2idx = (int *) malloc (sizeof(int) * n->nfiles);
	   n->epoll_idx2rs = (int *) malloc (sizeof(int) * n->nfiles);

	   n->ep_fd = epoll_create (n->nfiles + 1);
	   if (n->ep_fd < 0) {
		   PRINT_ERROR ("Fatal Error: Couldn't get /dev/epoll descriptor: epoll_create(%d) -> %d\n", n->nfiles+1, n->ep_fd);
		   exit (EXIT_EPOLL_CREATE);
	   }

	   return_if_fail (n->ep_events,    ret_nomem);
	   return_if_fail (n->epoll_rs2idx, ret_nomem);
	   return_if_fail (n->epoll_idx2rs, ret_nomem);

#ifdef HAVE_PTHREAD
	   pthread_rwlock_init(&n->rwlock, NULL); 
#endif

	   /* Return the object
	    */
	   *fdp = n;

	   return ret_ok;
}

ret_t
fdpoll_epoll_free (cherokee_fdpoll_t *fdp)
{
	close (fdp->ep_fd);

	free (fdp->ep_events);
	free (fdp->epoll_rs2idx);
	free (fdp->epoll_idx2rs);
	
	free (fdp);
	
	return ret_ok;
}

ret_t
fdpoll_epoll_add (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	struct epoll_event ev;

	ev.data.fd = fd;
	switch (rw)
	{
	case 0: 
		ev.events = EPOLLIN; 
		break;
	case 1: 
		ev.events = EPOLLOUT; 
		break;
	default:
		ev.events = 0;
		break;
	}

//	LOCK_WRITER(fdp);

	if (epoll_ctl (fdp->ep_fd, EPOLL_CTL_ADD, fd, &ev) < 0) 
	{
		PRINT_ERROR ("Error: epoll_ctl (EPOLL_CTL_ADD)\n");
		exit (EXIT_EPOLL_CTL_ADD);
	}

//	UNLOCK(fdp);

	return ret_ok;
}

void inline
fdpoll_epoll_set_mode (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	struct epoll_event ev;

	ev.data.fd = fd;
	switch (rw)
	{
	case 0: 
		ev.events = EPOLLIN; 
		break;
	case 1: 
		ev.events = EPOLLOUT; 
		break;
	default:
		ev.events = 0;
		break;
	}

//	LOCK_WRITER(fdp);

	if (epoll_ctl(fdp->ep_fd, EPOLL_CTL_MOD, fd, &ev) < 0) 
	{
		PRINT_ERROR ("Error: epoll_ctl (EPOLL_CTL_MOD)\n");
		exit (EXIT_EPOLL_CTL_MOD);
	}

//	UNLOCK(fdp);
}

ret_t
fdpoll_epoll_del (cherokee_fdpoll_t *fdp, int fd)
{
	struct epoll_event ev;

	if (epoll_ctl(fdp->ep_fd, EPOLL_CTL_DEL, fd, &ev) < 0)
	{
		PRINT_ERROR ("Error: epoll_ctl (EXIT_EPOLL_CTL_DEL)\n");
		exit (EXIT_EPOLL_CTL_DEL);
	}

	return ret_ok;
}

inline int
fdpoll_epoll_watch (cherokee_fdpoll_t *fdp, int timeout_msecs)
{
	int i, ridx;

	fdp->ep_readyfds = epoll_wait (fdp->ep_fd, fdp->ep_events, fdp->nfiles, timeout_msecs);

	for (i = 0, ridx = 0; i < fdp->ep_readyfds; ++i) {
		if (fdp->ep_events[i].events & (EPOLLIN | EPOLLOUT | EPOLLERR | EPOLLHUP))
		{
			fdp->epoll_idx2rs[fdp->ep_events[i].data.fd] = i;
			fdp->epoll_rs2idx[ridx] = i;
			ridx++;
		}
	}

	return ridx;
}

int
fdpoll_epoll_check (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	int fdidx = fdp->epoll_idx2rs[fd];

	if (fdidx < -1 || fdidx >= fdp->nfiles)
	{
		PRINT_ERROR ("Error: bad fdidx (%d) in epoll_check_fd!\n", fdidx);
		return 0;
	}

	if ((fdidx == -1) ||
	    (fdp->ep_events[fdidx].data.fd != fd))
	{
		return 0;
	}

	if (fdp->ep_events[fdidx].events == (EPOLLHUP | EPOLLERR)) 
	{
		return -1;
	}

//	switch (fdp->fd_rw[fd])
	switch (rw)
	{
	case 0: 
		return fdp->ep_events[fdidx].events & (EPOLLIN | EPOLLERR | EPOLLHUP);
	case 1: 
		return fdp->ep_events[fdidx].events & (EPOLLOUT | EPOLLERR | EPOLLHUP);
	}

	return 0;
}

ret_t
fdpoll_epoll_reset (cherokee_fdpoll_t *fdp, int fd)
{
	fdp->epoll_idx2rs[fd] = -1;

	return ret_ok;
}

#endif /* USE_EPOLL */




/***********************************************************************/
/* poll()                                                              */
/*                                                                     */
/* #include <sys/poll.h>                                               */
/* int poll(struct pollfd *ufds, unsigned int nfds, int timeout);      */
/*                                                                     */
/***********************************************************************/
#ifdef USE_POLL

ret_t 
fdpoll_poll_get_method (cherokee_fdpoll_t  *fdp, char **name)
{
	static char *method = "poll";

	*name = method;
	return ret_ok;
}

ret_t
fdpoll_poll_new (cherokee_fdpoll_t **fdp)
{
	int i;

	/* Get memory
	 */
	CHEROKEE_NEW_STRUCT (n, fdpoll);
	
	/* Look for max fd limit
	 */
	n->nfiles = get_fd_limit();
	n->npollfds = 0;
	
	/* Get memory
	 */
	n->pollfds = (struct pollfd *) malloc (sizeof(struct pollfd) * n->nfiles);
	return_if_fail (n->pollfds, ret_nomem);
	
	n->fdidx = (int*) malloc (sizeof(int) * n->nfiles);
	return_if_fail (n->fdidx, ret_nomem);
	
	for (i=0; i < n->nfiles; i++) {
		n->pollfds[i].fd = -1;
		n->fdidx[i]      = -1;
	}

#ifdef HAVE_PTHREAD
	   pthread_rwlock_init(&n->rwlock, NULL); 
#endif

	   /* Return it
	    */
	   *fdp = n;

	   return ret_ok;
}

ret_t
fdpoll_poll_free (cherokee_fdpoll_t *fdp)
{
	free (fdp->pollfds);
	free (fdp->fdidx);
	
	free (fdp);
	
	return ret_ok;
}


ret_t
fdpoll_poll_add (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	if (fdp->npollfds >= fdp->nfiles) {
		return ret_error;
	}

	fdp->pollfds[fdp->npollfds].fd = fd;
	switch (rw) {
        case 0:  
		fdp->pollfds[fdp->npollfds].events = POLL_READ; 
		break;
        case 1: 
		fdp->pollfds[fdp->npollfds].events = POLL_WRITE;
		break;
        }
	fdp->fdidx[fd] = fdp->npollfds;
	fdp->npollfds++;

	return ret_ok;
}


void inline
fdpoll_poll_set_mode (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	fdp->pollfds[fdp->fdidx[fd]].events = rw ? POLL_WRITE : POLL_READ;
}


ret_t
fdpoll_poll_del (cherokee_fdpoll_t *fdp, int fd)
{
	int idx = fdp->fdidx[fd];

	if (idx < 0 || idx >= fdp->nfiles) {
		PRINT_ERROR ("Error droping socket '%d' from fdpoll\n", idx);
		return ret_error;
	}

	fdp->npollfds--;
	fdp->pollfds[idx] = fdp->pollfds[fdp->npollfds];
	fdp->fdidx[fdp->pollfds[idx].fd] = idx;

	fdp->pollfds[fdp->npollfds].fd = -1;
	fdp->fdidx[fd] = -1;

	return ret_ok;
}


inline int
fdpoll_poll_watch (cherokee_fdpoll_t *fdp, int timeout_msecs)
{
	return poll (fdp->pollfds, fdp->npollfds, timeout_msecs);
}


int
fdpoll_poll_check (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	switch (rw) {
        case 0: 
		return fdp->pollfds[fdp->fdidx[fd]].revents & (POLLIN | POLLHUP | POLLNVAL);
        case 1: 
		return fdp->pollfds[fdp->fdidx[fd]].revents & (POLLOUT | POLLHUP | POLLNVAL);
        default: 
		SHOULDNT_HAPPEN;
		return -1;
	}
}


ret_t
fdpoll_poll_reset (cherokee_fdpoll_t *fdp, int fd)
{
}

#endif /* USE_EPOLL */




/***********************************************************************/
/* select()                                                            */
/*                                                                     */
/* #include <sys/select.h>                                             */
/* int select(int n, fd_set *readfds, fd_set *writefds,                */
/*            fd_set *exceptfds, struct timeval *timeout);             */
/*                                                                     */
/***********************************************************************/
#ifdef USE_SELECT

ret_t 
fdpoll_select_new (cherokee_fdpoll_t **fdp)
{
	int i;
	CHEROKEE_NEW_STRUCT (n, fdpoll);
	
	FD_ZERO (&n->master_rfdset);
	FD_ZERO (&n->master_wfdset);

	n->nfiles        = get_fd_limit();
	n->select_fds    = (int*) malloc(sizeof(int) * n->nfiles);
	n->select_fdidx  = (int*) malloc(sizeof(int) * n->nfiles);
	n->select_rfdidx = (int*) malloc(sizeof(int) * n->nfiles);

	n->nselect_fds   = 0;
	n->maxfd         = -1;
	n->maxfd_changed = 0;

	for (i = 0; i < n->nfiles; ++i) {
		n->select_fds[i]   = -1;
		n->select_fdidx[i] = -1;
	}

	*fdp = n;
	return ret_ok;
}


ret_t 
fdpoll_select_free (cherokee_fdpoll_t *fdp)
{
	if (fdp->select_fds != NULL) {
		free (fdp->select_fds);
	}
	
	if (fdp->select_fdidx != NULL) {
		free (fdp->select_fdidx);
	}
	
	if (fdp->select_rfdidx != NULL) {
		free (fdp->select_rfdidx);
	}

	free (fdp);

	return ret_ok;
}


ret_t 
fdpoll_select_get_method (cherokee_fdpoll_t *fdp, char **name)
{
	static char *method = "select";

	*name = method;
	return ret_ok;
}


ret_t 
fdpoll_select_add (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	if (fdp->nselect_fds >= fdp->nfiles) {
		PRINT_ERROR ("Too many fds in select_add_fd!\n");
		return ret_error;
        }

	fdp->select_fds[fdp->nselect_fds] = fd;
	switch (rw) {
        case fdp_read: 
		FD_SET (fd, &fdp->master_rfdset); 
		break;
        case fdp_write: 
		FD_SET (fd, &fdp->master_wfdset);
		break;
        default: 
		break;
        }

	if (fd > fdp->maxfd) {
		fdp->maxfd = fd;
	}

	fdp->select_fdidx[fd] = fdp->nselect_fds;
	fdp->nselect_fds++;

	return ret_ok;
}


ret_t 
fdpoll_select_del (cherokee_fdpoll_t *fdp, int fd)
{
	int idx = fdp->select_fdidx[fd];

	if ((idx < 0) || (idx >= fdp->nfiles)) {
		PRINT_ERROR ("Bad idx (%d) in select_del_fd!\n", idx);
		return ret_error;
        }

	fdp->nselect_fds--;
	fdp->select_fds[idx]                  = fdp->select_fds[fdp->nselect_fds];
	fdp->select_fdidx[fdp->select_fds[idx]] = idx;
	fdp->select_fds[fdp->nselect_fds]       = -1;
	fdp->select_fdidx[fd]                 = -1;

	FD_CLR (fd, &fdp->master_rfdset);
	FD_CLR (fd, &fdp->master_wfdset);

	if (fd >= fdp->maxfd) {
		fdp->maxfd_changed = 1;
	}

	return ret_ok;
}


void  
fdpoll_select_set_mode (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	ret_t ret;

	ret = fdpoll_select_del (fdp, fd);
	if (ret < ret_ok) return;

	fdpoll_select_add (fdp, fd, rw);
}


int   
fdpoll_select_check (cherokee_fdpoll_t *fdp, int fd, int rw)
{
/*
	if ((ridx < 0) || (ridx >= nfiles)) {
		PRINT_ERROR ("Bad ridx (%d) in select_get_fd!\n", ridx);
		return ret_error;
        }
	
	return fdp->select_rfdidx[ridx];
*/

	switch (fdp->fd_rw[fd]) {
        case fdp_read: 
		return FD_ISSET (fd, &fdp->working_rfdset);
        case fdp_write: 
		return FD_ISSET (fd, &fdp->working_wfdset);
        }

	return 0;
}


static int
select_get_maxfd (cherokee_fdpoll_t *fdp) 
{
	if (fdp->maxfd_changed) {
		int i;

		fdp->maxfd = -1;
		for (i = 0; i < fdp->nselect_fds; ++i) {
			if (fdp->select_fds[i] > fdp->maxfd ) {
				fdp->maxfd = fdp->select_fds[i];
			}
		}

		fdp->maxfd_changed = 0;
        }

	return maxfd;
}


int   
fdpoll_select_watch (cherokee_fdpoll_t *fdp, int timeout_msecs)
{
	int mfd;
	int r, idx, ridx;

	fdp->working_rfdset = fdp->master_rfdset;
	fdp->working_wfdset = fdp->master_wfdset;
	mfd = select_get_maxfd(fdp);
	if (timeout_msecs == INFTIM) {
		r = select (mfd + 1, &fdp->working_rfdset, &fdp->working_wfdset, NULL, NULL);
	} else {
		struct timeval timeout;
		timeout.tv_sec = timeout_msecs / 1000L;
		timeout.tv_usec = ( timeout_msecs % 1000L ) * 1000L;
		r = select (mfd + 1, &fdp->working_rfdset, &fdp->working_wfdset, NULL, &timeout);
        }

	if (r <= 0) {
		return ret_error;
	}

	ridx = 0;
	for (idx = 0; idx < nselect_fds; ++idx) {
		if (select_check_fd( select_fds[idx])) {
			fdp->select_rfdidx[ridx++] = fdp->select_fds[idx];
			if (ridx == r) break;
		}
	}

	return ridx;        /* should be equal to r */
}

#endif /* USE_SELECT */



