/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#ifdef HAVE_SYS_UTSNAME_H
# include <sys/utsname.h>
#endif

#include "handler_server_info.h"
#include "server.h"


cherokee_module_info_t cherokee_server_info_info = {
	cherokee_handler,                /* type     */
	cherokee_handler_server_info_new /* new func */
};


#define PAGE_HEADER                                                                                     \
"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"DTD/xhtml1-transitional.dtd\">" CRLF\
"<html><header>"                                                                                    CRLF\
"<meta  HTTP-EQUIV=\"refresh\" CONTENT=\"30\"><meta http-equiv=Cache-Control content=no-cache>"     CRLF\
"<style type=\"text/css\"><!--"                                                                     CRLF\
"body {background-color: #ffffff; color: #000000;}"                                                 CRLF\
"body, td, th, h1, h2 {font-family: sans-serif;}"                                                   CRLF\
"pre {margin: 0px; font-family: monospace;}"                                                        CRLF\
"a:link {color: #000099; text-decoration: none; background-color: #ffffff;}"                        CRLF\
"a:hover {text-decoration: underline;}"                                                             CRLF\
"table {border-collapse: collapse;}"                                                                CRLF\
".center {text-align: center;}"                                                                     CRLF\
".center table { margin-left: auto; margin-right: auto; text-align: left;}"                         CRLF\
".center th { text-align: center !important; }"                                                     CRLF\
"td, th { border: 1px solid #000000; font-size: 75%; vertical-align: baseline;}"                    CRLF\
"h1 {font-size: 150%;}"                                                                             CRLF\
"h2 {font-size: 125%;}"                                                                             CRLF\
".p {text-align: left;}"                                                                            CRLF\
".e {background-color: #ccccff; font-weight: bold; color: #000000;}"                                CRLF\
".h {background-color: #9999cc; font-weight: bold; color: #000000;}"                                CRLF\
".v {background-color: #cccccc; color: #000000;}"                                                   CRLF\
"i {color: #666666; background-color: #cccccc;}"                                                    CRLF\
"img {float: right; border: 0px;}"                                                                  CRLF\
"hr {width: 600px; background-color: #cccccc; border: 0px; height: 1px; color: #000000;}"           CRLF\
"//--></style>"                                                                                     CRLF\
"<title>Cherokee Web Server Info</title></header>"                                                  CRLF\
"<body><div class=\"center\">"                                                                      CRLF\
"<table border=\"0\" cellpadding=\"3\" width=\"600\">"                                              CRLF\
"  <tr class=\"h\"><td>"                                                                            CRLF\
"    <a href=\"http://www.0x50.org/\"><img border=\"0\" src=\"?logo\" alt=\"Cherokee Logo\" /></a>" CRLF\
"    <h1 class=\"p\">Cherokee Version " VERSION "</h1>"                                             CRLF\
"  </td></tr>"                                                                                      CRLF\
"</table><br />"

#define LICENSE                                                                                         \
"<p>Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega &lt;alvaro@alobbs.com&gt;</p>"               CRLF\
"<p>This program is free software; you can redistribute it and/or"                                  CRLF\
"modify it under the terms of the GNU General Public License as"                                    CRLF\
"published by the Free Software Foundation; either version 2 of the"                                CRLF\
"License, or (at your option) any later version.</p>"                                               CRLF\
"<p>This program is distributed in the hope that it will be useful,"                                CRLF\
"but WITHOUT ANY WARRANTY; without even the implied warranty of"                                    CRLF\
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"                                     CRLF\
"GNU General Public License for more details.</p>"                                                  CRLF\
"<p>You should have received a copy of the GNU General Public License"                              CRLF\
"along with this program; if not, write to the Free Software"                                       CRLF\
"Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA</p>"

#define PAGE_FOOT                                                                                   CRLF\
"<h2>Cherokee License</h2>"                                                                         CRLF\
"<table border=\"0\" cellpadding=\"3\" width=\"600\">"                                              CRLF\
"<tr class=\"v\"><td>" LICENSE                                                                      CRLF\
"</td></tr>"                                                                                        CRLF\
"</table><br />"                                                                                    CRLF\
"</div></body></html>"                                                                             


static void
server_info_add_table (cherokee_buffer_t *buf, char *name, char *a_name, cherokee_buffer_t *content)
{
	   cherokee_buffer_add_va (buf, "<h2><a name=\"%s\">%s</a></h2>", a_name, name);
	   cherokee_buffer_add_va (buf, "<table border=\"0\" cellpadding=\"3\" width=\"600\">");
	   cherokee_buffer_add_buffer (buf, content);
	   cherokee_buffer_add_va (buf, "</table><br />");
}

static void
table_add_row_str (cherokee_buffer_t *buf, char *name, char *value)
{
	   cherokee_buffer_add_va (buf, "<tr><td class=\"e\">%s</td><td class=\"v\">%s</td></tr>"CRLF, name, value);
}

static void
table_add_row_int (cherokee_buffer_t *buf, char *name, int value)
{
	   cherokee_buffer_add_va (buf, "<tr><td class=\"e\">%s</td><td class=\"v\">%d</td></tr>"CRLF, name, value);
}

static void
table_add_row_long (cherokee_buffer_t *buf, char *name, unsigned long int value)
{
	   cherokee_buffer_add_va (buf, "<tr><td class=\"e\">%s</td><td class=\"v\">%u</td></tr>"CRLF, name, value);
}

static char *
add_uptime_row (cherokee_buffer_t *buf, cherokee_server_t *srv)
{
	unsigned int elapse = srv->bogo_now - srv->start_time;
	unsigned int days;
	unsigned int hours;
	unsigned int mins;

	CHEROKEE_NEW(tmp,buffer);

	days = elapse / (60*60*24);
	elapse %= (60*60*24);

	hours = elapse / (60*60);
	elapse %= (60*60);
	
	mins = elapse / 60;
	elapse %= 60;

	if (days > 0) {
		cherokee_buffer_add_va (tmp, "%d Day%s, %d Hour%s, %d Minute%s, %d Seconds", 
					days, days>1?"s":"", hours, hours>1?"s":"", mins, mins>1?"s":"", elapse);
	} else if (hours > 0) {
		cherokee_buffer_add_va (tmp, "%d Hour%s, %d Minute%s, %d Seconds", 
					hours, hours>1?"s":"", mins, mins>1?"s":"", elapse);
	} else if (mins > 0) {
		cherokee_buffer_add_va (tmp, "%d Minute%s, %d Seconds", 
					mins, mins>1?"s":"", elapse);
	} else {
		cherokee_buffer_add_va (tmp, "%d Seconds", elapse);
	}
	
	table_add_row_str (buf, "Uptime", tmp->buf);
	cherokee_buffer_free (tmp);
}

static void
add_data_sent_row (cherokee_buffer_t *buf, cherokee_server_t *srv)
{
	unsigned long int rx, tx;
	char tmp[5];

	cherokee_server_get_total_traffic (srv, &rx, &tx);
	
	apr_strfsize (tx, tmp);
	table_add_row_str (buf, "Data sent", tmp);

	apr_strfsize (rx, tmp);
	table_add_row_str (buf, "Data received", tmp);
}

static void
build_general_table_content (cherokee_buffer_t *buf, cherokee_server_t *srv)
{
	   add_uptime_row (buf, srv);
	   add_data_sent_row (buf, srv);
}

static void
build_server_table_content (cherokee_buffer_t *buf, cherokee_server_t *srv)
{
	   const char *on       = "On";
	   const char *off      = "Off";
	   const char *disabled = "Disabled";

	   table_add_row_int (buf, "Thread Number ", (srv->ncpus == -1) ? 1 : srv->ncpus);
	   table_add_row_str (buf, "IPv6 ", (srv->ipv6 == 1) ? on : off);
	   table_add_row_str (buf, "TLS support ", (srv->socket_tls != -1) ? on : off);
	   table_add_row_int (buf, "TLS port ", srv->port_tls);
	   table_add_row_str (buf, "Chroot ", (srv->chrooted) ? on : off);
	   table_add_row_str (buf, "Users directory ", (srv->userdir) ? srv->userdir : disabled);
	   table_add_row_int (buf, "User ID", getuid());
	   table_add_row_int (buf, "Group ID", getgid());

#ifdef HAVE_SYS_UTSNAME
	   {
		   int rc;
		   struct utsname buf;

		   rc = uname ((struct utsname *)&buf);
		   if (rc >= 0) {
			   table_add_row_str (buf, "Operating system", buf.sysname);
		   }
	   }
#endif

}

static void
build_connections_table_content (cherokee_buffer_t *buf, cherokee_server_t *srv)
{
	int active;
	int reusable;
	
	cherokee_server_get_active_conns (srv, &active);
	cherokee_server_get_reusable_conns (srv, &reusable);

	table_add_row_int (buf, "Active connections", active);
	table_add_row_int (buf, "Reusable connections", reusable);
}

static int
build_modules_table_content_while (void **params, const char *key, void *value)
{
	   cherokee_buffer_t *buf = params[0];
	   cherokee_server_t *srv = params[1];
	   int *loggers    = params[2];
	   int *handlers   = params[3];
	   int *encoders   = params[4];
	   int *validators = params[5];
	   cherokee_module_info_t *mod = value;
	   
	   if (mod->type & cherokee_logger) {
			 *loggers += 1;
	   } else if (mod->type & cherokee_handler) {
			 *handlers += 1;
	   } else if (mod->type & cherokee_encoder) {
			 *encoders += 1;
	   } else if (mod->type & cherokee_validator) {
			 *validators += 1;
	   } else {
			 SHOULDNT_HAPPEN;
	   }

	   return 1;
}

static void
build_modules_table_content (cherokee_buffer_t *buf, cherokee_server_t *srv)
{
	   char *key, *value;
	   int   loggers    = 0;
	   int   handlers   = 0;
	   int   encoders   = 0;
	   int   validators = 0;	   
	   void *params[]   = {buf, srv, &loggers, &handlers, &encoders, &validators};

	   cherokee_table_while (srv->loader->table, build_modules_table_content_while, params, &key, &value);

	   table_add_row_int (buf, "Loggers", loggers);
	   table_add_row_int (buf, "Handlers", handlers);
	   table_add_row_int (buf, "Encoders",  encoders);
	   table_add_row_int (buf, "Validators", validators);
}

static ret_t
server_info_build_logo (cherokee_handler_server_info_t *hdl)
{
	   ret_t ret;
	   cherokee_buffer_t *buffer;

	   buffer = hdl->buffer;

#include "logo.inc"

	   return ret_ok;
}

static void
build_icons_table_content (cherokee_buffer_t *buf, cherokee_server_t *srv)
{
	table_add_row_str (buf, "Default icon", (srv->icons->default_icon) ? srv->icons->default_icon : "");
	table_add_row_str (buf, "Directory icon", (srv->icons->directory_icon) ? srv->icons->directory_icon : "");
	table_add_row_str (buf, "Parent directory icon", (srv->icons->parentdir_icon) ? srv->icons->parentdir_icon : "");
}

static void
server_info_build_page (cherokee_handler_server_info_t *hdl)
{
	   cherokee_server_t *srv;
	   cherokee_buffer_t *buf;
	   cherokee_buffer_t *table;

	   /* Init
	    */
	   buf = hdl->buffer;
	   srv = HANDLER_SRV(hdl);
	   
	   /* Add the page begining
	    */
	   cherokee_buffer_add (buf, PAGE_HEADER, strlen(PAGE_HEADER));

	   /* General table
	    */
	   cherokee_buffer_new (&table);
	   build_general_table_content (table, srv);
	   server_info_add_table (buf, "General Information", "general", table);

	   /* Server table
	    */
	   cherokee_buffer_clean (table);
	   build_server_table_content (table, srv);
	   server_info_add_table (buf, "Server Core", "server_core", table);

	   /* Connections table
	    */
	   cherokee_buffer_clean (table);
	   build_connections_table_content (table, srv);
	   server_info_add_table (buf, "Current connections", "connections", table);

	   /* Modules table
	    */
	   cherokee_buffer_clean (table);
	   build_modules_table_content (table, srv);
	   server_info_add_table (buf, "Modules", "modules", table);

	   /* Icons
	    */
	   cherokee_buffer_clean (table);
	   build_icons_table_content (table, srv);
	   server_info_add_table (buf, "Icons", "icons", table);

	   /* Add the page ending
	    */
	   cherokee_buffer_add (buf, PAGE_FOOT, strlen(PAGE_FOOT));

	   cherokee_buffer_free (table);
}


ret_t
cherokee_handler_server_info_new  (cherokee_handler_t **hdl, cherokee_connection_t *cnt, cherokee_table_t *properties)
{
	   CHEROKEE_NEW_STRUCT (n, handler_server_info);
	
	   /* Init the base class object
	    */
	   cherokee_handler_init_base(HANDLER(n), cnt);

	   HANDLER(n)->support = hsupport_length | hsupport_range;
	   
	   HANDLER(n)->init        = (handler_func_init_t) cherokee_handler_server_info_init;
	   HANDLER(n)->free        = (handler_func_free_t) cherokee_handler_server_info_free;
	   HANDLER(n)->step        = (handler_func_step_t) cherokee_handler_server_info_step;
	   HANDLER(n)->add_headers = (handler_func_add_headers_t) cherokee_handler_server_info_add_headers;

	   /* Init
	    */
	   cherokee_buffer_new (&n->buffer);
	   cherokee_buffer_ensure_size (n->buffer, 4*1024);
	   
	   *hdl = HANDLER(n);
	   return ret_ok;
}


ret_t 
cherokee_handler_server_info_init (cherokee_handler_server_info_t *hdl)
{
	ret_t  ret;
	char  *param;
	int    web_interface = 1;

	cherokee_connection_parse_args (HANDLER_CONN(hdl));
		
	ret = cherokee_table_get (HANDLER_CONN(hdl)->arguments, "logo", &param);
	if (ret == ret_ok) {
		
		/* Build the logo
		 */
		server_info_build_logo (hdl);
		hdl->action = send_logo;
		
		return ret_ok;
	}
	
	/* Build the page
	 */
	if (web_interface) {
		server_info_build_page (hdl);
	}

	hdl->action = send_page;
	
	return ret_ok;
}


ret_t 
cherokee_handler_server_info_free (cherokee_handler_server_info_t *hdl)
{
	   cherokee_buffer_free (hdl->buffer);
	   free (hdl);

	   return ret_ok;
}


ret_t 
cherokee_handler_server_info_step (cherokee_handler_server_info_t *hdl, cherokee_buffer_t *buffer)
{
	   cherokee_buffer_add_buffer (buffer, hdl->buffer);
	   return ret_eof_have_data;
}


ret_t 
cherokee_handler_server_info_add_headers (cherokee_handler_server_info_t *hdl, cherokee_buffer_t *buffer)
{
	   cherokee_buffer_add_va (buffer, "Content-length: %d"CRLF, hdl->buffer->len);

	   switch (hdl->action) {
	   case send_logo:
			 cherokee_buffer_add (buffer, "Content-Type: image/gif"CRLF, 25);
			 break;
	   case send_page:
	   default:
			 cherokee_buffer_add (buffer, "Content-Type: text/html"CRLF, 25);
			 break;
	   }

	   return ret_ok;
}
