/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *      Christopher Pruden <pruden@dyndns.org>
 *
 * Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include "header.h"


static void
clean_known_headers (cherokee_header_t *hdr)
{
	int i;
	
	for (i=0; i<HEADER_LENGTH; i++)
	{
		hdr->header[i].info     = NULL;
		hdr->header[i].info_len = -1;			 
	}
}

static void
clean_unknown_headers (cherokee_header_t *hdr)
{
	if (hdr->unknowns != NULL) {
		free (hdr->unknowns);
		hdr->unknowns     = NULL;
		hdr->unknowns_len = 0;
	}
}

static void
clean_headers (cherokee_header_t *hdr)
{
	clean_known_headers (hdr);
	clean_unknown_headers (hdr);
}


ret_t 
cherokee_header_new (cherokee_header_t **hdr)
{
	CHEROKEE_NEW_STRUCT(n,header);

	/* Known headers
	 */
	clean_known_headers (n);

	/* Unknown headers
	 */
	n->unknowns     = NULL;
	n->unknowns_len = 0;

	/* Properties
	 */
	n->method  = http_unknown;
	n->version = http_version_unknown;

	/* Request
	 */
	n->request     = NULL;
	n->request_len = 0;

	/* Query string
	 */
	n->query_string     = NULL;
	n->query_string_len = 0;
	
	*hdr = n;
	return ret_ok;
}


ret_t 
cherokee_header_free (cherokee_header_t *hdr)
{
	clean_unknown_headers (hdr);
	free (hdr);

	return ret_ok;
}


ret_t 
cherokee_header_clean (cherokee_header_t *hdr)
{
	clean_headers (hdr);	

	hdr->method  = http_unknown;
	hdr->version = http_version_unknown;

	hdr->request     = NULL;
	hdr->request_len = 0;

	hdr->query_string     = NULL;
	hdr->query_string_len = 0;	
	
	return ret_ok;
}


static ret_t
parse_first_line (cherokee_header_t *hdr, cherokee_buffer_t *buf, char **next_pos)
{
	char *line  = buf->buf;
	char *begin = line;
	char *end;
	char *ptr;

	/* Example:
	 * GET / HTTP/1.0
	 */

	end = index (line, '\r');

	/* Some security checks
	 */
	if (buf->len < 14) {
		return ret_error;
	}

	if (end == NULL) {
		return ret_error;
	}

	/* Return the line endding
	 */
	*next_pos = end + 2;

	/* Get the method
	 */
	if (strncmp (line, "GET ", 4) == 0) {
		hdr->method = http_get;
		begin += 4;

	} else if (strncmp (line, "POST ", 5) == 0) {
		hdr->method = http_post;
		begin += 5;

	} else if (strncmp (line, "HEAD ", 5) == 0) {
		hdr->method = http_head;
		begin += 5;
		
	} else if (strncmp (line, "OPTIONS ", 8) == 0) {
		hdr->method = http_options;
		begin += 8;
		
	} else {
		return ret_error;
	}

	
	/* Get the protocol version
	 */	
	switch (end[-3]) {
	case '1':
		hdr->version = http_version_11; 
		break;
	case '0':
		hdr->version = http_version_10; 
		break;
	case '9':
		hdr->version = http_version_09; 
		break;
	default:
		return ret_error;
	}

	/* Skip the HTTP version string: "HTTP/x.y"
	 */
	end -= 9;

	/* Look for the QueryString
	 */
	ptr = index (begin, '?');

	if ((ptr) && (ptr < end)) {
		end = ptr;
		hdr->query_string = ++ptr;
		hdr->query_string_len = index(ptr, ' ') - ptr;
	} else {
		hdr->query_string_len = 0;
	}

	/* Get the request
	 */
	hdr->request     = begin;
	hdr->request_len = end - begin;

	return ret_ok;
}


static ret_t 
add_known_header (cherokee_header_t *hdr, cherokee_common_header_t header, char *info, int info_len)
{
	hdr->header[header].info     = info;
	hdr->header[header].info_len = info_len;

	return ret_ok;
}


static ret_t 
add_unknown_header (cherokee_header_t *hdr, char *header, char *info, int info_len)
{
	cherokee_header_unknown_entry_t *entry;

	hdr->unknowns_len++;

	hdr->unknowns = realloc (hdr->unknowns, 
				 sizeof(cherokee_header_unknown_entry_t) * hdr->unknowns_len);
				 
	if (hdr->unknowns == NULL) {
		return ret_nomem;
	}
			
	entry = &hdr->unknowns[hdr->unknowns_len-1];
	entry->header          = header;
	entry->header_info     = info;
	entry->header_info_len = info_len;
	
	return ret_ok;
}


static char *
get_new_line (char *string)
{
	char *end1;
	char *end2;

	end1 = index (string, '\r');
	end2 = index (string, '\n');

	return min_string_not_null (end1, end2);
}

ret_t 
cherokee_header_parse (cherokee_header_t *hdr, cherokee_buffer_t *buffer, int http_request)
{
	ret_t  ret;
	char  *begin = buffer->buf;
	char  *end   = NULL;
	char  *points;

	if (begin == NULL) {
		return ret_error;
	}
	
	/* Parse the firdt line. Something like this:
	 * GET /icons/compressed.png HTTP/1.1\r\n
	 */
	if (http_request) {
		ret = parse_first_line (hdr, buffer, &begin);
		if (ret < ret_ok) return ret;
	}

	/* Parse the rest of headers
	 */
	while (end = get_new_line (begin))
	{
		points = index (begin, ':');
		if (points == NULL) {
			goto next;
		}
		
		if (end < points +2) {
			goto next;
		}

		if ((hdr->header[header_accept].info == NULL) && 
		    (strncasecmp(begin, "Accept", 6) == 0)) 
		{
			ret = add_known_header (hdr, header_accept, points+2, end-points-2);
		} 
		else if ((hdr->header[header_host].info == NULL) && 
			 (strncasecmp(begin, "Host", 4) == 0))
		{
			ret = add_known_header (hdr, header_host, points+2, end-points-2);
		} 
		else if ((hdr->header[header_range].info == NULL) && 
			 (strncasecmp(begin, "Range", 5) == 0))
		{
			ret = add_known_header (hdr, header_range, points+2, end-points-2);
		} 
		else if ((hdr->header[header_accept_encoding].info == NULL) && 
			 (strncasecmp(begin, "Accept-Encoding", 15) == 0))
		{
			ret = add_known_header (hdr, header_accept_encoding, points+2, end-points-2);
		} 
		else if ((hdr->header[header_connection].info == NULL) && 
			 (strncasecmp(begin, "Connection", 10) == 0))
		{
			ret = add_known_header (hdr, header_connection, points+2, end-points-2);
		}
		else if ((hdr->header[header_user_agent].info == NULL) && 
			 (strncasecmp(begin, "User-Agent", 10) == 0))
		{
			ret = add_known_header (hdr, header_user_agent, points+2, end-points-2);
		}
		else if ((hdr->header[header_keep_alive].info == NULL) && 
			 (strncasecmp(begin, "Keep-Alive", 10) == 0))
		{
			ret = add_known_header (hdr, header_keep_alive, points+2, end-points-2);
		}
		else if ((hdr->header[header_referer].info == NULL) && 
			 (strncasecmp(begin, "Referer", 7) == 0))
		{
			ret = add_known_header (hdr, header_referer, points+2, end-points-2);
		}
		else if ((hdr->header[header_location].info == NULL) && 
			 (strncasecmp(begin, "Location", 8) == 0))
		{
			ret = add_known_header (hdr, header_location, points+2, end-points-2);
		}
		else if ((hdr->header[header_content_length].info == NULL) && 
			 (strncasecmp(begin, "Content-Length", 14) == 0))
		{
			ret = add_known_header (hdr, header_content_length, points+2, end-points-2);
		}
		else {
			/* Add a unknown header
			 */
			ret = add_unknown_header (hdr, begin, points+2, end-points-2);

		}

		if (ret < ret_ok) {
			return ret;
		}

	next:
		while ((*end == '\r') || (*end == '\n')) end++;
		begin = end;
	}
	
	return ret_ok;
}


ret_t 
cherokee_header_get_unknown (cherokee_header_t *hdr, char *name, int name_len, char **header, int *header_len)
{
	int i;

	for (i=0; i < hdr->unknowns_len; i++)
	{
		if (strncasecmp (hdr->unknowns[i].header, name, name_len) == 0) {
			*header     = hdr->unknowns[i].header_info;
			*header_len = hdr->unknowns[i].header_info_len;

			return ret_ok;
		}
	}

	return ret_not_found;
}


ret_t
cherokee_header_copy_unknown (cherokee_header_t *hdr, char *name, int name_len, cherokee_buffer_t *buf)
{
	ret_t  ret;
	char  *info;
	int    info_len;

	ret = cherokee_header_get_unknown (hdr, name, name_len, &info, &info_len);
	if (ret != ret_ok) {
		return ret;
	}

	cherokee_buffer_add (buf, info, info_len);

	return ret_ok;
}


ret_t 
cherokee_header_has_known (cherokee_header_t *hdr, cherokee_common_header_t header)
{
	if (hdr->header[header].info != NULL) {
		return ret_ok;
	}

	return ret_not_found;
}


ret_t 
cherokee_header_get_known (cherokee_header_t *hdr, cherokee_common_header_t header, char **info, int *info_len)
{
	if (hdr->header[header].info == NULL) {
		return ret_not_found;
	}

	*info     = hdr->header[header].info;
	*info_len = hdr->header[header].info_len;

	return ret_ok;
}


ret_t 
cherokee_header_copy_known (cherokee_header_t *hdr, cherokee_common_header_t header, cherokee_buffer_t *buf)
{
	ret_t ret;
	char *info;
	int   info_len;

	ret = cherokee_header_get_known (hdr, header, &info, &info_len);
	if (ret != ret_ok) return ret;

	return cherokee_buffer_add (buf, info, info_len); 
}


ret_t 
cherokee_header_copy_request (cherokee_header_t *hdr, cherokee_buffer_t *request)
{
	ret_t ret;

	if ((hdr->request == NULL) ||
	    (hdr->request_len <= 0)) {
		return ret_error;
	}

	ret = cherokee_buffer_add (request, hdr->request, hdr->request_len);
	if (ret < ret_ok) return ret;

	return cherokee_buffer_decode (request);
}
