/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include "header.h"


static void
clean_known_headers (cherokee_header_t *hdr)
{
	int i;
	
	for (i=0; i<HEADER_LENGTH; i++)
	{
		hdr->header[i].info     = NULL;
		hdr->header[i].info_len = -1;			 
	}
}

static void
clean_unknown_headers (cherokee_header_t *hdr)
{
	if (hdr->unknowns != NULL) {
		free (hdr->unknowns);
		hdr->unknowns     = NULL;
		hdr->unknowns_len = 0;
	}
}

static void
clean_headers (cherokee_header_t *hdr)
{
	clean_known_headers (hdr);
	clean_unknown_headers (hdr);
}


ret_t 
cherokee_header_new (cherokee_header_t **hdr)
{
	CHEROKEE_NEW_STRUCT(n,header);

	/* Known headers
	 */
	clean_known_headers (n);

	/* Unknown headers
	 */
	n->unknowns     = NULL;
	n->unknowns_len = 0;

	/* Properties
	 */
	n->method  = http_unknown;
	n->version = http_version_unknown;

	/* Query string
	 */
	n->query_string     = NULL;
	n->query_string_len = 0;
	
	*hdr = n;
	return ret_ok;
}


ret_t 
cherokee_header_free (cherokee_header_t *hdr)
{
	clean_unknown_headers (hdr);
	free (hdr);

	return ret_ok;
}


ret_t 
cherokee_header_clean (cherokee_header_t *hdr)
{
	clean_headers (hdr);	

	hdr->method  = http_unknown;
	hdr->version = http_version_unknown;

	hdr->query_string     = NULL;
	hdr->query_string_len = 0;	

	return ret_ok;
}


static ret_t
parse_first_line (cherokee_header_t *hdr, cherokee_buffer_t *buf, char **next_pos)
{
	char *line  = buf->buf;
	char *begin = line;
	char *end;
	char *ptr;

	end = index (line, '\r');

	/* Some security checks
	 */
	if (buf->len < 14) {
		return ret_error;
	}

	if (end == NULL) {
		return ret_error;
	}

	/* Return the line endding
	 */
	*next_pos = end + 2;

	/* Get the method
	 */
	if (strncmp (line, "GET ", 4) == 0) {
		hdr->method = http_get;
		begin += 4;

	} else if (strncmp (line, "POST ", 5) == 0) {
		hdr->method = http_post;
		begin += 5;

	} else if (strncmp (line, "HEAD ", 5) == 0) {
		hdr->method = http_head;
		begin += 5;
		
	} else {
		return ret_error;
	}
	
	/* Get the protocol version
	 */
	
	switch (end[-3]) {
	case '1':
		hdr->version = http_version_11; 
		break;
	case '0':
		hdr->version = http_version_10; 
		break;
	case '9':
		hdr->version = http_version_09; 
		break;
	default:
		return ret_error;
	}

	/* Look for the QueryString
	 */
	ptr = index (begin, '?');

	if ((ptr) && (ptr < end)) {
		hdr->query_string = ++ptr;
		end = hdr->query_string;
	}	

	return ret_ok;
}


ret_t 
cherokee_header_parse (cherokee_header_t *hdr, cherokee_buffer_t *buffer)
{
	ret_t  ret;
	char  *begin = buffer->buf;
	char  *end   = NULL;
	char  *points;
	cherokee_common_header_t header;
	int                      known_header;

	if (begin == NULL) {
		return ret_error;
	}
	
	/* Parse the firdt line. Something like this:
	 * GET /icons/compressed.png HTTP/1.1\r\n
	 */
	ret = parse_first_line (hdr, buffer, &begin);
	if (ret < ret_ok) return ret;


	/* Parse the rest of headers
	 */
	while (end = index (begin, '\r'))
	{
		points = index (begin, ':');
		if (points == NULL) {
			continue;
		}
		
		if (end < points +2) {
			continue;
		}

		known_header = 0;

		if ((hdr->header[header_accept].info == NULL) && 
		    (strncasecmp(begin, "Accept", 6) == 0)) 
		{
			header = header_accept;
			known_header = 1;
		} 
		else if ((hdr->header[header_host].info == NULL) && 
			 (strncasecmp(begin, "Host", 4) == 0))
		{
			header = header_host;
			known_header = 1;
		} 
		else if ((hdr->header[header_range].info == NULL) && 
			 (strncasecmp(begin, "Range", 5) == 0))
		{
			header = header_range;
			known_header = 1;			
		} 
		else if ((hdr->header[header_accept_encoding].info == NULL) && 
			 (strncasecmp(begin, "Accept-Encoding", 15) == 0))
		{
			header = header_accept_encoding;
			known_header = 1;			
		} 
		else if ((hdr->header[header_connection].info == NULL) && 
			 (strncasecmp(begin, "Connection", 10) == 0))
		{
			header = header_connection;
			known_header = 1;
		}
		else if ((hdr->header[header_user_agent].info == NULL) && 
			 (strncasecmp(begin, "User-Agent", 10) == 0))
		{
			header = header_user_agent;
			known_header = 1;
		}
		else if ((hdr->header[header_keep_alive].info == NULL) && 
			 (strncasecmp(begin, "Keep-Alive", 11) == 0))
		{
			header = header_keep_alive;
			known_header = 1;
		}
		else {
			// Do something
		}

		if (known_header) {
			hdr->header[header].info     = points +2;
			hdr->header[header].info_len = end - points;
		}

		begin = end + 2;
	}
	
	return ret_ok;
}


ret_t 
cherokee_header_get_header (cherokee_header_t  *hdr, char *name, char **header, int **header_len)
{
	   return ret_ok;
}
