/* Cherokee
 *
 * Authors:
 *      Ayose Cazorla Len <setepo@gulic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <stdarg.h>

#include "module.h"
#include "handler_cgi.h"
#include "connection.h"

/* hmm.. it will be wonderful if cherokee had a
 * cherokee_connection_get_mehod_name (or similar), and use it 
 * instead of this vector
 */
static char *method[]  = {"GET", "POST", "HEAD", NULL};


cherokee_module_info_t cherokee_cgi_info = {
	cherokee_handler,           /* type     */
	cherokee_handler_cgi_new    /* new func */
};



void _LOG(const char* msg, ...)
{
#ifndef SILENT
	va_list l; 

	va_start(l, msg);
	vfprintf(stderr, msg, l);
	va_end(l);
#endif
}


ret_t
cherokee_handler_cgi_new  (cherokee_handler_t **hdl, void *cnt, cherokee_table_t *props)
{
	CHEROKEE_NEW_STRUCT (n, handler_cgi);
	
	/* Init the base class object
	 */
	cherokee_handler_init_base(HANDLER(n), cnt);
	HANDLER(n)->connection  = cnt;

//	HANDLER(n)->support = hsupport_length | hsupport_range;
	HANDLER(n)->support = hsupport_nothing;

	HANDLER(n)->init        = (handler_func_init_t) cherokee_handler_cgi_init;
	HANDLER(n)->free        = (handler_func_free_t) cherokee_handler_cgi_free;
	HANDLER(n)->step        = (handler_func_step_t) cherokee_handler_cgi_step;
	HANDLER(n)->add_headers = (handler_func_add_headers_t) cherokee_handler_cgi_add_headers;

	/* Process the request_string, and build the arguments table..
	 * We'll need it later
	 */
	cherokee_connection_parse_args (cnt);
	
	/* Init
	 */
	n->pipeInput  = 0;
	n->pipeOutput = 0;
	n->pid        = 0;
	n->pathInfo   = NULL;
	n->fileName   = NULL;
	n->data       = NULL;

	*hdl = HANDLER(n);

	return ret_ok;
}


ret_t
_set_envs_cgi(cherokee_connection_t* conn, cherokee_handler_cgi_t* n)
{
	char *p;
	int   r;

 	r  = setenv("SERVER_SIGNATURE", "<address>Cherokee web server</address>", 1);
	r += setenv("DOCUMENT_ROOT", conn->local_directory->buf, 1);
	r += setenv("HTTP_HOST", conn->host->buf, 1);

	if (r != 0) {
		   return ret_error;
	}

	if (conn->method < 4) {
		   r = setenv("REQUEST_METHOD", method[conn->method], 1);
		   if (r != 0) return ret_error;
	}


	p = index (conn->host->buf, ':');
	if (p != NULL) *p = '\0';
	r = setenv("SERVER_NAME", conn->host->buf, 1);
	if (p != NULL) *p = ':';
	if (r != 0) return ret_error;
	

#ifndef DISABLE_QUERYSTRING

	/* Query string 
	 */
	if(conn->query_string)
		setenv("QUERY_STRING", conn->query_string->buf, 1);
	else
		setenv("QUERY_STRING", "", 1);

	/* Set the REQUEST_URI variable. I think that is not
	 * the best way to do it, but I can't find any alternative
	 * in the cherokee API
	 */

	if ((conn->query_string) &&
	    (conn->query_string->buf))
	{
		cherokee_buffer_t *uri;
		char *qs, *req;

		qs = conn->query_string->buf;
		req = conn->request->buf;

		cherokee_buffer_new(&uri);
		cherokee_buffer_ensure_size(uri, strlen(req) + 2 + strlen(qs));

		cherokee_buffer_add(uri, req, strlen(req));
		cherokee_buffer_add(uri, "?", 1);
		cherokee_buffer_add(uri, qs, strlen(qs));

		setenv("REQUEST_URI", uri->buf, 1);
		cherokee_buffer_free(uri);
	} else {
		
#endif /* ndef DISABLE_QUERYSTRING */

		setenv("REQUEST_URI", conn->request->buf, 1);

#ifndef DISABLE_QUERYSTRING
	}
#endif

	/* Fake path */
	setenv("PATH", "/bin:/usr/bin:/sbin:/usr/sbin", 1);

	if (n->pathInfo)
		setenv("PATH_INFO", n->pathInfo->buf, 1);

	if (n->fileName)
		setenv("SCRIPT_FILE_NAME", n->fileName->buf, 1);

	/*
	 * TODO Fill the others CGI environment variables
	 *
	 * http://hoohoo.ncsa.uiuc.edu/cgi/env.html
	 */

	return ret_ok;
}

ret_t
_extract_path(cherokee_handler_cgi_t* n)
{
	int dr_len;
	char *cur;
	cherokee_connection_t *conn;

	conn = CONN(HANDLER(n)->connection);

	/* Append URI to DocumentRoot */
	dr_len = conn->local_directory->len;
	cherokee_buffer_add (conn->local_directory, 
			conn->request->buf, 
			conn->request->len);

	/* Search the executable file */
	for(cur = conn->local_directory->buf + dr_len + 1; *cur; ++cur)
	{
		if(*cur == '/')
		{
			struct stat st;

			*cur = 0;
			if (stat(conn->local_directory->buf, &st) == -1)
			{
				*cur = '/';
				_LOG("CGI: Not found %s\n", conn->local_directory->buf);
				return ret_error;
			}

			if (S_ISDIR(st.st_mode))
				*cur = '/';
			else
			{
				/* This can be the CGI to run */
				cherokee_buffer_new(&n->pathInfo);
				cherokee_buffer_new(&n->fileName);

				cherokee_buffer_add(n->fileName, 
					conn->local_directory->buf,
					strlen(conn->local_directory->buf));

				/* path info */
				*cur = '/';
				cherokee_buffer_add(n->pathInfo, 
					cur, strlen(cur));

				break;
			}
		}
	}

	/* filename is set? */
	if (n->fileName == NULL)
	{
		cherokee_buffer_new(&n->fileName);

		cherokee_buffer_add(n->fileName,
			conn->local_directory->buf,
			strlen(conn->local_directory->buf));

	}

	/* restore local_directory 
	 */
	conn->local_directory->len = dr_len;

	return ret_ok;
}

ret_t 
cherokee_handler_cgi_init (cherokee_handler_cgi_t *n)
{
	int pid;
	int pipesServer[2], pipesCGI[2];

	cherokee_connection_t *conn;

	conn = CONN(HANDLER(n)->connection);

	/* Creates the pipes ...*/
	pipe(pipesCGI);
	pipe(pipesServer);

	/* Extracts PATH_INFO and filename from request uri */
	if (_extract_path(n) != ret_ok)
		return ret_error;

	/* ... and fork the process */
	pid = fork();
	if (pid == 0)
	{
		char *absolute_path, *file;

		/* Change stdin and out */
		close(pipesCGI[0]);
		close(pipesServer[1]);
		
		dup2(pipesServer[0], 0);  /* stdin */
		dup2(pipesCGI[1], 1);     /* stdout */

		/* Sets the new environ. */
		_set_envs_cgi(conn, n);

		_LOG("CGI: Executing %s\n", n->fileName->buf);

		/* execute the new CGI */
		absolute_path = n->fileName->buf;
		file = strrchr(absolute_path, '/') + 1;
		execl(absolute_path, file, NULL);

		/* OH MY GOD!!! an error is here */
		_LOG("CGI: Error: %s\n", strerror(errno));

		exit(1);
	}

	_LOG("CGI: pid %d\n", pid);

	close(pipesServer[0]);
	close(pipesCGI[1]);

	/*sleep(1);*/

	n->pipeInput = pipesCGI[0];
	n->pipeOutput = pipesServer[0];
	n->pid = pid;


	/* TODO Yeah, I know, this is not a perfect solution, but it is very
	 * dificult do this in cherokee
	 */
	cherokee_buffer_new(&n->data);
	cherokee_buffer_ensure_size(n->data, 2 * 1024); /* 2 kb for headers */

	return ret_ok;
}


ret_t
cherokee_handler_cgi_free (cherokee_handler_cgi_t *n)
{
	_LOG("CGI: closing\n\n");

	close(n->pipeInput);
	close(n->pipeOutput);

	if(n->data)
		cherokee_buffer_free(n->data);

	if(n->pathInfo)
		cherokee_buffer_free(n->pathInfo);

	if(n->fileName)
		cherokee_buffer_free(n->fileName);

#if 0
        /* perhaps this is unnecesary, I'm unsure */
	kill(n->pid, SIGTERM);
	waitpid (0, &status, WNOHANG);
#endif

	free (n);
	
	return ret_ok;
}

ret_t
_read_from_cgi(cherokee_handler_cgi_t *n, cherokee_buffer_t *buffer)
{
	int len, ret_poll;
	struct pollfd polling = { n->pipeInput, POLLIN|POLLPRI, 0 };

	_LOG("CGI: Read from CGI... ");

	ret_poll = poll(&polling, 1, 10);

	if (ret_poll == 0)
	{
		/* Timeout. There is no data to read 
		 */

		_LOG("nothing to read. Timeout\n");

		buffer->len = 0;
		return ret_eagain;
	}

	if ((polling.revents & (POLLIN|POLLPRI)) == 0)	{
		/* An error. Perhaps, the CGI close its pipe 
		 */

		_LOG("Error in poll. revents = %i\n", polling.revents);
		return ret_error;
	}

	if (ret_poll == -1) {
		/* oops
		 */

		_LOG("returned error from poll: %s\n", strerror(errno));
		return ret_error;
	}

	/* read the data.
	 * This will put a \0 at the end of the string. If the returned
	 * data is a string, this is important. If it is binary data, 
	 * this change isn't dangerous
	 */
	len = read(n->pipeInput, buffer->buf, buffer->size - 1);
	buffer->buf[len] = '\0';

	if (!len) {
		_LOG("no bytes read\n");

		buffer->len = 0;
		return ret_eof;
	}

	if (len == -1) {
		_LOG("returned error form read: %s\n", strerror(errno));
		return ret_error;
	}

	_LOG(" %d bytes read.\n", len);
	buffer->len = len;

	return ret_ok;
}

ret_t
cherokee_handler_cgi_step (cherokee_handler_cgi_t *n, 
			    cherokee_buffer_t     *buffer)
{
	if (n->data)
	{
		ret_t r;

		_LOG("CGI: sending stored data: %d bytes\n", n->data->len);

		/* flush this buffer */
		if (n->data->len > 0)
		{
			cherokee_buffer_ensure_size(buffer, n->data->len);
			memcpy(buffer->buf, n->data->buf, n->data->len);
			buffer->len = n->data->len;

			r = ret_ok;
		} else
			r = ret_eagain;

		cherokee_buffer_free(n->data);
		n->data = NULL;
		return r;
	}

	return _read_from_cgi(n, buffer);
}


ret_t
cherokee_handler_cgi_add_headers (cherokee_handler_cgi_t *n,
				   cherokee_buffer_t     *buffer)
{
	/*int length;*/
	ret_t retcgi;
	char *start_body;
	char *p;
	char *last_nl;

	/*HANDLER_CONN(n)->keep_alive = 0;*/

	_LOG("CGI: Adding headers\n");

	/* Search the headers sent by the CGI 
	 */

	start_body = NULL;
	while(1)
	{
		/* read some data from the CGI */
		retcgi = _read_from_cgi(n, n->data);

		if (retcgi == ret_eagain)
			continue;

		if (retcgi != ret_ok)
			return retcgi;

		_LOG("CGI:   Headers: %d bytes recived\n", n->data->len);

		/* find the blank line, is any */
		cherokee_buffer_add(buffer, n->data->buf, n->data->len);

		last_nl = NULL;
		for (p = buffer->buf; *p; ++p)
		{
			if ((last_nl != NULL) && (*p == '\n'))
			{
				/* This is an empty line, so the headers are sent 
				 */
			
				_LOG("CGI:   blank line found\n");

				/* save the other bytes */
				start_body = p + 1;
				cherokee_buffer_make_empty(n->data);
				cherokee_buffer_add(n->data, 
					start_body, strlen(start_body));

				_LOG("CGI:   %d bytes to send\n", n->data->len);

				/* and send the header */
				buffer->len = last_nl - buffer->buf + 1;
				buffer->buf[buffer->len] = '\0';

				return ret_ok;

			} else if (*p == '\n') {
				last_nl = p;
				
			} else if (!isspace(*p) && (last_nl != NULL)) {
				last_nl = NULL;

			}
		}
	}

	return ret_ok;
}



/*   Library init function
 */

static int _cgi_is_init = 0;

#include <signal.h>

static void 
child_finished(int sng)
{
	int status;
	waitpid (0, &status, WNOHANG);
}

void
cgi_init ()
{
	if (_cgi_is_init)
		return;

	_cgi_is_init = 1;
	signal(SIGCHLD, child_finished);
}
