 /* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "module.h"
#include "encoder_gzip.h"


#define gzip_header_len 10
static char gzip_header[gzip_header_len] = {0x1F, 0x8B, 
					    Z_DEFLATED, 
					    0,          /* flags  */
					    0, 0, 0, 0, /* time   */
					    0,          /* xflags */
					    3};

/* GZIP
 * ====
 * gzip_header (10 bytes) + [gzip_encoder_content] + crc32 (4 bytes) + length (4 bytes)
 *
 */

cherokee_module_info_t cherokee_gzip_info = {
	cherokee_encoder,            /* type     */
	cherokee_encoder_gzip_new    /* new func */
};

ret_t 
cherokee_encoder_gzip_new (cherokee_encoder_gzip_t **encoder)
{
	CHEROKEE_NEW_STRUCT (n, encoder_gzip);

	cherokee_encoder_init_base (ENCODER(n));

	ENCODER(n)->free        = (encoder_func_free_t) cherokee_encoder_gzip_free;
	ENCODER(n)->add_headers = (encoder_func_add_headers_t) cherokee_encoder_gzip_add_headers;
	ENCODER(n)->init        = (encoder_func_encode_t) cherokee_encoder_gzip_init;
	ENCODER(n)->encode      = (encoder_func_encode_t) cherokee_encoder_gzip_encode;


	/* Return the object
	 */
	*encoder = n;

	return ret_ok;
}


ret_t 
cherokee_encoder_gzip_free (cherokee_encoder_gzip_t *encoder)
{
	free (encoder);
	return ret_ok;
}


ret_t 
cherokee_encoder_gzip_add_headers (cherokee_encoder_gzip_t *encoder,
				   cherokee_buffer_t       *buf)
{
	cherokee_buffer_add (buf, "Content-Encoding: gzip\r\n", 24);
	return ret_ok;
}


ret_t 
cherokee_encoder_gzip_init (cherokee_encoder_gzip_t *encoder, 
			    cherokee_buffer_t       *in, 
			    cherokee_buffer_t       *out)
{
	
	return ret_ok;
}


ret_t 
cherokee_encoder_gzip_encode (cherokee_encoder_gzip_t *encoder, 
			      cherokee_buffer_t       *in, 
			      cherokee_buffer_t       *out)
{
	int result;
	uLong the_crc, size;

	
 	if (in->len <= 0) 
		return ret_ok;


	/* Ensure output buffer size
	 */
 	size = 4 + 4 + in->len + (in->len * .1) + 13; 
  	cherokee_buffer_ensure_size (out, size);
	
	/* Encode:
	 */
	out->len = out->size;

	result = compress2 (out->buf,              /* dest */
			    (uLongf *) &out->len,  /* destLen */
			    in->buf,               /* source */
			    in->len,               /* sourceLen */
			    6);	                   /* level */


	out->len -= 4;
	cherokee_buffer_move_to_begin(out, 2);

	if (result != Z_OK && result != Z_STREAM_END) {
		fprintf (stderr, "zlib error: %d\n", result);
		return ret_error;
	}


	/* CRC32 
	 */
	the_crc = cherokee_buffer_crc32 (in);

	/* Add the GZip header at the beggining
	 */
	cherokee_buffer_prepend (out, gzip_header, gzip_header_len);

	/* Endding
	 */
	cherokee_buffer_add (out, (char *)&the_crc, 4);

	size = in->len;
	cherokee_buffer_add (out, (char *)&size, 4);

#if 0
	printf ("encoder %p\n", encoder);
	printf ("\tresult  %d\n", result);
	printf ("\tin->len  %d, out->len %d\n", in->len, out->len);

	cherokee_buffer_print_debug (out, -1);
#endif

	
	return ret_ok;
}



/*   Library init function
 */

static int _gzip_is_init = 0;

void
gzip_init ()
{
	/* Init flag
	 */
	if (_gzip_is_init) {
		return;
	}
	_gzip_is_init = 1;
}
