/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <pwd.h>
#include <time.h>

/* inet_ntoa()
 */
#include <sys/types.h>
#include <sys/socket.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#include "log.h"
#include "handler.h"
#include "connection.h"
#include "buffer.h"
#include "handler_table_entry.h"
#include "encoder_table.h"
#include "server.h"


ret_t
cherokee_connection_new  (cherokee_connection_t **cnt)
{
	CHEROKEE_NEW_STRUCT(n, connection);

	cherokee_buffer_new (&n->buffer);
	   
	INIT_LIST_HEAD((list_t*)n);
	n->tcp_cork    = 0;
	n->error_code  = http_ok;
	n->phase       = phase_reading_header;
	n->method      = http_unknown;
	n->version     = http_version_unknown;
	n->handler     = NULL; 
	n->encoder     = NULL;
	n->arguments   = NULL;
	n->logger_ref  = NULL;
	n->keep_alive  = 0;
	n->timeout     = time(NULL) + 15;   // CHECK ME !!
	n->range_start = 0;
	n->range_end   = 0;
	n->vserver     = NULL;
	n->log_at_end  = 0;

	cherokee_buffer_new (&n->local_directory);
	cherokee_buffer_new (&n->web_directory);
	cherokee_buffer_new (&n->request);
	cherokee_buffer_new (&n->redirect);
	cherokee_buffer_new (&n->host);
	cherokee_buffer_new (&n->query_string);

	cherokee_socket_new (&n->socket);

	*cnt = n;
	return ret_ok;
}


ret_t
cherokee_connection_free (cherokee_connection_t  *cnt)
{
	   cherokee_socket_free (cnt->socket);
	   cherokee_buffer_free (cnt->buffer);

	   if (cnt->handler != NULL) {
		   cherokee_handler_free (cnt->handler);
		   cnt->handler = NULL;
	   }

	   if (cnt->encoder != NULL) {
		   cherokee_encoder_free (cnt->encoder);
		   cnt->encoder = NULL;
	   }

	   cherokee_buffer_free (cnt->local_directory);
	   cherokee_buffer_free (cnt->web_directory);
	   cherokee_buffer_free (cnt->request);
	   cherokee_buffer_free (cnt->redirect);
	   cherokee_buffer_free (cnt->host);

	   if (cnt->arguments) {
		   cherokee_table_free2 (cnt->arguments, free);
		   cnt->arguments = NULL;
	   }

	   free (cnt);
	   return ret_ok;
}


ret_t
cherokee_connection_clean (cherokee_connection_t *cnt)
{	   
	   cnt->phase        = phase_reading_header;
	   cnt->version      = http_version_unknown;
	   cnt->method       = http_unknown;
	   cnt->error_code   = http_ok;
	   cnt->timeout      = -1;
	   cnt->range_start  = 0;
	   cnt->range_end    = 0;
	   cnt->logger_ref   = NULL;
	   cnt->vserver      = NULL;
	   cnt->tcp_cork     = 0;
	   cnt->log_at_end   = 0;

	   if (cnt->handler != NULL) {
		   cherokee_handler_free (cnt->handler);
		   cnt->handler = NULL;
	   }

	   if (cnt->encoder != NULL) {
		   cherokee_encoder_free (cnt->encoder);
		   cnt->encoder = NULL;
	   }

	   cherokee_buffer_clean (cnt->buffer);
	   cherokee_buffer_clean (cnt->local_directory);
	   cherokee_buffer_clean (cnt->web_directory);
	   cherokee_buffer_clean (cnt->request);
	   cherokee_buffer_clean (cnt->redirect);
	   cherokee_buffer_clean (cnt->host);
	   cherokee_buffer_clean (cnt->query_string);

	   if (cnt->arguments) {
		   cherokee_table_free2 (cnt->arguments, free);
		   cnt->arguments = NULL;
	   }

	   return ret_ok;
}


ret_t 
cherokee_connection_mrproper (cherokee_connection_t *cnt)
{
	ret_t ret;

	cnt->keep_alive = 0;

	ret  = cherokee_connection_clean (cnt);
	ret |= cherokee_socket_close (cnt->socket);

	return ret;
}


static inline void
add_error_code_string_to_buffer (cherokee_connection_t *cnt)
{
	switch(cnt->error_code) {
	case http_ok:
		cherokee_buffer_add (cnt->buffer, http_ok_string, 6); break;
	case http_accepted:
		cherokee_buffer_add (cnt->buffer, http_accepted_string CRLF, 12); break;
	case http_partial_content:
		cherokee_buffer_add (cnt->buffer, http_partial_content_string CRLF, 19); break;		
	case http_bad_request:
		cherokee_buffer_add (cnt->buffer, http_bad_request_string CRLF, 15); break;
	case http_access_denied:
		cherokee_buffer_add (cnt->buffer, http_access_denied_string CRLF, 13); break;
	case http_not_found:
		cherokee_buffer_add (cnt->buffer, http_not_found_string CRLF, 13); break;
	case http_internal_error:
		cherokee_buffer_add (cnt->buffer, http_internal_error_string CRLF, 25); break;
	case http_moved_permanently: 
		cherokee_buffer_add (cnt->buffer, http_moved_permanently_string CRLF, 21); break;
	default:
		SHOULDNT_HAPPEN;
	}
}


static ret_t
send_buffer_unsafe (cherokee_connection_t *cnt)
{
	int ret;

	ret = write (SOCKET_FD(cnt->socket), cnt->buffer->buf, cnt->buffer->len);
	cherokee_buffer_make_empty (cnt->buffer);

	if (ret <= 0) {
		cnt->keep_alive = 0;
		return ret_error;
	}

	return ret_ok;
}


ret_t
cherokee_connection_send_header (cherokee_connection_t *cnt)
{
	ret_t ret;
	
	cherokee_buffer_make_empty (cnt->buffer);
	
	/* Add protocol string + error_code
	 */
	switch (cnt->version) {
	case http_version_09:
		cherokee_buffer_add (cnt->buffer, "HTTP/0.9 ", 9); 
		break;
	case http_version_11:
		cherokee_buffer_add (cnt->buffer, "HTTP/1.1 ", 9); 
		break;
	case http_version_10:
	default:
		cherokee_buffer_add (cnt->buffer, "HTTP/1.0 ", 9); 
		break;
	}
	
	add_error_code_string_to_buffer (cnt);
	cherokee_buffer_add (cnt->buffer, CRLF, 2); 

	/* Add server name
	 */
	if (CONN_SRV(cnt)->hideservername == 0) {
		if (CONN_SRV(cnt)->hideversion) {
			cherokee_buffer_add (cnt->buffer, "Server: Cherokee" CRLF, 16+2);
		} else {
			cherokee_buffer_add (cnt->buffer, "Server: Cherokee/" VERSION CRLF, 17+5+2);
		}
	}

	/* Redirected connections
	 */
	if (cnt->redirect->len >= 1) {
		cherokee_buffer_add (cnt->buffer, "Location: ", 10);
		cherokee_buffer_add (cnt->buffer, cnt->redirect->buf, cnt->redirect->len);
		cherokee_buffer_add (cnt->buffer, CRLF, 2);
	}
	
	/* Encoder headers
	 */
	if (cnt->encoder) {
		cherokee_encoder_add_headers (cnt->encoder, cnt->buffer);
		
		/* Keep-alive is not possible w/o a file cache
		 */
		cnt->keep_alive = 0;

		if (cnt->handler->support & hsupport_length) {
			cnt->handler->support ^= hsupport_length;
		}
	}
	
	/* Maybe the handler add some headers
	 */
	if (cnt->handler) {
		cherokee_handler_add_headers (cnt->handler, cnt->buffer);
	}
	
	/* Add the "Connection:" header
	 */
	if (cnt->handler && cnt->keep_alive) {
		cherokee_buffer_add (cnt->buffer, "Connection: Keep-Alive"CRLF
				                  "Keep-Alive: timeout=15"CRLF, 48);
	} else {
		cherokee_buffer_add (cnt->buffer, "Connection: close"CRLF, 19);
	}
	
	/* If it's an error page we've to add the content-type header
	 */
	if (! http_type_200(cnt->error_code)) {
		cherokee_buffer_add (cnt->buffer, "Content-Type: text/html"CRLF, 25);		
	}

	/* Add the response header ends
	 */
	cherokee_buffer_add (cnt->buffer, CRLF, 2);
	
	/* Send it
	 */
	ret = send_buffer_unsafe (cnt);

	cherokee_buffer_make_empty  (cnt->buffer);

	return ret;
}


ret_t
cherokee_connection_recv (cherokee_connection_t  *cnt)
{
	int  readed;
	char tmp[DEFAULT_RECV_SIZE];
	
	readed = read (SOCKET_FD(cnt->socket), tmp, DEFAULT_RECV_SIZE);

	if (readed > 0) {
		cherokee_buffer_add (cnt->buffer, tmp, readed);
		return ret_ok;
	}

	/* Remote machine cancel the download
	 */
	return ret_eof;
}


int
cherokee_connection_eoh (cherokee_connection_t *cnt)
{
	if (cherokee_buffer_is_empty(cnt->buffer))
		return 0;
	
	if (cnt->buffer->len < 4)
		return 0;
	
	return ((cnt->buffer->buf[cnt->buffer->len-1] == '\n') &&
		(cnt->buffer->buf[cnt->buffer->len-2] == '\r') &&
		(cnt->buffer->buf[cnt->buffer->len-3] == '\n') &&
		(cnt->buffer->buf[cnt->buffer->len-4] == '\r'));
}


ret_t 
cherokee_connection_reading_check (cherokee_connection_t *cnt)
{
	/* Check for too long headers
	 */
	if (cnt->buffer->len > MAX_HEADER_LEN) {
		cnt->error_code = http_bad_request;
		return ret_error;
	}

	return ret_ok;
}


ret_t 
cherokee_connection_set_cork (cherokee_connection_t *cnt, int enable)
{
	int on = 0;
	int fd;

#ifdef HAVE_TCP_CORK
	fd = SOCKET_FD(cnt->socket);
	if (enable) {
		setsockopt(fd, IPPROTO_TCP, TCP_NODELAY,  &on, sizeof on);

		on = 1;
		setsockopt(fd, IPPROTO_TCP, TCP_CORK,  &on, sizeof on);
	} else {
		setsockopt(fd, IPPROTO_TCP, TCP_CORK,  &on, sizeof on);

		on = 1;
		setsockopt(fd, IPPROTO_TCP, TCP_NODELAY,  &on, sizeof on);
	}

	cnt->tcp_cork = enable;
#endif

	return ret_ok;
}


ret_t
cherokee_connection_send (cherokee_connection_t *cnt)
{
	int sent;

	sent = write (SOCKET_FD(cnt->socket), cnt->buffer->buf, cnt->buffer->len);

	if (sent > 0) {
		if (sent == cnt->buffer->len) {
			cherokee_buffer_make_empty (cnt->buffer);
		} else {
			cherokee_buffer_move_to_begin (cnt->buffer, sent);
		}

		/* If this connection has a handler without content-length support
		 * it has to count the bytes sent
		 */
		if (HANDLER_SUPPORT_LENGTH(cnt->handler)) {
			cnt->range_end += sent;
		}

		return ret_ok;		
	}
	
	if (sent == 0) {
		return ret_eof;
	}

	if (sent < 0) {
		switch (errno) {
		case EAGAIN:
			return ret_eagain;
		default:
			/* Remote machine cancel the download
			 */
			return ret_error;
		}
	}
}


ret_t
cherokee_connection_close (cherokee_connection_t *cnt)
{
	return cherokee_socket_close (cnt->socket);
}


ret_t
cherokee_connection_step (cherokee_connection_t *cnt)
{
	ret_t ret_step = ret_ok;

	return_if_fail (cnt->handler != NULL, ret_error);


	/* Need to 'read' from handler ?
	 */
	if (cherokee_buffer_is_empty (cnt->buffer)) 
	{
		ret_step = cherokee_handler_step (cnt->handler, cnt->buffer);
		if ((ret_step != ret_ok) && (ret_step != ret_eof_have_data)) return ret_step;

		/* May be encode..
		 */
		if (cnt->encoder != NULL) 
		{
			ret_t ret;
			cherokee_buffer_t *encoded;
		
			/* New buffer	
			 */
			ret = cherokee_buffer_new (&encoded);
			if (ret != ret_ok) return ret;
			
			/* Encode
			 */
			cherokee_encoder_encode (cnt->encoder, cnt->buffer, encoded);
			
			/* Swap buffers	
			 */
			cherokee_buffer_free (cnt->buffer);
			cnt->buffer = encoded;
		}
	}
	
	return ret_step;
}


static inline ret_t
get_host (cherokee_connection_t *cnt, 
	  char                  *ptr) 
{
	/* ptr - Header at the "Host:" position 
	 */
	char *end, *end2;
	int   size;

	end = strchr (ptr, '\r');
	if (end == NULL) {
		return ret_error;
	}

	/* Drop the port if present
	 * Eg: www.alobbs.com:8080 -> www.alobbs.com
	 */ 
	end2 = strchr (ptr, ':');
	if (end2 && (end2 < end))  {
		end = end2;
	}

	size = (end - ptr);
	return cherokee_buffer_add (cnt->host, ptr, size);
}

static inline ret_t
get_encoding (cherokee_connection_t    *cnt,
	      char                     *ptr,
	      cherokee_encoder_table_t *encoders) 
{
	char tmp;
	char *i1, *i2;
	char *end;
	char *ext;

	/* ptr = Header at the "Accept-Encoding position 
	 */
	end = strchr (ptr, '\r');
	if (end == NULL) {
		return ret_error;
	}

	/* Look for the request extension
	 */
	ext = rindex (cnt->request->buf, '.');
	if (ext == NULL) {
		return ret_ok;
	}

	*end = '\0'; /* (1) */
	
	i1 = ptr;
	
	do {
		i2 = index (i1, ',');
		if (!i2) i2 = index (i1, ';');
		if (!i2) i2 = end;

		tmp = *i2;    /* (2) */
		*i2 = '\0';

		cherokee_encoder_table_new_encoder (encoders, i1, ext+1, &cnt->encoder);
		if (cnt->encoder) break;

		*i2 = tmp;    /* (2') */

		if (i2 < end) {
			i1 = i2+1;
		}

	} while (i2 < end);

	*end = '\r'; /* (1') */

	return ret_ok;
}


static inline ret_t
get_uri (cherokee_connection_t *cnt)
{
	int begin, end;
	char *ptr;
	
	/* Malformed header
	 */
	if (cnt->buffer->len < 14)
		return ret_error;

	
	/* HTTP method
	 */
	if (strncmp (cnt->buffer->buf, "GET ", 4) == 0) {
		cnt->method = http_get;
		begin = 4;

	} else if (strncmp (cnt->buffer->buf, "POST ", 5) == 0) {
		cnt->method = http_post;
		begin = 5;

	} else if (strncmp (cnt->buffer->buf, "HEAD ", 5) == 0) {
		cnt->method = http_head;
		begin = 5;
		
	} else {
		return ret_error;
	}
	
	
	/* End the request of line
	 */	
	ptr = strstr (cnt->buffer->buf, CRLF);
	if (ptr == NULL) {
		return ret_error;
	}
	end = ptr - cnt->buffer->buf;
	
	
	/* HTTP version: HTTP/x.y
	 */
	end -= 9;
	
	switch (cnt->buffer->buf[end+8]) {
	case '1':
		cnt->version = http_version_11; break;
	case '0':
		cnt->version = http_version_10; break;
	case '9':
		cnt->version = http_version_09; break;
	default:
		return ret_error;
	}
	
	/* Look for the QueryString
	 */
	cnt->qs_begin_ref = index (cnt->buffer->buf + begin, '?');
	if (cnt->qs_begin_ref != NULL) {
		end = cnt->qs_begin_ref - cnt->buffer->buf;
		cnt->qs_begin_ref++;
	}

	/* Dup the request
	 */
	cherokee_buffer_add (cnt->request, cnt->buffer->buf + begin, end-begin);

	/* Decode %HEX URL encoding
	 */
	cherokee_buffer_decode (cnt->request);

	/* Look for "Host:"
	 */
	if ((ptr = strstr (cnt->buffer->buf, CRLF "Host: "))) {
		ptr += 8;
		get_host (cnt, ptr);
	} else {
		/* HTTP/1.1 connections *must* have an `Host' header
		 */
		if (cnt->version == http_version_11) {
			cnt->error_code = http_bad_request;
			return ret_error;
		}
	}
	
	return ret_ok;
}


int inline
cherokee_connection_is_userdir (cherokee_connection_t *cnt)
{
	return ((cnt->request->len > 4) && (cnt->request->buf[1] == '~'));
}


ret_t
cherokee_connection_build_local_directory (cherokee_connection_t *cnt, cherokee_virtual_server_t *vsrv, cherokee_handler_table_entry_t *entry)
{
	ret_t ret;

	if (entry->document_root->len >= 1) {
		/* Have a special DocumentRoot
		 */
		ret = cherokee_buffer_add (cnt->local_directory, entry->document_root->buf, entry->document_root->len);
		
		/* It has to drop the webdir from the request:
		 *
		 * Directory /thing {
		 *    DocumentRoot /usr/share/this/rocks
		 * }
		 *
		 * on petition: http://server/thing/cherokee
		 * should read: /usr/share/this/rocks/cherokee
		 */
		cherokee_buffer_move_to_begin (cnt->request, cnt->web_directory->len - 1);
		
	} else {

		/* Normal request
		 */
		ret = cherokee_buffer_add (cnt->local_directory, vsrv->root, vsrv->root_len);
	}

	return ret;
}


ret_t
cherokee_connection_build_local_directory_userdir (cherokee_connection_t *cnt, char *userdir)
{
	struct passwd *pwd;	
	char          *begin, *end;

	begin = &cnt->request->buf[2];
	end   = &cnt->request->buf[cnt->request->len-1];

	/* It's an special case. The request is something like:
	 * 	http://www.alobbs.com/~alo/dir
	 * and Cherokee should redirect it to:
	 * 	http://www.alobbs.com/~alo/dir/
	 */
	if (*end != '/') {
		cherokee_buffer_ensure_size (cnt->redirect, cnt->request->len+2);
		cherokee_buffer_add (cnt->redirect, cnt->request->buf, cnt->request->len);
		cherokee_buffer_add (cnt->redirect, "/", 1);

		cnt->error_code = http_moved_permanently;
		return ret_error;
	}

	/* Find user name endding
	 */
	end = index (begin, '/');
	if (end == NULL)
		return ret_error;
		
	/* Copy the username
	 */
	memcpy (gbl_buffer, begin, end-begin);
	gbl_buffer[end-begin] = '\0';
	
	/* Get the user home directory
	 */
	pwd = (struct passwd *) getpwnam (gbl_buffer);
	if ((pwd == NULL) || (pwd->pw_dir == NULL)) {
		cnt->error_code = http_not_found;
		return ret_error;
	}
		
	/* Build the local_directory
	 */
	cherokee_buffer_add (cnt->local_directory, pwd->pw_dir, strlen(pwd->pw_dir));
	cherokee_buffer_add (cnt->local_directory, "/", 1);
	cherokee_buffer_add (cnt->local_directory, userdir, strlen(userdir));

	/* Drop username from the request
	 */
	cherokee_buffer_move_to_begin (cnt->request, (end - cnt->request->buf));

	return ret_ok;
}


static inline ret_t
get_range (cherokee_connection_t *cnt, 
	   char                  *ptr) /* Header at the "Range: bytes=" position */
{
	int num_len = 0;

	/* Read the start position
	 */
	while ((ptr[num_len] != '-') && (ptr[num_len] != '\0') && (num_len < gbl_buffer_size-1)) {
		gbl_buffer[num_len] = ptr[num_len];
		num_len++;
	}
	gbl_buffer[num_len] = '\0';
	cnt->range_start = atoi (gbl_buffer);
	
	/* Advance the pointer
	 */
	ptr += num_len;
	if (*ptr == '-') {
		ptr++;
	} else {
		return ret_error;
	}

	/* Maybe there're an ending position
	 */
	if ((*ptr != '\0') && (*ptr != '\r') && (*ptr != '\n')) {
		num_len = 0;
		
		/* Read the end
		 */
		while ((ptr[num_len] != '-') && (ptr[num_len] != '\0') && (num_len < gbl_buffer_size-1)) {
			gbl_buffer[num_len] = ptr[num_len];
			num_len++;
		}
		gbl_buffer[num_len] = '\0';
		cnt->range_end = atoi (gbl_buffer);
	}

	return ret_ok;
}


static void
delete_double_slash (cherokee_buffer_t *buffer)
{
	char *a      = buffer->buf;
	int   offset = 0;

	if (buffer->len < 2) return;

	do {
		if ((a[0] == '/') && (a[offset+1] == '/')) {
			offset++;
			continue;
		}
		
		a[0] = a[offset];
		a++;

	} while ((a && *a != '\0') && (a < buffer->buf + buffer->len));

	buffer->len -= offset;
}


ret_t 
cherokee_connection_get_request (cherokee_connection_t *cnt)
{
	ret_t  ret;

        /* Get the request     
	 */
	ret = get_uri (cnt);
	if (cherokee_buffer_is_empty(cnt->request) || (ret != ret_ok)) {
		cnt->error_code = http_bad_request;
		return ret_error;
	}

	/* Look for ".."
	 */
	if (strstr (cnt->request->buf, "..")) {
		cnt->error_code = http_bad_request;
		return ret_error;
	}

	/* Look for starting '/' in the request
	 */
	if (cnt->request->buf[0] != '/') {
		cnt->error_code = http_bad_request;
		return ret_error;		
	}

	/* Look for "//" 
	 */
	delete_double_slash (cnt->request);

	cnt->error_code = http_ok;
	return ret_ok;	
}


ret_t 
cherokee_connection_get_plugin_entry (cherokee_connection_t *cnt, cherokee_handler_table_t *plugins, cherokee_handler_table_entry_t **plugin_entry)
{
	ret_t ret;

	/* Look for the handler "*_new" function
	 */
	ret = cherokee_handler_table_get (plugins, cnt->request->buf, plugin_entry, cnt->web_directory);
	if (ret != ret_ok) {
		cnt->error_code = http_internal_error;
		return ret_error;
	}	

	return ret_ok;
}


ret_t 
cherokee_connection_create_handler (cherokee_connection_t *cnt, cherokee_handler_table_entry_t *plugin_entry)
{
	ret_t ret;

	return_if_fail (plugin_entry->handler_new_func != NULL, ret_error);

	/* Create and assign a handler object
	 */
	ret = (plugin_entry->handler_new_func) ((void **)&cnt->handler, cnt, plugin_entry->properties);
	if ((ret != ret_ok) || (cnt->handler == NULL)) {
		cnt->error_code = http_internal_error;
		return ret_error;
	}

	/* Check the handler state after execute the constructor
	 */
	if (ret == ret_ok) {
		cherokee_buffer_ensure_size (cnt->buffer, DEFAULT_READ_SIZE);
	}

	return ret_ok;
}


ret_t
cherokee_connection_parse_header (cherokee_connection_t *cnt, cherokee_encoder_table_t *encoders)
{	   
	char *ptr;

	/* Look for "Connection: Keep-Alive / Close"
	 */
	if ((ptr = strstr (cnt->buffer->buf, "Connection: "))) {
		char *end;

		ptr += 12;

		end = strchr(ptr, '\r');
		if (end == NULL) return ret_error;
		
		*end = '\0';
		if (strncasecmp (ptr, "Keep-Alive", 10) == 0) {
			if (cnt->keep_alive == 0) {
				cnt->keep_alive = CONN_SRV(cnt)->keepalive_max;
			} else {
				cnt->keep_alive--;
			}
		} else if (strncasecmp (ptr, "close", 5) == 0) {
			cnt->keep_alive = 0;
		}
		*end = '\r';
	}
	
	/* Look for "Range:" 
	 */
	if (cnt->handler->support & hsupport_range) {
		if ((ptr = strstr (cnt->buffer->buf, "Range: bytes="))) {
			ptr += 13;
			get_range (cnt, ptr);
		}
	}

	/* Look for "Accept-Encoding:"
	 */
	if ((ptr = strstr (cnt->buffer->buf, "Accept-Encoding: "))) {
		ptr += 17;
		get_encoding (cnt, ptr, encoders);
	}

	return ret_ok;
}



/* cherokee_connection_parse_args:
 *  Process the incoming parameters.
 *
 *  1.- Build cnt->arguments
 *  2.- Build cnt->query_string
 */
ret_t 
cherokee_connection_parse_args (cherokee_connection_t *cnt)
{
	int  len;
	char tmp_end;
	char *string, *token;

	if (cnt->qs_begin_ref == NULL) {
		return ret_ok;
	}

	if (cnt->arguments != NULL) {
		return ret_error;
	}

	/* Look for the end
	 */
	len = strcspn (cnt->qs_begin_ref, "\n\r ");
	cherokee_buffer_add (cnt->query_string, cnt->qs_begin_ref, len);

	/* Parse the parameters
	 */
	string = cnt->query_string->buf;
	while ((token = strtok(string, "&")) != NULL) 
	{
		char *equ, *key, *val;
		string = NULL;

		if (equ = index (token, '='))
		{
			*equ = '\0';

			key = token;
			val = equ+1;

			// printf ("key='%s' val='%s'\n", key, val);
			cherokee_table_add (cnt->arguments, key, val);

			*equ = '=';			
		} else {
			// printf ("key='%s'\n", token);
			cherokee_table_add (cnt->arguments, token, NULL);
		}
	}

	return ret_ok;
}


ret_t
cherokee_connection_open_request (cherokee_connection_t *cnt)
{	
#if 0
	printf ("cherokee_connection_open_request:\n\tweb_dir '%s'\n\trequest '%s'\n\tlocal   '%s'\n", 
		cnt->web_directory->buf,
		cnt->request->buf,
		cnt->local_directory->buf);
#endif

	return cherokee_handler_init (cnt->handler);
}


ret_t
cherokee_connection_send_response_page_hardcoded (cherokee_connection_t *cnt)
{
	   ret_t  ret;
	   
	   cherokee_buffer_make_empty  (cnt->buffer);
	   
	   cherokee_buffer_add (cnt->buffer, "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">" CRLF, 50);
	   
	   /* Add page title
	    */
	   cherokee_buffer_add (cnt->buffer, "<html><head><title>", 19);
	   add_error_code_string_to_buffer (cnt);

	   /* Add big banner
	    */
	   cherokee_buffer_add (cnt->buffer, "</title></head><body><h1>", 25);
	   add_error_code_string_to_buffer (cnt);
	   cherokee_buffer_add (cnt->buffer, "</h1>", 5);

	   /* Maybe add some info
	    */
	   switch (cnt->error_code) {
	   case http_not_found:
		   if (cnt->request) {
			   cherokee_buffer_add (cnt->buffer, "The requested URL ", 18);
			   cherokee_buffer_add (cnt->buffer, cnt->request->buf, cnt->request->len);
			   cherokee_buffer_add (cnt->buffer, " was not found on this server.", 30);
		   }
		   break;
	   case http_bad_request:
		   cherokee_buffer_add (cnt->buffer, "Your browser sent a request that this server could not understand.", 66);
		   break;
	   case http_moved_permanently:
		   cherokee_buffer_add (cnt->buffer, "The document has moved", 22);
		   break;
	   default:
		   break;
	   }
	   
	   /* Add page foot
	    */
	   if (CONN_SRV(cnt)->hideservername == 0) {
		   cherokee_buffer_add (cnt->buffer, "<p><hr>", 7); 

		   if (CONN_SRV(cnt)->hideversion) {
			   cherokee_buffer_add_version (cnt->buffer, CONN_SRV(cnt)->port, ver_port_html);
		   } else {
			   cherokee_buffer_add_version (cnt->buffer, CONN_SRV(cnt)->port, ver_full_html);
		   }
	   }
  
	   cherokee_buffer_add (cnt->buffer, "</body></html>", 14); 
 
	   /* Send it
	    */
	   return send_buffer_unsafe (cnt);
}


ret_t 
cherokee_connection_send_response_page_file (cherokee_connection_t *cnt, char *filename)
{
	cherokee_buffer_make_empty  (cnt->buffer);
	cherokee_buffer_read_file (cnt->buffer, filename);

	return send_buffer_unsafe (cnt);
}


ret_t 
cherokee_connection_log_or_delay (cherokee_connection_t *conn)
{
	ret_t ret = ret_ok;

	conn->log_at_end = ! HANDLER_SUPPORT_LENGTH(conn->handler);

	if (conn->log_at_end == 0) {
		if (conn->logger_ref != NULL) {
			ret = cherokee_logger_write_access (conn->logger_ref, conn);
		}
	}

	return ret;
}


ret_t 
cherokee_connection_log_delayed (cherokee_connection_t *conn)
{
	ret_t ret = ret_ok;

	if (conn->log_at_end) {
		if (conn->logger_ref != NULL) {
			ret = cherokee_logger_write_access (conn->logger_ref, conn);
		}		
	}

	return ret_ok;
}
