/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "buffer.h"

ret_t
cherokee_buffer_new  (cherokee_buffer_t **buf)
{
	   CHEROKEE_NEW_STRUCT(n, buffer);

	   n->buf  = NULL;
	   n->size = 0;
	   n->len  = 0;
	   
	   *buf = n;
	   
	   return ret_ok;
}


ret_t
cherokee_buffer_free (cherokee_buffer_t *buf)
{
	if (buf->buf)
		free (buf->buf);
	
	free (buf);
	
	return ret_ok;
}


ret_t
cherokee_buffer_clean (cherokee_buffer_t *buf)
{
	   return cherokee_buffer_make_empty (buf);
}


ret_t
cherokee_buffer_add (cherokee_buffer_t  *buf, char *txt, int size)
{	   
	   int free = buf->size - buf->len;

	   /* Get memory
	    */
	   if (free <= (size+1)) {
		   buf->buf = (char *) realloc(buf->buf, buf->size + size - free + 1);
		   return_if_fail (buf->buf, ret_nomem);
		   
		   buf->size += size - free + 1;
	   }
	   
	   /* Copy	
	    */
	   memcpy (buf->buf + buf->len, txt, size);

	   buf->len += size;
	   buf->buf[buf->len] = '\0';
	   
	   return ret_ok;
}


ret_t 
cherokee_buffer_add_va (cherokee_buffer_t  *buf, char *format, ...)
{
	int len;
	va_list ap;
	CHEROKEE_TEMP (tmp, 200);

	va_start (ap, format);
	len = vsnprintf (tmp, tmp_size, format, ap);
	va_end (ap);

	if (len >= 199) return ret_error;

	return cherokee_buffer_add (buf, tmp, len);
}


ret_t
cherokee_buffer_prepend (cherokee_buffer_t  *buf, char *txt, int size)
{
	   int free = buf->size - buf->len;

	   /* Get memory
	    */
	   if (free <= size) {
		   buf->buf = (char *) realloc(buf->buf, buf->size + size - free + 1);
		   return_if_fail (buf->buf, ret_nomem);
		   
		   buf->size += size - free + 1;
	   }

	   memmove (buf->buf+size, buf->buf, buf->len);

	   memcpy (buf->buf, txt, size);
	   buf->len += size;
	   buf->buf[buf->len] = '\0';
	   
	   return ret_ok;
}


int
cherokee_buffer_is_empty (cherokee_buffer_t *buf)
{
	   return (buf->len == 0);
}


ret_t
cherokee_buffer_make_empty (cherokee_buffer_t *buf)
{
	   buf->len = 0;
	   return ret_ok;
}


ret_t
cherokee_buffer_move_to_begin (cherokee_buffer_t *buf, int pos)
{
	   return_if_fail (pos <= buf->size, ret_error);

	   if (pos <= buf->len) {
			 memcpy (buf->buf, buf->buf+pos, buf->size-pos);
			 buf->len -= pos;
	   }
	   
	   return ret_ok;
}


ret_t
cherokee_buffer_ensure_size (cherokee_buffer_t *buf, int size)
{
	if (size > buf->size) {
		buf->buf = (char *) realloc(buf->buf, size);
		return_if_fail (buf->buf, ret_error);
		buf->size = size;
	}
	   
	return ret_ok;
}


/* This function is based on code from thttpd
 * Copyright: Jef Poskanzer <jef@acme.com>
 *
 * Copies and decodes a string.  It's ok for from and to to be the
 * same string.
 */

void
cherokee_buffer_decode (cherokee_buffer_t *buffer)
{
	char *from;
	char *to;

	from = to = buffer->buf;

	for (; *from != '\0'; ++to, ++from) {
		if (from[0] == '%' && isxdigit( from[1] ) && isxdigit( from[2] )) {
			*to = hexit( from[1] ) * 16 + hexit( from[2] );
			from += 2;
			buffer->len -= 2;
		} else {
			*to = *from;
		}
	}
	*to = '\0';
}

ret_t 
cherokee_buffer_drop_endding (cherokee_buffer_t *buffer, int num_chars)
{
	buffer->buf[buffer->len - num_chars] = '\0';
	buffer->len -= num_chars;

	return ret_ok;
}


void  
cherokee_buffer_swap_chars (cherokee_buffer_t *buffer, char a, char b)
{
	int i = 0;

	if (buffer->buf == NULL) return;

	for (i=0; i<buffer->len; i++) {
		if (buffer->buf[i] == a) {
			buffer->buf[i] = b;
		}
	}
}


crc_t 
cherokee_buffer_crc32 (cherokee_buffer_t  *buf)
{
	return crc32_sz (buf->buf, buf->len);
}


ret_t 
cherokee_buffer_read_file (cherokee_buffer_t *buf, char *filename)
{
	int r, ret, f;
	int remain;
	struct stat info;

	/* Stat() the file
	 */
	r = stat (filename, &info);
	if (r != 0) return ret_error;

	/* Is a regular file?
	 */
	if (S_ISREG(info.st_mode) == 0) {
		return ret_error;
	}

	/* Maybe get memory
	 */
	remain = buf->size - buf->len;
	ret = cherokee_buffer_ensure_size (buf, info.st_size - remain + 1);
	if (ret != ret_ok) return ret;
	
	/* Open the file
	 */
	f = open (filename, O_RDONLY);
	if (f < 0) return ret_error;

	/* Read the content
	 */
	r = buf->len;
	buf->len = read (f, buf->buf+r, info.st_size);
	if (buf->len < 0) {
		buf->len = 0;
		return ret_error;
	}
	buf->len += r;
	
	/* Close it and exit
	 */
	close(f);

	return ret_ok;
}


ret_t
cherokee_buffer_print_debug (cherokee_buffer_t *buf, int len)
{
	int i, length;

	if ((len == -1) || (buf->len <= len)) {
		length = buf->len;
	} else {
		length = len;
	}


	for (i=0; i < length; i++) {
		if (i%16 == 0) {
			printf ("%08x ", i);
		}

		printf ("%02x", buf->buf[i] & 0xFF);

		if ((i+1)%2 == 0) {
			printf (" ");
		}

		if ((i+1)%16 == 0) {
			printf ("\n");
		}

		fflush(stdout);
	}

	return ret_ok;
}


ret_t 
cherokee_buffer_add_version (cherokee_buffer_t *buf, int port, cherokee_version_t ver)
{
	ret_t ret;
	static int  len = 0;
	static char port_str[6];
	static int  port_len = 0;

	if (len == 0) {
		len = strlen (VERSION);
		port_len = snprintf (port_str, 6, "%d", port);
	}

	switch (ver) {
	case ver_full_html:
		cherokee_buffer_ensure_size (buf, buf->len + 29 + len + 6 + port_len + 10);

		cherokee_buffer_add (buf, "<address>Cherokee web server ", 29);
		cherokee_buffer_add (buf, VERSION, len);
		cherokee_buffer_add (buf, " Port ", 6);
		cherokee_buffer_add (buf, port_str, port_len);
		cherokee_buffer_add (buf, "</address>", 10);
		break;

	case ver_port_html:
		cherokee_buffer_ensure_size (buf, buf->len + 34 + port_len + 10);

		cherokee_buffer_add (buf, "<address>Cherokee web server Port ", 34);
		cherokee_buffer_add (buf, port_str, port_len);
		cherokee_buffer_add (buf, "</address>", 10);
		break;

	default:	
		SHOULDNT_HAPPEN;
	}

	return ret;
}
