/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "virtual_server.h"


ret_t 
cherokee_virtual_server_new (cherokee_virtual_server_t **vserver)
{
	int i;

       	CHEROKEE_NEW_STRUCT (vsrv, virtual_server);

	vsrv->domain = NULL;
	vsrv->root   = NULL;

	vsrv->logger       = NULL;
	vsrv->logger_props = NULL;

#ifdef HAVE_LIBGNUTLS
	vsrv->server_cert = NULL;
	vsrv->server_key  = NULL;
#endif

	for (i=0; i<=http_type_400_max; i++) {
		vsrv->error_document400[i].file = NULL;
		vsrv->error_document400[i].url  = NULL;		
	}
	
	/* Initialize GNU/TLS library if present
	 */
	init_gnutls(vsrv);

	cherokee_plugin_table_new (&vsrv->plugins);
	return_if_fail (vsrv->plugins!=NULL, ret_nomem);	   

	*vserver = vsrv;
	
	return ret_ok;
}


ret_t 
cherokee_virtual_server_free (cherokee_virtual_server_t *vserver)
{
	int i;

	if (vserver->domain != NULL) {
		free (vserver->domain);
		vserver->domain = NULL;
	}
	
	if (vserver->root != NULL) {
		free (vserver->root);
		vserver->root = NULL;
	}

	if (vserver->logger != NULL) {
		cherokee_logger_free (vserver->logger);
		vserver->logger = NULL;
	}
	if (vserver->logger_props != NULL) {
		cherokee_table_free (vserver->logger_props);
		vserver->logger_props = NULL;
	}
	
	cherokee_plugin_table_free (vserver->plugins);
	vserver->plugins = NULL;

	for (i=0; i<=http_type_400_max; i++) {
		if (vserver->error_document400[i].file) 
			free (vserver->error_document400[i].file);

		if (vserver->error_document400[i].url)
			free (vserver->error_document400[i].url);
	}

	free (vserver);
	
	return ret_ok;
}



ret_t 
cherokee_virtual_server_clean (cherokee_virtual_server_t *vserver)
{
	int i;

	if (vserver->domain != NULL) {
		free (vserver->domain);
		vserver->domain = NULL;
	}
	
	if (vserver->root != NULL) {
		free (vserver->root);
		vserver->root = NULL;
	}
	
	if (vserver->logger != NULL) {
		cherokee_logger_free (vserver->logger);
		vserver->logger = NULL;
	}
	if (vserver->logger_props != NULL) {
		cherokee_table_free (vserver->logger_props);
		vserver->logger_props = NULL;
	}

	for (i=0; i<=http_type_400_max; i++) {
		if (vserver->error_document400[i].file) 
			free (vserver->error_document400[i].file);

		if (vserver->error_document400[i].url)
			free (vserver->error_document400[i].url);
	}

	return ret_ok;
}


ret_t 
cherokee_virtual_server_set (cherokee_virtual_server_t *vserver, char *web_dir,  char *handler_name, cherokee_plugin_table_entry_t *plugin_entry)
{
	ret_t ret;

	ret = cherokee_plugin_table_add (vserver->plugins, web_dir, plugin_entry);
	if (ret != ret_ok) {
		switch (ret) {
		case ret_file_not_found:
			PRINT_ERROR ("Can't load handler '%s': File not found\n", handler_name);
			break;
		default:
			PRINT_ERROR ("Can't load handler '%s': Unknown error\n", handler_name);
		}
	}
	
	return ret;

}


static ret_t
init_gnutls (cherokee_virtual_server_t *vsrv)
{
#ifdef HAVE_LIBGNUTLS

	int r;
	int need_dh_params  = 0;
	gnutls_dh_params _dh_params[2];

	int need_rsa_params = 0;
	gnutls_rsa_params _rsa_params[2];
		

	if ((vsrv->server_cert == NULL) ||
	    (vsrv->server_key  == NULL))
	{
		return ret_error;
	}
	
	r = gnutls_global_init();
	if (r < 0) {
		PRINT_ERROR ("gnutls_global_init() = %d\n", r);
		return ret_error;
	}

	r = gnutls_global_init_extra();
	if (r < 0) {
		PRINT_ERROR ("gnutls_global_init_extra() = %d\n", r);
		return ret_error;
	}

	r = gnutls_certificate_allocate_credentials (&vsrv->credentials[0]);
	if (r < 0) {
		PRINT_ERROR ("gnutls_certificate_allocate_credentials (&vsrv->credentials[0]=%p) = %d\n", &vsrv->credentials[0], r);
		return ret_error;
	}

	r = gnutls_certificate_set_x509_key_file (vsrv->credentials[0],
						  vsrv->server_cert,
						  vsrv->server_key,
						  GNUTLS_X509_FMT_PEM);
	if (r < 0) {
		PRINT_ERROR ("gnutls_certificate_set_x509_key_file ('%s', '%s', GNUTLS_X509_FMT_PEM) = %d\n",  
			     vsrv->server_cert, vsrv->server_key, r);
		return ret_error;
	}

	if (vsrv->ca_cert != NULL) {
		r = gnutls_certificate_set_x509_trust_file (vsrv->credentials[0],
							    vsrv->ca_cert,
							    GNUTLS_X509_FMT_PEM);
		if (r < 0) {
			PRINT_ERROR ("gnutls_certificate_set_x509_trust_file (..) = %d\n",  r);
			return ret_error;
		}
	}
	
	/* Add ciphers 
	 */
	vsrv->cipher_priority[0] = GNUTLS_CIPHER_RIJNDAEL_128_CBC;
	vsrv->cipher_priority[1] = GNUTLS_CIPHER_ARCFOUR_128;
	vsrv->cipher_priority[2] = GNUTLS_CIPHER_3DES_CBC;
	vsrv->cipher_priority[3] = GNUTLS_CIPHER_ARCFOUR_40;
	vsrv->cipher_priority[4] = 0;

	/* Add key exchange methods
	 */
	vsrv->kx_priority[0] = GNUTLS_KX_RSA;
	vsrv->kx_priority[1] = GNUTLS_KX_RSA_EXPORT;
	vsrv->kx_priority[2] = GNUTLS_KX_DHE_RSA;
	vsrv->kx_priority[3] = GNUTLS_KX_DHE_DSS;
	vsrv->kx_priority[4] = 0;

	/* Add MAC Algorithms
	 */
	vsrv->mac_priority[0] = GNUTLS_MAC_MD5;
	vsrv->mac_priority[1] = GNUTLS_MAC_SHA;
	vsrv->mac_priority[2] = 0;

	/* Add Compression algorithms
	 */
	vsrv->comp_priority[0] = GNUTLS_COMP_NULL;
	vsrv->comp_priority[1] = GNUTLS_COMP_ZLIB;
	vsrv->comp_priority[2] = GNUTLS_COMP_LZO;
	vsrv->comp_priority[3] = 0;

	/* Add protocols
	 */
	vsrv->protocol_priority[0] = GNUTLS_TLS1;
	vsrv->protocol_priority[0] = GNUTLS_SSL3;
	vsrv->protocol_priority[0] = 0;

	

#endif /* HAVE_LIBGNUTLS */

	return ret_ok;
}


ret_t 
cherokee_virtual_server_get_error_page (cherokee_virtual_server_t *vserver, cherokee_http_t error_code, char **filename_ref)
{
	if (! http_type_400(error_code)) {
		return ret_error;
	}

	if (vserver->error_document400[error_code-400].file != NULL) {
		*filename_ref = vserver->error_document400[error_code-400].file;
		return ret_ok;
	}
	
	return ret_not_found;
}


ret_t 
cherokee_virtual_server_get_error_url (cherokee_virtual_server_t *vserver, cherokee_http_t error_code, char **url_ref)
{
	if (! http_type_400(error_code)) {
		return ret_error;
	}

	if (vserver->error_document400[error_code-400].url != NULL) {
		*url_ref = vserver->error_document400[error_code-400].url;
		return ret_ok;
	}
	
	return ret_not_found;
}
