/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <fcntl.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <errno.h>
#include <sys/ioctl.h>

#include "common.h"
#include "server.h"
#include "socket.h"
#include "connection.h"
#include "log.h"
#include "ncpus.h"

#ifdef HAVE_PTHREAD
#include "thread.h"
#endif


ret_t
cherokee_server_new  (cherokee_server_t **srv)
{
	/* Get memory
	 */
	CHEROKEE_NEW_STRUCT(n, server);
	
	/* Sockets
	 */
	n->socket = -1;
	n->ipv6   =  1;
	
	/* Threads
	 */
	INIT_LIST_HEAD(&n->thread_list);
	cherokee_thread_new (&n->main_thread, n, 0);

	/* Default Index files list
	 */
	INIT_LIST_HEAD(&n->index_list);

	/* Server config
	 */
	n->port           = 80;
	n->listen_to      = NULL;
	n->fdwatch_msecs  = 999;
	n->timeout        = 15;
	n->keepalive      = 1;
	n->hideversion    = 0;
	n->hideservername = 0;
	n->mimetypes_file = NULL;
	n->userdir        = NULL;
	n->userdir_handler= NULL;

	n->ncpus          = -1;
	n->thread_num     = 0;

	n->chroot         = NULL;
	n->chrooted       = 0;

	n->user_orig      = getuid();
	n->user           = n->user_orig;
	n->group_orig     = getgid();
	n->group          = n->group_orig;


	/* Virtual servers table
	 */
	cherokee_table_new (&n->vservers);
	return_if_fail (n->vservers!=NULL, ret_nomem);

	cherokee_virtual_server_new (&n->vserver_default);
	return_if_fail (n->vserver_default!=NULL, ret_nomem);

		
	/* Encoders 
	 */
	cherokee_encoder_table_new (&n->encoders);
	return_if_fail (n->encoders != NULL, ret_nomem);

	
	/* Loggers
	 */
	n->log_flush_next   = 0;
	n->log_flush_elapse = 10;

	cherokee_logger_table_new (&n->loggers);
	return_if_fail (n->loggers != NULL, ret_nomem);


	/* Return the object
	 */
	*srv = n;

	return ret_ok;
}


static inline ret_t
change_execution_user (cherokee_server_t *srv)
{
	int error;

	/* Change of group requested
	 */
	if (srv->group != srv->group_orig) {
		error = setgid (srv->group);
		if (error != 0) {
			PRINT_ERROR ("Can't change group to GID %d, running with GID=%d\n",
				     srv->group, srv->group_orig);
		}
	}

	/* Change of user requested
	 */
	if (srv->user != srv->user_orig) {
		error = setuid (srv->user);		
		if (error != 0) {
			PRINT_ERROR ("Can't change user to UID %d, running with UID=%d\n",
				     srv->user, srv->user_orig);
		}
	}

	return ret_ok;
}


void  
cherokee_server_set_min_latency (cherokee_server_t *srv, int msecs)
{
	srv->fdwatch_msecs = msecs;
}


static ret_t
initialize_server_socket4 (cherokee_server_t *srv)
{
	ret_t ret;
	int   on = 1;
	struct sockaddr_in  sockaddr;
	
	srv->socket = socket (AF_INET, SOCK_STREAM, 0);
	if (srv->socket <= 0) {
		PRINT_ERROR ("Error creating IPv4 server socket\n");
		exit(EXIT_CANT_CREATE_SERVER_SOCKET4);
	}

	/* Set 'close-on-exec'
	 */
	fcntl(srv->socket, F_SETFD, FD_CLOEXEC);
		
	/* To re-bind without wait to TIME_WAIT
	 */
	ret = setsockopt (srv->socket, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
	return_if_fail (ret == 0, ret_error);

	/* Bind the socket
	 */
	sockaddr.sin_port   = htons (srv->port);
	sockaddr.sin_family = AF_INET;

	if (srv->listen_to == NULL) {
		sockaddr.sin_addr.s_addr = INADDR_ANY; 
	} else {
#ifdef HAVE_INET_PTON
		inet_pton (sockaddr.sin_family, srv->listen_to, &sockaddr.sin_addr);
#else
		/* IPv6 needs inet_pton. inet_addr just doesn't work without
		 * it. We'll suppose then that we haven't IPv6 support 
		 */
		sockaddr.sin_addr.s_addr = inet_addr (srv->listen_to);
#endif
	}

	ret = bind (srv->socket, (struct sockaddr *)&sockaddr, sizeof(sockaddr));
	return ret;
}


static ret_t
initialize_server_socket6 (cherokee_server_t *srv)
{
#ifdef HAVE_IPV6
	ret_t ret;
	int   on = 1;
	struct sockaddr_in6 sockaddr;


	/* Create the socket
	 */
	srv->socket = socket (AF_INET6, SOCK_STREAM, 0);
	if (srv->socket <= 0) {
		PRINT_ERROR ("Error creating IPv6 server socket.. switching to IPv4\n");
		srv->ipv6 = 0;
		return ret_error;
	}

	/* Set 'close-on-exec'
	 */
	fcntl(srv->socket, F_SETFD, FD_CLOEXEC);
	
	/* To re-bind without wait to TIME_WAIT
	 */
	ret = setsockopt (srv->socket, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
	return_if_fail (ret == 0, ret_error);

	/* Bind the socket
	 */
	sockaddr.sin6_port   = htons (srv->port);
	sockaddr.sin6_family = AF_INET6;		

	if (srv->listen_to == NULL) {
		sockaddr.sin6_addr = in6addr_any; 
	} else {
		inet_pton (sockaddr.sin6_family, srv->listen_to, &sockaddr.sin6_addr);
	}

	ret = bind (srv->socket, (struct sockaddr *)&sockaddr, sizeof(sockaddr));
	return ret;
#endif
}


static ret_t
initialize_server_socket (cherokee_server_t *srv)
{
	int   flags;
	ret_t ret = ret_ok;

#ifdef HAVE_IPV6
	if (srv->ipv6) {
		ret = initialize_server_socket6 (srv);
	} 
#endif
	if ((ret != ret_ok) || (srv->ipv6 == 0)) {
		ret = initialize_server_socket4 (srv);		
	}

	if (ret != 0) {
		uid_t uid = getuid();
		gid_t gid = getgid();

		PRINT_ERROR ("Can't bind() socket (port=%d, UID=%d, GID=%d)\n", srv->port, uid, gid);
		return ret_error;
	}

	printf ("Listening on port %d\n", srv->port);
	   
	/* Set no-delay mode
	 */
	flags = fcntl (srv->socket, F_GETFL, 0);
	return_if_fail (flags != -1, ret_error);
	
	ret = fcntl (srv->socket, F_SETFL, flags | O_NDELAY);
	return_if_fail (ret >= 0, ret_error);
	
	
	/* Listen
	 */
	ret = listen (srv->socket, 1024);
	if (ret < 0) {
		close (srv->socket);
		return_if_fail (0, ret_error);
	}


	/* Add the socket to fdpoll
	 */
	cherokee_fdpoll_add (srv->main_thread->fdpoll, srv->socket, 0);

	return ret_ok;
}




static ret_t
initialize_server_threads (cherokee_server_t *srv)
{
#ifdef HAVE_PTHREAD
	int i;

	for (i=0; i<srv->ncpus - 1; i++) {
		ret_t ret;
		cherokee_thread_t *thread;

		ret = cherokee_thread_new (&thread, srv, 1);
		if (ret != ret_ok) {
			return ret;
		}
		
		list_add ((list_t *)thread, &srv->thread_list);
		srv->thread_num++;
	}
#endif

	return ret_ok;
}


ret_t
cherokee_server_init (cherokee_server_t *srv) 
{   
	ret_t ret;

	/* If the server has a previous server socket opened, Eg:
	 * because a SIGHUP, it shouldn't init the server socket.
	 */
	if (srv->socket == -1) {
		ret = initialize_server_socket (srv);
		if (ret != ret_ok) return ret;
	}

        /* Threads / CPUs
	 */
	if (srv->ncpus == -1) {
		dcc_ncpus (&srv->ncpus);
	}
	
	if (srv->ncpus != 1) {
		ret = initialize_server_threads (srv);
		if (ret != ret_ok) return ret;
	}

	/* Change the user
	 */
	ret = change_execution_user (srv);
	if (ret != ret_ok) {
		return ret;
	}
	
	/* Chroot
	 */
	if (srv->chroot) {
		srv->chrooted = (chroot (srv->chroot) == 0);
		if (srv->chrooted == 0) {
			PRINT_ERROR ("Cannot chroot() to '%s'\n", srv->chroot);
		}
	} 

	chdir ("/");

	return ret_ok;
}


static inline void
close_all_connections (cherokee_server_t *srv)
{
	list_t *i;

	cherokee_thread_close_all_connections (srv->main_thread);
	
	list_for_each (i, &srv->thread_list) {
		cherokee_thread_close_all_connections (THREAD(i));
	}
}


static void
for_each_func_free_vserver (const char *key, void *vserver)
{	
	cherokee_virtual_server_free (vserver);
}


ret_t 
cherokee_server_clean (cherokee_server_t *srv)
{
	/* Close all active connections
	 */
	close_all_connections (srv);


	/* Clean the configuration
	 */
	if (srv->listen_to != NULL) {
		free (srv->listen_to);
		srv->listen_to = NULL;
	}

	if (srv->mimetypes_file != NULL) {
		free (srv->mimetypes_file);
		srv->mimetypes_file = NULL;
	}

	if (srv->userdir != NULL) {
		free (srv->userdir);
		srv->userdir = NULL;
	}

	if (srv->userdir_handler != NULL) {
		cherokee_plugin_table_entry_free (srv->userdir_handler);
		srv->userdir_handler = NULL;
	}

	if (srv->chroot != NULL) {
		free (srv->chroot);
		srv->chroot = NULL;
	}
	
	/* Clean the encoders table
	 */
	cherokee_encoder_table_clean (srv->encoders);

	/* Clean the loggers table
	 */
	cherokee_logger_table_clean (srv->loggers);

	/* Clean the virtual servers table
	 */
	cherokee_virtual_server_clean (srv->vserver_default);

	/* Clean the virtual servers table
	 */
	cherokee_table_foreach (srv->vservers, for_each_func_free_vserver);
	cherokee_table_clean (srv->vservers);

	/* Clean the index list
	 */
	cherokee_list_free (&srv->index_list, free);
	INIT_LIST_HEAD(&srv->index_list);
 
	srv->hideversion    = 0;
	srv->hideservername = 0;

	return ret_ok;
}


ret_t
cherokee_server_free (cherokee_server_t *srv)
{
	close (srv->socket);

	cherokee_encoder_table_free (srv->encoders);
	cherokee_logger_table_free (srv->loggers);

	cherokee_list_free (&srv->index_list, free);

	/* Virtual servers
	 */
	cherokee_virtual_server_free (srv->vserver_default);
	srv->vserver_default = NULL;

	cherokee_table_foreach (srv->vservers, for_each_func_free_vserver);
	cherokee_table_free (srv->vservers);


	if (srv->listen_to != NULL) {
		free (srv->listen_to);
		srv->listen_to = NULL;
	}

	if (srv->mimetypes_file != NULL) {
		free (srv->mimetypes_file);
		srv->mimetypes_file = NULL;
	}

	if (srv->userdir != NULL) {
		free (srv->userdir);
		srv->userdir = NULL;
	}

	if (srv->userdir_handler != NULL) {
		cherokee_plugin_table_entry_free (srv->userdir_handler);
		srv->userdir_handler = NULL;
	}
	
	if (srv->chroot != NULL) {
		free (srv->chroot);
		srv->chroot = NULL;
	}

	free (srv);
	
	return ret_ok;
}


static inline int
look_for_new_connection (cherokee_server_t *srv)
{
	int new_socket;
	list_t *thread;

	cherokee_thread_t     *lazy_thread;
	cherokee_connection_t *new_conn;

	/* Return if there're no new connections
	 */
	if (cherokee_fdpoll_check (srv->main_thread->fdpoll, srv->socket, 0) == 0) {
		return 0;
	}

	/* Get the thread
	 */
	lazy_thread = srv->main_thread;

	if (! list_empty(&srv->thread_list)) {
		list_for_each (thread, &srv->thread_list) {
			if (cherokee_thread_connection_num (lazy_thread) >
			    cherokee_thread_connection_num (THREAD(thread))) 
			{
				lazy_thread = THREAD(thread);
			}
		}
	}

	/* Get a new connection object
	 */
	cherokee_thread_get_new_connection (lazy_thread, &new_conn);

	/* Accept the connection in the new socket
	 */
	cherokee_socket_accept (CONN_SOCK(new_conn), srv->socket);

	/* Add the configured connection to the thread
	 */
	cherokee_thread_add_connection (lazy_thread, new_conn);

	return 1;
}


void static
flush_vserver (const char *key, void *value)
{
	/* There's no logger in this virtual server
	 */
	if ((value == NULL) || (VSERVER_LOGGER(value) == NULL))
		return;

	cherokee_logger_flush (VSERVER_LOGGER(value));
}


void static 
flush_logs (cherokee_server_t *srv)
{
	flush_vserver (NULL, srv->vserver_default);
	cherokee_table_foreach (srv->vservers, flush_vserver);
}


void
cherokee_server_step (cherokee_server_t *srv)
{
	ret_t ret;
 	int   i   = MAX_NEW_CONNECTIONS_PER_STEP; 

	/* Watch the sockets, look for new connections
	 */
	do {
		cherokee_fdpoll_reset (srv->main_thread->fdpoll, srv->socket);
		cherokee_fdpoll_watch (srv->main_thread->fdpoll, srv->fdwatch_msecs);
		ret = look_for_new_connection(srv);
	} while (ret && (i-- > 0));


	/* Get the time
	 */
 	srv->bogo_now = time (NULL); 
	ret = cherokee_thread_step (srv->main_thread);

	/* Logger flush 
	 */
	if (srv->log_flush_next < srv->bogo_now) {
		flush_logs (srv);
		srv->log_flush_next = srv->bogo_now + srv->log_flush_elapse;
	}
}


ret_t 
cherokee_server_read_config_file (cherokee_server_t *srv, char *filename)
{
	int error;

	extern FILE *yyin;
	extern int yyparse (void *);

	/* Set the file to read
	 */
	yyin = fopen (filename, "r");
	if (yyin == NULL) {
		PRINT_ERROR("Can't read the configuration file: '%s'\n", filename);
	}

	/* Cooooome on :-)
	 */
	error = yyparse ((void *)srv);

	if (yyin != NULL)
		fclose (yyin);

	if (error != 0)
		return ret_error;

	return ret_ok;
}


ret_t 
cherokee_server_read_config_string (cherokee_server_t *srv, char *config_string)
{
	int   error;
	void *bufstate;

	extern int  yy_scan_string (void *);
	extern int  yyparse (void *);
	extern void yy_switch_to_buffer (void *);
	extern void yy_delete_buffer (void *);

	bufstate = (void *) yy_scan_string (config_string);
	yy_switch_to_buffer(bufstate);

	error = yyparse((void *)srv);

	yy_delete_buffer (bufstate);

	if (error != 0)
		return ret_error;

	return ret_ok;
}


ret_t 
cherokee_server_handle_HUP (cherokee_server_t *srv)
{
	ret_t ret;

	if (srv->chrooted) {
		fprintf (stderr, 
			 "WARNING: Chrooted cherokee cannot be reloaded. "
			 "Please, stop and restart it again.\n");
		return ret_ok;
	}


	ret = cherokee_server_clean (srv);
	if (ret != ret_ok) {
		exit(EXIT_SERVER_CLEAN);
	}

	ret = cherokee_server_read_config_file (srv, CHEROKEE_CONFDIR"/cherokee.conf");
	if (ret != ret_ok) {
		exit(EXIT_SERVER_READ_CONFIG);
	}

	ret = cherokee_server_init (srv);
	if (ret != ret_ok) {
		exit(EXIT_SERVER_INIT);	
	}

	return ret_ok;
}
