/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include "common.h"
#include "mime.h"
#include "handler_file.h"
#include "connection.h"

#if defined(LINUX_SENDFILE_API) || defined(SOLARIS_SENDFILE_API)
# include <sys/sendfile.h>
#elif defined(HPUX_SENDFILE_API)
# include <sys/socket.h>
# include <sys/uio.h>
#elif defined(FREEBSD_SENDFILE_API)
# include <sys/types.h>
# include <sys/socket.h>
# include <sys/uio.h>
#elif defined(LINUX_BROKEN_SENDFILE_API)
 extern int32 sendfile (int out_fd, int in_fd, int32 *offset, uint32 count);
#endif


ret_t
cherokee_handler_file_new  (cherokee_handler_t **hdl, void *cnt, cherokee_table_t *properties)
{
	CHEROKEE_NEW_STRUCT (n, handler_file);
	
	/* Init the base class object
	 */
	cherokee_handler_init_base(HANDLER(n), cnt);
	HANDLER(n)->connection  = cnt;

	HANDLER(n)->support = hsupport_length | hsupport_range;

	HANDLER(n)->init        = (handler_func_init_t) cherokee_handler_file_init;
	HANDLER(n)->free        = (handler_func_free_t) cherokee_handler_file_free;
	HANDLER(n)->step        = (handler_func_step_t) cherokee_handler_file_step;
	HANDLER(n)->add_headers = (handler_func_add_headers_t) cherokee_handler_file_add_headers;

	*hdl = HANDLER(n);
	
	/* Init
	 */
	n->offset = 0;
	n->mime   = NULL;

	return ret_ok;
}


ret_t 
cherokee_handler_file_init (cherokee_handler_file_t *n)
{
	int   ret;
	char *ext;
	cherokee_connection_t *conn;

	conn = CONN(HANDLER(n)->connection);

	cherokee_buffer_add (conn->local_directory, conn->request->buf, conn->request->len); /* (1) */
	n->fd = open (conn->local_directory->buf, O_RDONLY);

	/* Cannot open the file
	 */
	if (n->fd != -1) {
		ret = fstat (n->fd, &n->info);
	} else {
		ret = stat (conn->local_directory->buf, &n->info);
	}

	cherokee_buffer_drop_endding (conn->local_directory, conn->request->len); /* (1') */


	if (n->fd == -1) {
		if ((ret == 0) && S_ISREG(n->info.st_mode)) {
			conn->error_code = http_access_denied;
		} else {
			conn->error_code = http_not_found;
		}
		return ret_error;
	}


	/* Is dir?
	 */
	if (S_ISDIR(n->info.st_mode)) {
		conn->error_code = http_access_denied;
		return ret_error;		
	}


	/* Range
	 */
	if (conn->range_start != 0) {
		n->offset = conn->range_start;
		lseek (n->fd, n->offset, SEEK_SET);
		conn->error_code = http_partial_content;
	}

	if (conn->range_end == 0) {
		conn->range_end = n->info.st_size;
	}


	/* Maybe use sendfile
	 */
#if HAVE_SENDFILE
	n->using_sendfile = ((n->info.st_size >= 32768) && 
			     (n->info.st_size < 2147483647) &&
			     conn->encoder == NULL);

	if (n->using_sendfile) {
		cherokee_connection_set_cork(conn, 1);
	}
#endif


	/* Look for the mime type
	 */
	ext = rindex (conn->request->buf, '.');
	if (ext != NULL) {
		cherokee_mime_t *m;

		cherokee_mime_get_default (&m);		
		n->mime = cherokee_mime_get (m, ext+1);
	}

	return ret_ok;
}


ret_t
cherokee_handler_file_free (cherokee_handler_file_t *fhdl)
{
	if (fhdl->fd != -1) {
		close (fhdl->fd);
		fhdl->fd = -1;
	}

	free (fhdl);
	
	return ret_ok;
}


ret_t
cherokee_handler_file_step (cherokee_handler_file_t *fhdl, 
			    cherokee_buffer_t       *buffer)
{
	int total;

#if HAVE_SENDFILE
	if (fhdl->using_sendfile) {
		ssize_t sent;
		cherokee_connection_t *conn = CONN(HANDLER(fhdl)->connection);
		

#ifdef FREEBSD_SENDFILE_API
		/* FreeBSD sendfile: in_fd and out_fd are reversed
		 *
		 * int
		 * sendfile(int fd, int s, off_t offset, size_t nbytes,
		 * struct sf_hdtr *hdtr, off_t *sbytes, int flags);
		 */
		sent = sendfile (fhdl->fd,                        /* int             fd     */
				 SOCKET_FD(conn->socket),         /* int             s      */
				 &fhdl->offset,                   /* off_t           offset */
				 conn->range_end - fhdl->offset,  /* size_t          nbytes */
				 NULL,                            /* struct sf_hdtr *hdtr   */
				 &fhdl->offset,                   /* off_t          *sbytes */ 
				 0);                              /* int             flags  */
#else
		sent = sendfile (SOCKET_FD(conn->socket),         /* int    out_fd */
				 fhdl->fd,                        /* int    in_fd  */
				 &fhdl->offset,                   /* off_t *offset */
				 conn->range_end - fhdl->offset); /* size_t count  */
#endif

		/* cherokee_handler_file_init() activated the TCP_CORK flags.
		 * After it, the header was sent.  And now, the first
		 * chunk of the file with sendfile().  It's time to turn
		 * off again the TCP_CORK flag
		 */
		if (conn->tcp_cork) {
			cherokee_connection_set_cork(conn, 0);
		}


		if (sent < 0) {
			return ret_error;
		}

		if (fhdl->offset >= conn->range_end) {
			return ret_eof;
		}

		return ret_ok_and_sent;
	}
#endif


	total = read (fhdl->fd, buffer->buf, buffer->size);	
	switch (total) {
	case 0:
		return ret_eof;
	case -1:
		return ret_error;
	default:
		buffer->len = total;
		fhdl->offset += total;
	}	

	/* Maybe it was the last file chunk
	 */
	if (fhdl->offset >= CONN(HANDLER(fhdl)->connection)->range_end) {
		return ret_eof_have_data;
	}

	return ret_ok;
}


ret_t
cherokee_handler_file_add_headers (cherokee_handler_file_t *fhdl,
				   cherokee_buffer_t       *buffer)
{
	int length;

	/* Add "Content-length:" header
	 */
	if (fhdl->fd != -1) {
		/* We stat()'ed the file in the handler constructor
		 */
		length = HANDLER_CONN(fhdl)->range_end - HANDLER_CONN(fhdl)->range_start;
		if (length < 0) {
			length = 0;
		}

		if (HANDLER(fhdl)->support & hsupport_length) {
			length = snprintf (gbl_buffer, gbl_buffer_size, "Content-length: %d"CRLF, length);
			cherokee_buffer_add (buffer, gbl_buffer, length);
		}

	} else {
		/* Can't use Keep-alive w/o "Content-length:", so
		 */
		HANDLER_CONN(fhdl)->keep_alive = 0;
	}

	/* Add "Content-Type:" header
	 */
	if (fhdl->mime != NULL) {
		length = snprintf (gbl_buffer, gbl_buffer_size, "Content-Type: %s"CRLF, fhdl->mime);
		cherokee_buffer_add (buffer, gbl_buffer, length);		
	}

	return ret_ok;
}



/*   Library init function
 */

static int _file_is_init = 0;

void
file_init ()
{
	cherokee_mime_t *mime;

	/* Init flag
	 */
	if (_file_is_init) {
		return;
	}
	_file_is_init = 1;


	/* Library dependences
	 */
	cherokee_mime_init (&mime);
}
