/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>

#include "plugin_table.h"
#include "handler.h"
#include "static_handlers.h"
#include "library.h"



static ret_t
load_static_table (cherokee_plugin_table_t *pt)
{
	int i;
	
	for (i=0; static_handlers[i].name; i++) {
		ret_t ret;
		ret = cherokee_table_add (&pt->handlers, static_handlers[i].name, static_handlers[i].func); 
		if (ret != ret_ok) 
			return ret;
	}

	return ret_ok;
}



ret_t
cherokee_plugin_table_new  (cherokee_plugin_table_t **pt)
{
	CHEROKEE_NEW_STRUCT(n, plugin_table);

	/* Init tables 
	 */
	cherokee_table_init (&n->table);
	cherokee_table_init (&n->handlers);
	
	/* Add the static modules
	 */
	load_static_table (n);

	/* Return the object
	 */
	*pt = n;
	
	return ret_ok;
}


ret_t
cherokee_plugin_table_free (cherokee_plugin_table_t *pt)
{
	cherokee_table_clean (&pt->table);
	cherokee_table_clean (&pt->handlers);
	
	free (pt);
	
	return ret_ok;
}


ret_t 
cherokee_plugin_table_get (cherokee_plugin_table_t *pt, char *requested_url, cherokee_plugin_table_entry_t **plugin_entry, cherokee_buffer_t *web_directory)
{
	char *slash;
	cherokee_plugin_table_entry_t *entry = NULL;

	cherokee_buffer_add (web_directory, requested_url, strlen(requested_url));

	do {
		entry = cherokee_table_get (&pt->table, web_directory->buf);
		
		/* Found
		 */
		if (entry != NULL)
			goto go_out;
 
		if (web_directory->len <= 0)
			goto go_out;


		/* Modify url for next loop:
		 * Find the last / and finish the string there.
		 */
		if (web_directory->buf[web_directory->len-1] == '/') {
			web_directory->buf[web_directory->len-1] = '\0';
			web_directory->len--;
		} else {
			slash = rindex (web_directory->buf, '/');
			if (slash != NULL) {
                                slash[1] = '\0';
				web_directory->len -= ((web_directory->buf + web_directory->len) - slash -1);
			} else {
				
			}
		}
	} while (entry == NULL);


go_out:	
	/* Set the in-out values
	 */
	*plugin_entry = entry;

	return (entry == NULL) ? ret_error : ret_ok;
}


/*
static inline handler_func_new_t
load_plugin_library (char *plugin_name)
{
	char lib[200];
	char sym[200];

	void (*init) (void);
	handler_func_new_t ret;

 	snprintf (lib, 200, "libcherokee_%s." SO_SUFFIX, plugin_name);
	snprintf (sym, 200, "%s_init", plugin_name);

	init = get_sym_from_dylib (lib, sym);
	if (init != NULL) {
		init();
	}

 	snprintf (sym, 200, "cherokee_handler_%s_new", plugin_name);
	ret = get_sym_from_dylib (lib, sym);
	
	return ret;
}
*/

static inline handler_func_new_t
load_plugin_library (char *plugin_name)
{
	cherokee_library_t *clib;
	encoder_func_new_t ret;
	char sym[200];
	
	cherokee_library_new (&clib, plugin_name);

 	snprintf (sym, 200, "cherokee_handler_%s_new", plugin_name);
	cherokee_library_get (clib, sym, &ret);

	cherokee_library_free (clib);
	
	return ret;
}


static inline handler_func_new_t
get_loaded_function (char *plugin_name)
{
	void (*func_init) (void);

	/* If the library init function exist, it should be
	 * called to, maybe, init the library.
	 */
	snprintf (gbl_buffer, gbl_buffer_size, "%s_init", plugin_name);
	func_init = dlsym(NULL, gbl_buffer);

#if 0
	printf ("get_loaded_function: %s: %p\n", gbl_buffer, func_init);
#endif

	if ((func_init == NULL) || (dlerror() != NULL)) {
		return NULL;
	} 
	func_init();

	
	snprintf (gbl_buffer, gbl_buffer_size, "cherokee_handler_%s_new", plugin_name);
	return (handler_func_new_t) dlsym(NULL, gbl_buffer);	
}


ret_t 
cherokee_plugin_table_fill_entry (cherokee_plugin_table_t *pt, char *handler_name, cherokee_plugin_table_entry_t *plugin_entry)
{
	ret_t              ret;
	handler_func_new_t func;

	/* Get the cached "handler"_new function
	 */
	func = (handler_func_new_t) cherokee_table_get (TABLE(&pt->handlers), handler_name);
	if (func == NULL) {

		/* Maybe the plugin is compiled statically
		 */
		func = (handler_func_new_t) get_loaded_function (handler_name);

		if (func == NULL) {
			func = load_plugin_library (handler_name);
			if (func == NULL) {
				return ret_error;
			}
		}
		
		
		/* Add to table
		 */
		ret = cherokee_table_add (&pt->handlers, handler_name, (void*)func); 
		if (ret != ret_ok) {
			return ret_error;
		}
	}
	
	plugin_entry->handler_new_func = func;

	return ret_ok;
}


ret_t inline
cherokee_plugin_table_add (cherokee_plugin_table_t *pt, char *dir, cherokee_plugin_table_entry_t *plugin_entry)
{
	/* Add to "dir <-> plugin_entry" table
	 */
	return cherokee_table_add (TABLE(&pt->table), dir, (void*)plugin_entry);
}


