/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "thread.h"
#include "server.h"
#include "socket.h"
#include "connection.h"


#ifdef HAVE_PTHREAD
static void *
thread_routine (void *data)
{
        cherokee_thread_t *thread = THREAD(data);
        cherokee_server_t *srv = THREAD(data)->server;

        while (thread->exit == 0) {
                cherokee_fdpoll_watch (thread->fdpoll, srv->fdwatch_msecs);
                cherokee_thread_step (thread);
        }

        return NULL;
}
#endif


ret_t 
cherokee_thread_new  (cherokee_thread_t **thd, void *server, int real_thread)
{
	CHEROKEE_NEW_STRUCT (n, thread);

	/* Init
	 */
	INIT_LIST_HEAD((list_t*)&n->base);
	INIT_LIST_HEAD((list_t*)&n->active_list);
	INIT_LIST_HEAD((list_t*)&n->reuse_list);
	
	cherokee_fdpoll_new(&n->fdpoll);

	n->active_list_num = 0;
	n->server = server;
	n->exit = 0;
	
	/* Return the object
	 */
	*thd = n;
	
	/* Thread
	 */
	if (real_thread) {	
#ifdef HAVE_PTHREAD
		pthread_create (&n->thread, NULL, thread_routine, n);
#else
		SHOULDNT_HAPPEN;
#endif
	}
	
	return ret_ok;
}


static void
conn_set_mode (cherokee_thread_t *thd, cherokee_connection_t *conn, cherokee_socket_status_t s)
{
	cherokee_socket_set_status (conn->socket, s);
	cherokee_fdpoll_set_mode (thd->fdpoll, SOCKET_FD(conn->socket), s);
}


static void
add_connection (cherokee_thread_t *thd, cherokee_connection_t *conn)
{
	list_add ((list_t *)conn, &thd->active_list);
	thd->active_list_num++;
}


static void
del_connection (cherokee_thread_t *thd, cherokee_connection_t *conn)
{
	list_del ((list_t *)conn);
	thd->active_list_num--;
}


static void
purge_closed_connection (cherokee_thread_t *thread, cherokee_connection_t *conn)
{
	/* Delete from file descriptors poll
	 */
	cherokee_fdpoll_del (thread->fdpoll, SOCKET_FD(conn->socket));

	/* Maybe have a delayed log
	 */
	cherokee_connection_log_delayed (conn);

	/* Clean the connection
	 */
	cherokee_connection_mrproper (conn);

	/* Delete from active connections list and 
	 * add it to the rusable one
	 */
	del_connection (thread, conn);
	list_add ((list_t *)conn, &thread->reuse_list);
}


static void
maybe_purge_closed_connection (cherokee_thread_t *thread, cherokee_connection_t *conn)
{
	cherokee_server_t *srv = thread->server;

	/* No-keepalive connection
	 */
	if (conn->keep_alive == 0) {
		purge_closed_connection (thread, conn);
		return;
	}	

	/* Keepalive
	 */
	if (conn->tcp_cork) {
		cherokee_connection_set_cork (conn, 0);
	}
	
	cherokee_connection_log_delayed (conn);
	cherokee_connection_clean (conn);
	conn_set_mode (thread, conn, socket_reading);
	conn->timeout = srv->bogo_now + srv->timeout;
}


static ret_t 
process_active_connections (cherokee_thread_t *thd)
{
	ret_t ret;
	list_t *i, *tmp;

	cherokee_connection_t *conn = NULL;
	cherokee_server_t     *srv  = thd->server;


	/* Process active connections
	 */
	list_for_each_safe (i, tmp, (list_t*)&thd->active_list) {
		conn = CONN(i);

		/* May the connection was too much time w/o any work
		 */
		if (conn->timeout < srv->bogo_now) {
			purge_closed_connection (thd, conn);
			continue;
		}

		/* Inspect the file descriptor
		 */
		if (cherokee_fdpoll_check (thd->fdpoll, 
					   SOCKET_FD(conn->socket), 
					   SOCKET_STATUS(conn->socket)) == 0) 
		{
			continue;
		}

		/* The connection has work, so..
		 */
		conn->timeout = srv->bogo_now + srv->timeout;

		/* Phases
		 */
		switch (conn->phase) {
		case phase_reading_header:

			ret = cherokee_connection_recv (conn);
			if (ret != ret_ok) {
				purge_closed_connection (thd, conn);
			}


			/* Check security
			 */
			ret = cherokee_connection_reading_check (conn);
			if (ret != ret_ok) {
//				conn_set_mode (thd, conn, stat_writing);
				conn_set_mode (thd, conn, socket_writing);
				conn->phase = phase_add_headers;
				continue;
			}

			/* May it already has the full header
			 */
			if (cherokee_connection_eoh (conn)) {
				conn->phase = phase_processing_header;
				
				/* Turn the connection in write mode
				 */
//				conn_set_mode (thd, conn, stat_writing);
				conn_set_mode (thd, conn, socket_writing);
			} else 
				break;

		case phase_processing_header: {
			
			cherokee_virtual_server_t *vserver = NULL;
			cherokee_plugin_table_entry_t *plugin_entry = NULL;
			
			/* Get the request
			 */
			ret = cherokee_connection_get_request (conn);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}
			
			
			/* Get the virtual host
			 */
			if (conn->host->len > 0) {
				vserver = cherokee_table_get (srv->vservers, conn->host->buf);
			}
			if (vserver == NULL) {
				vserver = srv->vserver_default;
			}


			/* Set the logger of the connection
			 */
			conn->logger_ref = vserver->logger;


			/* Build local request string
			 */
			if ((srv->userdir != NULL) && 
			    (srv->userdir_handler != NULL) && 
			    (cherokee_connection_is_userdir (conn)))
			{
				plugin_entry = srv->userdir_handler;
				ret = cherokee_connection_build_local_directory_userdir (conn, srv->userdir);
				if (ret != ret_ok) {
					conn->phase = phase_add_headers;
					continue;
				}
				
				if (conn->redirect->len >= 1) {
					cherokee_connection_send_header (conn, srv->hideversion);
					maybe_purge_closed_connection (thd, conn);
					continue;
				}
			} else {
				ret = cherokee_connection_get_plugin_entry (conn, vserver->plugins, &plugin_entry);
				if (ret != ret_ok) {
					conn->phase = phase_add_headers;
					continue;
				}
				ret = cherokee_connection_build_local_directory (conn, vserver, plugin_entry);
			}
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}
			
			ret = cherokee_connection_create_handler (conn, plugin_entry);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}
			
			
			/* Parse the rest of headers
			 */
			ret = cherokee_connection_parse_header (conn, srv->encoders);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}

			conn->phase = phase_init;
		}

		case phase_init: // {

			/* Server's "Keep-Alive" can be turned "Off"
			 */
			if (srv->keepalive == 0) {
				conn->keep_alive = 0;
			}

			/* Look for the request
			 */
			ret = cherokee_connection_open_request (conn);
			if (ret != ret_eagain) {
				conn->phase = phase_add_headers;
			}

//		} break;
			
		case phase_add_headers:
			ret = cherokee_connection_send_header (conn, srv->hideversion);
				
			if (ret == ret_ok) {				
				if (conn->method == http_head) {
					maybe_purge_closed_connection (thd, conn);
					continue;
				}
				
				if (! http_type_200(conn->error_code)) {
					cherokee_connection_send_response_page (conn, srv->hideversion);

					/* Log the error
					 */
					if (conn->logger_ref != NULL) {
						cherokee_logger_write_error (conn->logger_ref, conn);
					}
					
					purge_closed_connection (thd, conn);
					continue;
				}
			}

			/* Log the connection
			 */
			cherokee_connection_log_or_delay (conn);

/*			if (conn->logger_ref != NULL) {
				cherokee_logger_write_access (conn->logger_ref, conn);
			}
*/
			
			conn->phase = phase_steping;
			
		case phase_steping: {
			ret_t sent_ret;
			
			/* Handler step: read or make new data to send
			 */
			ret = cherokee_connection_step (conn);

			switch (ret) {
			case ret_eof_have_data:
				sent_ret = cherokee_connection_send (conn);

				switch (sent_ret) {
				case ret_eof:
				case ret_error:
					purge_closed_connection (thd, conn);
					continue;
				default:	
					maybe_purge_closed_connection (thd, conn);
					continue;
				}
				break;

			case ret_ok:
				/* We've data, so..
				 */
				sent_ret = cherokee_connection_send (conn);

				if ((sent_ret == ret_eof) ||
				    (sent_ret == ret_error)) 
				{
					purge_closed_connection (thd, conn);					
				}
				break;

			case ret_ok_and_sent:
				break;

			case ret_eof:
				maybe_purge_closed_connection (thd, conn);
				break;
			case ret_error:
				purge_closed_connection (thd, conn);
				break;
			case ret_eagain:
				printf ("eagain en servidor\n");
				break;
			default:
				SHOULDNT_HAPPEN; 
			}
			
		} break;

		default:
 			SHOULDNT_HAPPEN;
		}
	} /* list */	

	return ret_ok;
}


ret_t 
cherokee_thread_free (cherokee_thread_t *thd)
{
	list_t *i, *tmp;

	cherokee_fdpoll_free (thd->fdpoll);
	thd->fdpoll = NULL;

	/* Free the connection
	 */
	list_for_each_safe (i, tmp, (list_t*)&thd->active_list) {
		cherokee_connection_free (CONN(i));
	}

	free (thd);

	return ret_ok;
}


ret_t 
cherokee_thread_step (cherokee_thread_t *thread)
{
	ret_t              ret = ret_ok;
	cherokee_server_t *srv = THREAD_SRV(thread);


	/* Process connections
	 */
	if (thread->active_list_num > 0) 
	{
		ret = process_active_connections (thread);
	}

	return ret;
}


ret_t 
cherokee_thread_get_new_connection (cherokee_thread_t *thd, cherokee_connection_t **conn)
{
	cherokee_connection_t *new_connection;
	cherokee_server_t     *server;

	server = thd->server;

	if (list_empty (&thd->reuse_list)) {
		cherokee_connection_new (&new_connection);
		new_connection->server = server;
	} else {
		new_connection = CONN(thd->reuse_list.next);
		list_del ((list_t *)new_connection);		
		new_connection->timeout = server->bogo_now + server->timeout;
	}

	*conn = new_connection;

	return ret_ok;
}


ret_t 
cherokee_thread_add_connection (cherokee_thread_t *thd, cherokee_connection_t  *conn)
{
	cherokee_fdpoll_add (thd->fdpoll, SOCKET_FD(conn->socket), 0);
	conn_set_mode (thd, conn, socket_reading);

	add_connection (thd, conn);

	return ret_ok;
}



int   
cherokee_thread_connection_num (cherokee_thread_t *thd) 
{
	return thd->active_list_num;
}


ret_t 
cherokee_thread_close_all_connections (cherokee_thread_t *thd)
{
	list_t *i, *tmp;
	list_for_each_safe (i, tmp, (list_t*)&thd->active_list) {
		purge_closed_connection (thd, CONN(i));
	}

	return ret_ok;
}
