/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "mmap.h"


#ifndef MAP_FILE
# define MAP_OPTIONS MAP_SHARED
#else
# define MAP_OPTIONS MAP_SHARED | MAP_FILE
#endif


typedef struct {
	struct list_head base;
	
	dev_t  dev;
	ino_t  ino;

	char  *mmap;

	int    use_count;
	size_t len;
	int    available;
	int    times_used;
} cherokee_mmap_entry_t ;

#define MMAP_ENTRY(e) ((cherokee_mmap_entry_t *)(e))


ret_t 
cherokee_mmap_new  (cherokee_mmap_t **mmap, int num)
{
	int i;
	CHEROKEE_NEW_STRUCT(n,mmap);

	/* Init
	 */
	for (i=0; i<num; i++) {
		INIT_LIST_HEAD((list_t*)n->hash[i]);
	}
	
	/* Return it
	 */
	*mmap = n;
	   
	return ret_ok;
}


ret_t 
cherokee_mmap_free (cherokee_mmap_t *mmap)
{
	free (mmap);
	return ret_ok;
}


static char *
find_hash (struct stat *s)
{
	list_t *i;

	list_for_each (i, (list_t*)MMAP_HASH_FUNC(s)) {
		if ((s->st_dev == MMAP_ENTRY(i)->dev) &&
		    (s->st_ino == MMAP_ENTRY(i)->ino))
		{
			return MMAP_ENTRY(i)->mmap;
		}
	}

	return NULL;
}

static cherokee_mmap_entry_t *
find (int fd, struct stat *s) 
{
	void *area;

	area = find_hash (s);
	if (area != NULL) {
		return NULL; // TODO
	}


	area = mmap (NULL,           /* void   *start  */
		     s->st_size,     /* size_t  length */
		     PROT_READ,      /* int     prot   */
		     MAP_OPTIONS,    /* int     flag   */
		     fd,             /* int     fd     */
		     0);             /* off_t   offset */

	if (area == MAP_FAILED) {
		return NULL;
	}
	
	

	return NULL;
}


void * 
cherokee_mmap_get (cherokee_mmap_t *mmap, const char *filename)
{
	int fd;
	struct stat s;
	cherokee_mmap_entry_t *entry;

	fd = open (filename, O_RDONLY);
	if (fd == -1) {
		return NULL;
	}

	fstat(fd, &s);
	if (S_ISDIR(s.st_mode)) {
		return NULL;
	}

	entry = MMAP_ENTRY (find (fd, &s));
	close(fd);

	return entry;
}
