/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include "fdpoll.h"

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#ifdef HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#else
#include <resource.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_PTHREAD
# define LOCK_WRITER(fdp) pthread_rwlock_wrlock (&fdp->rwlock)
# define LOCK_READER(fdp) pthread_rwlock_rdlock (&fdp->rwlock)
# define UNLOCK(fdp)      pthread_rwlock_unlock (&fdp->rwlock)
#else
# define LOCK_WRITER(fdp)
# define LOCK_READER(fdp)
# define UNLOCK(fdp)
#endif

#define POLL_ERROR  POLLHUP | POLLERR 
#define POLL_READ   POLLIN | POLLPRI | POLL_ERROR
#define POLL_WRITE  POLLOUT | POLL_ERROR


/* Useful documentation:
 *
 * http://www.opengroup.org/onlinepubs/007904975/functions/poll.html
 */


/*
** Copyright (c) 1999,2000 by Jef Poskanzer <jef@acme.com>
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
*/

static inline int
get_fd_limit (void)
{
	   int nfiles;
	   struct rlimit rl;

	   if (getrlimit (RLIMIT_NOFILE, &rl) == 0)
	   {
			 nfiles = rl.rlim_cur;

			 if (rl.rlim_max == RLIM_INFINITY) {
                        /* arbitrary */
				    rl.rlim_cur = 8192;  
			 } else if (rl.rlim_max > rl.rlim_cur) {
				    rl.rlim_cur = rl.rlim_max;
			 }
			 
			 if (setrlimit( RLIMIT_NOFILE, &rl ) == 0) {
				    nfiles = rl.rlim_cur;
			 }

			 return nfiles;
	   }

	   return getdtablesize();
}


ret_t
cherokee_fdpoll_new (cherokee_fdpoll_t **fdp)
{
	   /* Get memory
	    */
	   CHEROKEE_NEW_STRUCT (n, fdpoll);

	   /* Look for max fd limit
	    */
	   n->nfiles = get_fd_limit();
	   n->npollfds = 0;
	 
#ifdef HAVE_PTHREAD
	   pthread_rwlock_init(&n->rwlock, NULL); 
#endif
	   
	   /* Get memory
	    */
	   n->pollfds = (struct pollfd*) malloc (sizeof(struct pollfd) * n->nfiles);
	   return_if_fail (n->pollfds, ret_nomem);
	   
	   n->fdidx = (int*) malloc (sizeof(int) * n->nfiles);
	   return_if_fail (n->fdidx, ret_nomem);


	   /* Return it
	    */
	   *fdp = n;

	   return ret_ok;
}


ret_t
cherokee_fdpoll_free (cherokee_fdpoll_t *fdp)
{
	   free (fdp->pollfds);
	   free (fdp->fdidx);

	   free (fdp);

	   return ret_ok;
}


ret_t
cherokee_fdpoll_add (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	LOCK_WRITER(fdp);

	fdp->pollfds[fdp->npollfds].fd = fd;
	switch (rw)
        {
        case 0:  
//		fdp->pollfds[fdp->npollfds].events = POLLIN; 
		fdp->pollfds[fdp->npollfds].events = POLL_READ; 
		break;
        case 1: 
//		fdp->pollfds[fdp->npollfds].events = POLLOUT;
		fdp->pollfds[fdp->npollfds].events = POLL_WRITE;
		break;
        }
	fdp->fdidx[fd] = fdp->npollfds;
	fdp->npollfds++;

	UNLOCK(fdp);
	
	return ret_ok;
}


void inline
cherokee_fdpoll_set_mode (cherokee_fdpoll_t *fdp, int fd, int rw)
{
//	LOCK_WRITER(fdp);

//	fdp->pollfds[fdp->fdidx[fd]].events = rw ? POLLOUT : POLLIN;
	fdp->pollfds[fdp->fdidx[fd]].events = rw ? POLL_WRITE : POLL_READ;

//	UNLOCK(fdp);
}



ret_t
cherokee_fdpoll_del (cherokee_fdpoll_t *fdp, int fd)
{
	int idx;
	ret_t ret = ret_ok;

	LOCK_WRITER(fdp);
	   
	idx = fdp->fdidx[fd];
	if (idx < 0 || idx >= fdp->nfiles) {
		ret = ret_error;
		goto out;
	}

	fdp->npollfds--;
	fdp->pollfds[idx] = fdp->pollfds[fdp->npollfds];
	fdp->fdidx[fdp->pollfds[idx].fd] = idx;
	
out:
	UNLOCK(fdp);
	return ret;
}


int
cherokee_fdpoll_check (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	int ret;

//	LOCK_READER(fdp);

	switch (rw) {
        case 0: 
		ret = fdp->pollfds[fdp->fdidx[fd]].revents & (POLLIN | POLLERR | POLLHUP | POLLNVAL);
		break;
        case 1: 
		ret = fdp->pollfds[fdp->fdidx[fd]].revents & (POLLOUT | POLLERR | POLLHUP | POLLNVAL);
		break;
        default: 
		SHOULDNT_HAPPEN;
	}


//	UNLOCK(fdp);
	return ret;
}


inline int
cherokee_fdpoll_watch (cherokee_fdpoll_t *fdp, int timeout_msecs)
{
	int ret;

//	LOCK_WRITER(fdp);

	ret = poll (fdp->pollfds, fdp->npollfds, timeout_msecs);

//	UNLOCK(fdp);
	return ret;
}


