/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>

#include "common.h"
#include "server.h"
#include "connection.h"
#include "log.h"


static void purge_closed_connection       (cherokee_server_t *srv, cherokee_connection_t *conn);
static void maybe_purge_closed_connection (cherokee_server_t *srv, cherokee_connection_t *conn);



ret_t
cherokee_server_new  (cherokee_server_t **srv)
{
	cherokee_server_t *n;

	
	/* Get memory
	 */
	n = (cherokee_server_t*) malloc (sizeof(cherokee_server_t));
	return_if_fail (n!=NULL, ret_nomem);
	
	/* Server socket
	 */
	n->socket = -1;
	
	/* Lists
	 */
	INIT_LIST_HEAD(&n->active_list);
	INIT_LIST_HEAD(&n->reuse_list);
 
	/* File descriptors poll
	 */
	cherokee_fdpoll_new(&n->fdpoll);
	
	/* Server config
	 */
	n->port           = 80;
	n->listen_to      = NULL;
	n->fdwatch_msecs  = 999;
	n->log            = 0;
	n->timeout        = 15;
	n->keepalive      = 1;
	n->hideversion    = 0;
	n->mimetypes_file = NULL;
	n->userdir        = NULL;
	n->userdir_handler= NULL;

	n->user_orig      = getuid();
	n->user           = n->user_orig;
	n->group_orig     = getgid();
	n->group          = n->group_orig;

 
	/* Virtual servers table
	 */
	cherokee_table_new (&n->vservers);
	return_if_fail (n->vservers!=NULL, ret_nomem);

	cherokee_virtual_server_new (&n->vserver_default);
	return_if_fail (n->vserver_default!=NULL, ret_nomem);

		
	/* Encoders 
	 */
	cherokee_encoder_table_new (&n->encoders);
	return_if_fail (n->encoders != NULL, ret_nomem);
	

	/* Return the object
	 */
	*srv = n;

	return ret_ok;
}


static inline ret_t
change_execution_user (cherokee_server_t *srv)
{
	int error;

	/* Change of group requested
	 */
	if (srv->group != srv->group_orig) {
		error = setgid (srv->group);
		if (error != 0) {
			PRINT_ERROR ("Can't change group to GID %d, running with GID=%d\n",
				     srv->group, srv->group_orig);
		}
	}

	/* Change of user requested
	 */
	if (srv->user != srv->user_orig) {
		error = setuid (srv->user);		
		if (error != 0) {
			PRINT_ERROR ("Can't change user to UID %d, running with UID=%d\n",
				     srv->user, srv->user_orig);
		}
	}

	return ret_ok;
}


void  
cherokee_server_set_min_latency (cherokee_server_t *srv, int msecs)
{
	srv->fdwatch_msecs = msecs;
}


static ret_t
initialize_server_socket (cherokee_server_t *srv)
{
	ret_t ret;
	int   on = 1;
	int                 flags;
	struct sockaddr_in  sockaddr;


	/* Create the socket
	 */
	srv->socket = socket (AF_INET, SOCK_STREAM, 0);
	return_if_fail (srv->socket > 0, ret_error);


	/* Add the socket to fdpoll
	 */
	cherokee_fdpoll_add(srv->fdpoll, srv->socket, 0);


	/* Set 'close-on-exec'
	 */
	fcntl(srv->socket, F_SETFD, FD_CLOEXEC);
	
	
	/* To re-bind without wait to TIME_WAIT
	 */
	ret = setsockopt (srv->socket, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
	return_if_fail (ret == 0, ret_error);


	/* Bind the socket
	 */
	sockaddr.sin_port        = htons (srv->port);
	sockaddr.sin_family      = AF_INET;

	if (srv->listen_to != NULL) {
#ifdef HAVE_INET_PTON
		inet_pton(sockaddr.sin_family, srv->listen_to, &sockaddr.sin_addr);
#else
		sockaddr.sin_addr.s_addr = inet_addr(srv->listen_to);
#endif
	} else {
		sockaddr.sin_addr.s_addr = INADDR_ANY;
	}

	ret = bind (srv->socket, (struct sockaddr *)&sockaddr, sizeof(sockaddr));
	if (ret != 0) {
		uid_t uid = getuid();
		gid_t gid = getgid();

		PRINT_ERROR ("Can't bind() socket (port=%d, UID=%d, GID=%d)\n", srv->port, uid, gid);
		return ret_error;
	}

	cherokee_log (LOG_INFO, "Listening on port %d", srv->port);

	   
	/* Set no-delay mode
	 */
	flags = fcntl (srv->socket, F_GETFL, 0);
	return_if_fail (flags != -1, ret_error);
	
	ret = fcntl (srv->socket, F_SETFL, flags | O_NDELAY);
	return_if_fail (ret >= 0, ret_error);
	
	
	/* Listen
	 */
	ret = listen (srv->socket, 1024);
	if (ret < 0) {
		close (srv->socket);
		return_if_fail (0, ret_error);
	}

	return ret_ok;
}


ret_t
cherokee_server_init (cherokee_server_t *srv) 
{   
	ret_t ret;


	/* Init the log system
	 */
	if (srv->log) {
		cherokee_server_log (srv, 1);
	}


	/* If the server has a previous server socket opened, Eg:
	 * because a SIGHUP, it shouldn't init the server socket.
	 */
	if (srv->socket == -1) {
		ret = initialize_server_socket (srv);
		if (ret != ret_ok) {
			return ret;
		}
	}


	/* Change the user
	 */
	ret = change_execution_user (srv);
	if (ret != ret_ok) {
		return ret;
	}

	
	return ret_ok;
}


static inline void
close_all_connections (cherokee_server_t *srv)
{
	list_t *i, *tmp;

	/* Close connectios and move to reusable list
	 */
	list_for_each_safe (i, tmp, (list_t*)&srv->active_list) {
		purge_closed_connection (srv, CONN(i));
	}
}


ret_t 
cherokee_server_clean (cherokee_server_t *srv)
{
	/* Close all active connections
	 */
	close_all_connections (srv);


	/* Clean the configuration
	 */
	if (srv->listen_to != NULL) {
		free (srv->listen_to);
		srv->listen_to = NULL;
	}

	if (srv->mimetypes_file != NULL) {
		free (srv->mimetypes_file);
		srv->mimetypes_file = NULL;
	}

	if (srv->userdir != NULL) {
		free (srv->userdir);
		srv->userdir = NULL;
	}
	if (srv->userdir_handler != NULL) {
		cherokee_plugin_table_entry_free (srv->userdir_handler);
		srv->userdir_handler = NULL;
	}

	
	/* Clean the encoders table
	 */
	cherokee_encoder_table_clean (srv->encoders);

	/* Clean the virtual servers table
	 */
	cherokee_virtual_server_clean (srv->vserver_default);
 
	// TODO:
	// Free each vserver
	cherokee_table_free (srv->vservers);
	cherokee_table_new (&srv->vservers);

	srv->hideversion = 0;

	return ret_ok;
}


ret_t
cherokee_server_free (cherokee_server_t *srv)
{
	close (srv->socket);

	cherokee_fdpoll_free (srv->fdpoll);
	srv->fdpoll = NULL;

	cherokee_virtual_server_free (srv->vserver_default);
	srv->vserver_default = NULL;

	cherokee_encoder_table_free (srv->encoders);

	// TODO:
	// Free each vserver
	cherokee_table_free (srv->vservers);
	cherokee_table_new (&srv->vservers);


	if (srv->listen_to != NULL) {
		free (srv->listen_to);
		srv->listen_to = NULL;
	}

	if (srv->mimetypes_file != NULL) {
		free (srv->mimetypes_file);
		srv->mimetypes_file = NULL;
	}

	if (srv->userdir != NULL) {
		free (srv->userdir);
		srv->userdir = NULL;
	}
	if (srv->userdir_handler != NULL) {
		cherokee_plugin_table_entry_free (srv->userdir_handler);
		srv->userdir_handler = NULL;
	}
	
	free (srv);
	
	return ret_ok;
}


ret_t 
cherokee_server_log (cherokee_server_t *srv, int active)
{
	/* Shutdown loggin
	 */
	if (srv->log && !active) {
		srv->log = active;
		cherokee_log (LOG_NOTICE, "Closing log");
		return cherokee_log_close();
	}
	
	/* Active logging
	 */
	if (active && !srv->log) {
		ret_t ret;
		srv->log = active;
		ret = cherokee_log_init();
		cherokee_log (LOG_NOTICE, "Starting log");
		return ret;
	}

	return ret_error;
}


static inline int
look_for_new_connection (cherokee_server_t *srv)
{
        /* Look for new connections
	 */
	if (cherokee_fdpoll_check (srv->fdpoll, srv->socket, 0))
	{
		int tmp = 1;
		int new_socket;
		cherokee_connection_t *new_connection;
		

		/* Look for a reusable connection
		 */
		if (list_empty (&srv->reuse_list)) {
			cherokee_connection_new (&new_connection);
			new_connection->server = srv;
		} else {
			new_connection = CONN(srv->reuse_list.next);
			list_del ((list_t *)new_connection);

			new_connection->timeout = srv->bogo_now + srv->timeout;
		}

		
		/* Get the new socket connection
		 */
		if (srv->log) {
			socklen_t len; 
			len = sizeof(struct sockaddr_in);
			new_socket = accept (srv->socket, (struct sockaddr *) &new_connection->addr_in, &len);
		} else {
			new_socket = accept (srv->socket, NULL, NULL);
		}
		cherokee_fdpoll_add (srv->fdpoll, new_socket, 0);


		/*
		 *  EXPERIMENTAL
		 *
		 */

		fcntl (new_socket, F_SETFD, 1);  /* close-on-exec */
		
		/* Disable Nagle's algorithm for this connection. 
		 * Written data to the network is not buffered pending 
		 * acknowledgement of previously written data.
		 */
		setsockopt (new_socket, IPPROTO_TCP, TCP_NODELAY, &tmp, sizeof(tmp));

		/*
		 *  EXPERIMENTAL
		 *
		 */

		
		/* Set the socket
		 */
		cherokee_connection_set_socket (new_connection, new_socket);
		
		
		/* Add to active connection list
		 */
		list_add ((list_t *)new_connection, &srv->active_list);

		return 1;
	}
	
	return 0;
}


static inline void
conn_set_mode (cherokee_server_t *srv, cherokee_connection_t *conn, cherokee_connection_status_t rw)
{
	conn->status = rw;
	cherokee_fdpoll_set_mode (srv->fdpoll, conn->socket, rw);
}


static void
purge_closed_connection (cherokee_server_t *srv, cherokee_connection_t *conn)
{
	/* Delete from file descriptors poll
	 */
	cherokee_fdpoll_del (srv->fdpoll, conn->socket);
	
	/* Close
	 */
	conn->status = stat_closed;
	cherokee_connection_close (conn);
	
	/* Delete from active connections list
	 */
	list_del ((list_t *)conn);

	/* Add to reusable connections list
	 */
	cherokee_connection_mrproper (conn);
	list_add ((list_t *)conn, &srv->reuse_list);
}


static void
maybe_purge_closed_connection (cherokee_server_t *srv, cherokee_connection_t *conn)
{
	/* Maybe it's a keep-alive connection
	 */
	if (conn->keep_alive != 0) {
		cherokee_connection_clean (conn);
		conn_set_mode (srv, conn, stat_reading);
		conn->timeout = srv->bogo_now + srv->timeout;
		return;
	}

	purge_closed_connection (srv, conn);
}


static inline void
process_active_connections (cherokee_server_t *srv)
{
	ret_t  ret;
	cherokee_connection_t *conn;
	list_t *i, *tmp;


	/* Process active connections
	 */
	list_for_each_safe (i, tmp, (list_t*)&srv->active_list) {
		conn = CONN(i);			 											   

		/* May the connection was too much time w/o any work
		 */
		if (conn->timeout < srv->bogo_now) {
			purge_closed_connection (srv, conn);
			continue;
		}

		/* Inspect the descriptor
		 */
		if (cherokee_fdpoll_check (srv->fdpoll, conn->socket, conn->status) == 0) {
			continue;
		}

		/* The connection has work, so..
		 */
		conn->timeout = srv->bogo_now + srv->timeout;

		/* Phases
		 */
		switch (conn->phase) {
		case phase_reading_header:

			ret = cherokee_connection_recv (conn);
			if (ret != ret_ok) {
				purge_closed_connection (srv, conn);
			}


			/* Check security
			 */
			ret = cherokee_connection_reading_check (conn);
			if (ret != ret_ok) {
				conn_set_mode (srv, conn, stat_writing);
				conn->phase = phase_add_headers;
				continue;
			}

			/* May it already has the full header
			 */
			if (cherokee_connection_eoh (conn)) {
				conn->phase = phase_processing_header;
				
				/* Turn the connection in write mode
				 */
				conn_set_mode (srv, conn, stat_writing);
			} else 
				break;

		case phase_processing_header: {
			
			cherokee_virtual_server_t *vserver = NULL;
			cherokee_plugin_table_entry_t *plugin_entry = NULL;
			
			/* Get the request
			 */
			ret = cherokee_connection_get_request (conn);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}
			
			/* Maybe log it
			 */
			if (srv->log) {
				cherokee_connection_log (conn, srv->bogo_now);
			}

			
			/* Get the virtual host
			 */
			if (conn->host->len > 0) {
				vserver = cherokee_table_get (srv->vservers, conn->host->buf);
			}
			if (vserver == NULL) {
				vserver = srv->vserver_default;
			}
			
			/* Build local request string
			*/
			if ((srv->userdir != NULL) && 
			    (srv->userdir_handler != NULL) && 
			    (cherokee_connection_is_userdir (conn)))
			{
				ret = cherokee_connection_build_local_directory_userdir (conn, srv->userdir);
				plugin_entry = srv->userdir_handler;
				
			} else {
				ret = cherokee_connection_build_local_directory (conn, vserver);
				if (ret != ret_ok) {
					conn->phase = phase_add_headers;
					continue;
				}
				ret = cherokee_connection_get_plugin_entry (conn, vserver->plugins, &plugin_entry);
			}
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}
			
			ret = cherokee_connection_create_handler (conn, plugin_entry);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}
			
			
			/* Parse the rest of headers
			 */
			ret = cherokee_connection_parse_header (conn);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}

			conn->phase = phase_init;
		}

		case phase_init: {

			/* Server's "Keep-Alive" can be turned "Off"
			 */
			if (srv->keepalive == 0) {
				conn->keep_alive = 0;
			}

			/* Look for the request
			 */
			ret = cherokee_connection_open_request (conn);
			if (ret != ret_eagain) {
				conn->phase = phase_add_headers;
			}

		} break;
			
		case phase_add_headers:
			ret = cherokee_connection_send_header (conn, srv->hideversion);
				
			if (ret == ret_ok) {				
				if (conn->method == http_head) {
					maybe_purge_closed_connection (srv, conn);
					continue;
				}
				
				if (! http_type_200(conn->error_code)) {
					cherokee_connection_send_response_page (conn, srv->hideversion);
					purge_closed_connection (srv, conn);
					continue;
				}
			}

			conn->phase = phase_steping;
			
		case phase_steping: {
			ret_t sent_ret;
			
			/* Handler step: read or make new data to send
			 */
			ret = cherokee_connection_step (conn);
			
			switch (ret) {
			case ret_eof_have_data:
				sent_ret = cherokee_connection_send (conn);

				switch (sent_ret) {
				case ret_error:
					purge_closed_connection (srv, conn);
					continue;
				default:	
					maybe_purge_closed_connection (srv, conn);
					continue;
				}
				break;

			case ret_ok:
				/* We've data, so..
				 */
				sent_ret = cherokee_connection_send (conn);

				if ((sent_ret == ret_eof) ||
				    (sent_ret == ret_error)) 
				{
					purge_closed_connection (srv, conn);					
				}
				break;

			case ret_eof:
				maybe_purge_closed_connection (srv, conn);
				break;
			case ret_error:
				purge_closed_connection (srv, conn);
				break;
			case ret_eagain:
				printf ("eagain en servidor\n");
				break;
			default:
				SHOULDNT_HAPPEN; 
			}
			
		} break;

		default:
 			SHOULDNT_HAPPEN;
		}
	} /* list */
}


void
cherokee_server_step (cherokee_server_t *srv)
{
	int ret, i=MAX_NEW_CONNECTIONS_PER_STEP;
	
	do {
		cherokee_fdpoll_watch (srv->fdpoll, srv->fdwatch_msecs);

		/* Look for new connections
		 */	   
		ret = look_for_new_connection(srv);

	} while (ret && (i-- > 0));

	/* Get the time
	 */
	srv->bogo_now = time (NULL);

	/* Active connections
	 */
	process_active_connections(srv);
}


ret_t 
cherokee_server_read_config_file (cherokee_server_t *srv, char *filename)
{
	int error;

	extern FILE *yyin;
	extern int yyparse (void *);

	/* Set the file to read
	 */
	yyin = fopen (filename, "r");
	return_if_fail (yyin!=NULL, ret_file_not_found);

	/* Cooooome on :-)
	 */
	error = yyparse ((void *)srv);

	if (yyin != NULL)
		fclose (yyin);

	if (error != 0)
		return ret_error;

	return ret_ok;
}


ret_t 
cherokee_server_read_config_string (cherokee_server_t *srv, char *config_string)
{
	int   error;
	void *bufstate;

	extern int  yy_scan_string (void *);
	extern int  yyparse (void *);
	extern void yy_switch_to_buffer (void *);
	extern void yy_delete_buffer (void *);

	bufstate = (void *) yy_scan_string (config_string);
	yy_switch_to_buffer(bufstate);

	error = yyparse((void *)srv);

	yy_delete_buffer (bufstate);

	if (error != 0)
		return ret_error;

	return ret_ok;
}
