/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef __CHEROKEE_CONNECTION_H__
#define __CHEROKEE_CONNECTION_H__

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#include <sys/types.h>	
#include <sys/socket.h>
#include <arpa/inet.h>

#include "list.h"
#include "http.h"
#include "common.h"
#include "buffer.h"
#include "handler.h"
#include "handler_file.h"
#include "plugin_table.h"
#include "virtual_server.h"
#include "encoder.h"


typedef enum {
	stat_reading = 0,
	stat_writing = 1,
	stat_closed
} cherokee_connection_status_t;

typedef enum {
	phase_reading_header,
	phase_processing_header,
	phase_init,
	phase_add_headers,
	phase_steping,
} cherokee_connection_phase_t;


typedef struct {
	struct list_head              list;
	
	void                         *server;

	int                           socket;
	struct sockaddr_in            addr_in;
	
	cherokee_connection_status_t  status;
	cherokee_connection_phase_t   phase;


	cherokee_http_version_t       version;
	cherokee_http_method_t        method;

	cherokee_encoder_t           *encoder;

	cherokee_http_t               error_code;
	
	cherokee_buffer_t            *buffer;
	cherokee_handler_t           *handler;

	/* Eg:
	 * http://www.alobbs.com/cherokee/dir/file
	 */
	cherokee_buffer_t            *local_directory;  /* /var/www/  or  /home/alo/public_html/ */
	cherokee_buffer_t            *web_directory;    /* /cherokee/ */
	cherokee_buffer_t            *request;          /* /dir/file */
	cherokee_buffer_t            *host;

	time_t                        timeout;
	int                           keep_alive;

	int                           range_start;
	int                           range_end;
} cherokee_connection_t;


#define CONN(c)     ((cherokee_connection_t *)(c))
#define CONN_SRV(c) ((cherokee_server_t *) CONN(c)->server)


ret_t cherokee_connection_new        (cherokee_connection_t **cnt);
ret_t cherokee_connection_free       (cherokee_connection_t  *cnt);
ret_t cherokee_connection_clean      (cherokee_connection_t  *cnt);
ret_t cherokee_connection_mrproper   (cherokee_connection_t  *cnt);
ret_t cherokee_connection_set_socket (cherokee_connection_t  *cnt, int socket);
void  cherokee_connection_log        (cherokee_connection_t  *cnt, time_t bogo_now);

/* 	Send & Recv
 */
ret_t cherokee_connection_recv  (cherokee_connection_t  *cnt);
ret_t cherokee_connection_send  (cherokee_connection_t  *cnt);
ret_t cherokee_connection_close (cherokee_connection_t  *cnt);

/* 	Sending
 */
ret_t cherokee_connection_send_header        (cherokee_connection_t *cnt, int hideversion);
ret_t cherokee_connection_send_response_page (cherokee_connection_t *cnt, int hideversion);

/*	Iteration
 */
ret_t cherokee_connection_open_request  (cherokee_connection_t  *cnt);
ret_t cherokee_connection_reading_check (cherokee_connection_t  *cnt);
ret_t cherokee_connection_step          (cherokee_connection_t  *cnt);

ret_t cherokee_connection_get_plugin_entry (cherokee_connection_t  *cnt, cherokee_plugin_table_t *plugins, cherokee_plugin_table_entry_t **plugin_entry);
ret_t cherokee_connection_create_handler   (cherokee_connection_t  *cnt, cherokee_plugin_table_entry_t *plugin_entry);

/* 	Headers
 */
int   cherokee_connection_eoh          (cherokee_connection_t  *cnt);
ret_t cherokee_connection_get_request  (cherokee_connection_t  *cnt);
ret_t cherokee_connection_parse_header (cherokee_connection_t  *cnt);

int   cherokee_connection_is_userdir                    (cherokee_connection_t  *cnt);
ret_t cherokee_connection_build_local_directory         (cherokee_connection_t  *cnt, cherokee_virtual_server_t *vsrv);
ret_t cherokee_connection_build_local_directory_userdir (cherokee_connection_t  *cnt, char *userdir);


#endif /* __CHEROKEE_CONNECTION_H__  */
