/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <pwd.h>

#include "handler.h"
#include "handler_dirlist.h"
#include "handler_file.h"
#include "connection.h"
#include "buffer.h"
#include "plugin_table_entry.h"


ret_t
cherokee_connection_new  (cherokee_connection_t **cnt)
{
	   cherokee_connection_t *n;
	   n = (cherokee_connection_t *)malloc(sizeof(cherokee_connection_t));

	   cherokee_buffer_new (&n->buffer);
	   
	   INIT_LIST_HEAD((list_t*)n);
	   n->socket        = -1;
	   n->error_code    = http_ok;
	   n->status        = stat_reading;
	   n->phase         = phase_reading_header;
	   n->method        = http_unknown;
	   n->version       = http_version_unknown;
	   n->handler       = NULL; 
	   n->request       = NULL;
	   n->request_len   = 0;
	   n->request_local = NULL;
	   n->host          = NULL;
	   n->web_directory = NULL;
	   n->keep_alive    = 0;
	   n->timeout       = time(NULL) + 15;
	   n->range_start   = 0;
	   n->range_end     = 0;

	   *cnt = n;
	   return ret_ok;
}


ret_t
cherokee_connection_free (cherokee_connection_t  *cnt)
{
	   cherokee_buffer_free (cnt->buffer);
 
	   if (cnt->request) {
		   free (cnt->request);
		   cnt->request = NULL;
	   }
 
	   if (cnt->request_local) {
		   free (cnt->request_local);
		   cnt->request_local = NULL;
	   }
	   
	   if (cnt->host) {
		   free (cnt->host);
		   cnt->host = NULL;
	   }

	   if (cnt->web_directory) {
		   free (cnt->web_directory);
		   cnt->web_directory = NULL;
	   }

	   free (cnt);
	   return ret_ok;
}


ret_t
cherokee_connection_clean (cherokee_connection_t *cnt)
{
	   cherokee_buffer_clean (cnt->buffer);
	   
	   cnt->status      = stat_reading;
	   cnt->phase       = phase_reading_header;
	   cnt->version     = http_version_unknown;
	   cnt->method      = http_unknown;
	   cnt->error_code  = http_ok;
	   cnt->timeout     = -1;
	   cnt->range_start = 0;
	   cnt->range_end   = 0;

	   if (cnt->request) {
		   free (cnt->request);
		   cnt->request = NULL;
	   }
	   cnt->request_len = 0;

	   if (cnt->request_local) {
		   free (cnt->request_local);
		   cnt->request_local = NULL;
	   }

	   if (cnt->handler) {
		   cherokee_handler_free (cnt->handler);
		   cnt->handler = NULL;
	   }

	   if (cnt->host) {
		   free (cnt->host);
		   cnt->host = NULL;
	   }

	   if (cnt->web_directory) {
		   free (cnt->web_directory);
		   cnt->web_directory = NULL;
	   }

	   return ret_ok;
}


ret_t 
cherokee_connection_mrproper (cherokee_connection_t *cnt)
{
	cnt->socket     = -1;
	cnt->keep_alive = 0;

	return cherokee_connection_clean (cnt);
}


static inline void
add_error_code_string_to_buffer (cherokee_connection_t *cnt)
{
	switch(cnt->error_code) {
	case http_ok:
		cherokee_buffer_add (cnt->buffer, http_ok_string, 6); break;
	case http_accepted:
		cherokee_buffer_add (cnt->buffer, http_accepted_string CRLF, 12); break;
	case http_partial_content:
		cherokee_buffer_add (cnt->buffer, http_partial_content_string CRLF, 19); break;		
	case http_bad_request:
		cherokee_buffer_add (cnt->buffer, http_bad_request_string CRLF, 15); break;
	case http_access_denied:
		cherokee_buffer_add (cnt->buffer, http_access_denied_string CRLF, 13); break;
	case http_not_found:
		cherokee_buffer_add (cnt->buffer, http_not_found_string CRLF, 13); break;
	case http_internal_error:
		cherokee_buffer_add (cnt->buffer, http_internal_error_string CRLF, 25); break;
	case http_moved_permanently: 
		cherokee_buffer_add (cnt->buffer, http_moved_permanently_string CRLF, 21); break;
	default:
		SHOULDNT_HAPPEN;
	}
}


ret_t
cherokee_connection_send_header (cherokee_connection_t  *cnt)
{
	int ret;
	
	cherokee_buffer_make_empty (cnt->buffer);
	
	/* Add protocol string + error_code
	 */
	switch (cnt->version) {
	case http_version_09:
		cherokee_buffer_add (cnt->buffer, "HTTP/0.9 ", 9); 
		break;
	case http_version_10:
		cherokee_buffer_add (cnt->buffer, "HTTP/1.0 ", 9); 
		break;
	case http_version_11:
		cherokee_buffer_add (cnt->buffer, "HTTP/1.1 ", 9); 
		break;
	default:
		break;
	}
	
	add_error_code_string_to_buffer (cnt);
	cherokee_buffer_add (cnt->buffer, CRLF, 2); 

	/* Add server name
	 */
	cherokee_buffer_add (cnt->buffer, "Server: Cherokee/" VERSION CRLF, 17+5+2);
	
	/* Maybe the handler add some headers
	 */
	if (cnt->handler) {
		cherokee_handler_add_headers (cnt->handler, cnt->buffer);
	}
	
	/* Add the "Connection:" header
	 */
	if (cnt->handler && cnt->keep_alive) {
		cherokee_buffer_add (cnt->buffer, "Connection: Keep-Alive"CRLF
				                  "Keep-Alive: timeout=15"CRLF, 48);
	} else {
		cherokee_buffer_add (cnt->buffer, "Connection: close"CRLF, 19);
	}
	
	/* Add the response header ends
	 */
	cherokee_buffer_add (cnt->buffer, CRLF, 2);
	
	/* Send it
	 */
	ret = write (cnt->socket, cnt->buffer->buf, cnt->buffer->len);
	cherokee_buffer_make_empty  (cnt->buffer);
	
	if (ret > 0)
		return ret_ok;
	else
		return ret_error;
}


ret_t
cherokee_connection_recv (cherokee_connection_t  *cnt)
{
	int  readed;
	char tmp[DEFAULT_RECV_SIZE];
	
	readed = read (cnt->socket, tmp, DEFAULT_RECV_SIZE);

	if (readed > 0) {
		cherokee_buffer_add (cnt->buffer, tmp, readed);
		return ret_ok;
	}

	/* Remote machine cancel the download
	 */
	return ret_eof;
}


int
cherokee_connection_eoh (cherokee_connection_t *cnt)
{
	if (cherokee_buffer_is_empty(cnt->buffer))
		return 0;
	
	if (cnt->buffer->len < 4)
		return 0;
	
	return ((cnt->buffer->buf[cnt->buffer->len-1] == '\n') &&
		(cnt->buffer->buf[cnt->buffer->len-2] == '\r') &&
		(cnt->buffer->buf[cnt->buffer->len-3] == '\n') &&
		(cnt->buffer->buf[cnt->buffer->len-4] == '\r'));
}


ret_t 
cherokee_connection_reading_check (cherokee_connection_t *cnt)
{
	/* Check for too long headers
	 */
	if (cnt->buffer->len > MAX_HEADER_LEN) {
		cnt->error_code = http_bad_request;
		return ret_error;
	}

	return ret_ok;
}


ret_t
cherokee_connection_set_socket (cherokee_connection_t *cnt, int socket)
{
	   cnt->socket = socket;
	   return ret_ok;
}


ret_t
cherokee_connection_send (cherokee_connection_t *cnt)
{
	int sent;
	
	sent = write (cnt->socket, cnt->buffer->buf, cnt->buffer->len);
	
	if (sent > 0) {
		if (sent == cnt->buffer->len) {
			cherokee_buffer_make_empty (cnt->buffer);
		} else {
			cherokee_buffer_move_to_begin (cnt->buffer, sent);
		}
		
	} else if (sent == 0) {
		return ret_eof;
		
	} else {
		switch (errno) {
		case EAGAIN:
			return ret_eagain;
		default:
			/* Remote machine cancel the download
			 */
			return ret_error;
		}
	}
	
	return ret_ok;
}


ret_t
cherokee_connection_close (cherokee_connection_t *cnt)
{
	if (close (cnt->socket) == 0)
		return ret_ok;
	else
		return ret_error;
}


/* From thttpd: Jef Poskanzer <jef@acme.com>
 *
 * Copies and decodes a string.  It's ok for from and to to be the
 * same string.
 */
static void
strdecode (char* to, char* from)
{
	for (; *from != '\0'; ++to, ++from) {
		if (from[0] == '%' && isxdigit( from[1] ) && isxdigit( from[2] )) {
			*to = hexit( from[1] ) * 16 + hexit( from[2] );
			from += 2;
		} else {
			*to = *from;
		}
	}
	*to = '\0';
}


ret_t
cherokee_connection_step (cherokee_connection_t *cnt)
{
	return_if_fail (cnt->handler != NULL, ret_error);

	/* Need to 'read' from handler ?
	 */
	if (cherokee_buffer_is_empty (cnt->buffer)) 
	{
		return cherokee_handler_step (cnt->handler, cnt->buffer);
	}
	
	return ret_ok;
}


static inline ret_t
get_host (cherokee_connection_t *cnt, 
	  char                  *ptr) /* Header at the "Host:" position */
{
	char *end;
	int   size;

	end = strstr (ptr, "\r");
	if (end == NULL) {
		return ret_error;
	}
	
	size = (end - ptr);

	cnt->host = (char *) malloc (size);
	return_if_fail (cnt->host!=NULL, ret_nomem);
	cnt->host[size] = '\0';
	
	memcpy (cnt->host, ptr, size);

	return ret_ok;
}


static inline ret_t
get_uri (cherokee_connection_t *cnt)
{
	int begin, end;
	char *ptr;
	
	/* Malformed header
	 */
	if (cnt->buffer->len < 14)
		return ret_error;
	
	
	/* HTTP method
	 */
	if (strncmp (cnt->buffer->buf, "GET ", 4) == 0) {
		cnt->method = http_get;
		begin = 4;

	} else if (strncmp (cnt->buffer->buf, "POST ", 5) == 0) {
		cnt->method = http_post;
		begin = 5;

	} else if (strncmp (cnt->buffer->buf, "HEAD ", 5) == 0) {
		cnt->method = http_head;
		begin = 5;
		
	} else {
		return ret_error;
	}
	
	
	/* End the request of line
	 */
	end = strstr (cnt->buffer->buf, CRLF) - cnt->buffer->buf;
	
	
	/* HTTP version: HTTP/x.y
	 */
	end -= 9;
	
	switch (cnt->buffer->buf[end+8]) {
	case '9':
		cnt->version = http_version_09; break;
	case '0':
		cnt->version = http_version_10; break;
	case '1':
		cnt->version = http_version_11; break;
	default:
		return ret_error;
	}
	
	
	/* Dup the request
	 */
	cnt->request_len = end - begin;
	
	cnt->request = (char *) malloc(cnt->request_len + 1);
	memcpy (cnt->request, cnt->buffer->buf + begin, cnt->request_len);
	cnt->request[cnt->request_len] = '\0';


	/* Decode %HEX URL encoding
	 */
	strdecode (cnt->request, cnt->request);


	/* Look for "Host:"
	 */
	if ((ptr = strstr (cnt->buffer->buf, "Host: "))) {
		ptr += 6;
		get_host (cnt, ptr);
	}

	
	return ret_ok;
}


int inline
cherokee_connection_is_userdir (cherokee_connection_t *cnt)
{
	return ((cnt->request_len > 4) && (cnt->request[1] == '~'));
}


ret_t 
cherokee_connection_build_local_request (cherokee_connection_t *cnt, cherokee_virtual_server_t *vsrv)
{
	cnt->request_local = (char *) malloc(cnt->request_len + vsrv->root_len + 1);
	memcpy (cnt->request_local, vsrv->root, vsrv->root_len);
	memcpy (cnt->request_local + vsrv->root_len, cnt->request, cnt->request_len);
	cnt->request_local[cnt->request_len + vsrv->root_len] = '\0';

	return ret_ok;
}


ret_t 
cherokee_connection_build_userdir_request (cherokee_connection_t *cnt, cherokee_virtual_server_t *vsrv, char *userdir)
{
	struct passwd *pwd;	
	char          *end;
	char          *begin;

	begin = &cnt->request[2];

	end = index (begin, '/');
	if (end == NULL) return ret_error;

	/* Copy the username
	 */
	memcpy (gbl_buffer, begin, end-begin);
	gbl_buffer[end-begin] = '\0';
	
	/* Get the user home directory
	 */
	pwd = (struct passwd *) getpwnam (gbl_buffer);
	if ((pwd == NULL) || (pwd->pw_dir == NULL)) return ret_error;
	
	/* Build the new request
	 */
	cnt->request_local = (char *) malloc (strlen(pwd->pw_dir) + strlen (userdir) + strlen(end) + 2);
	sprintf (cnt->request_local, "%s/%s%s", pwd->pw_dir, userdir, end);

	return ret_ok;
}


static inline ret_t
get_range (cherokee_connection_t *cnt, 
	   char                  *ptr) /* Header at the "Range: bytes=" position */
{
	int num_len = 0;

	/* Read the start position
	 */
	while ((ptr[num_len] != '-') && (ptr[num_len] != '\0') && (num_len < gbl_buffer_size-1)) {
		gbl_buffer[num_len] = ptr[num_len];
		num_len++;
	}
	gbl_buffer[num_len] = '\0';
	cnt->range_start = atoi (gbl_buffer);
	
	/* Advance the pointer
	 */
	ptr += num_len;
	if (*ptr == '-') {
		ptr++;
	} else {
		return ret_error;
	}

	/* Maybe there're an ending position
	 */
	if ((*ptr != '\0') && (*ptr != '\r') && (*ptr != '\n')) {
		num_len = 0;
		
		/* Read the end
		 */
		while ((ptr[num_len] != '-') && (ptr[num_len] != '\0') && (num_len < gbl_buffer_size-1)) {
			gbl_buffer[num_len] = ptr[num_len];
			num_len++;
		}
		gbl_buffer[num_len] = '\0';
		cnt->range_end = atoi (gbl_buffer);
	}

	return ret_ok;
}


ret_t 
cherokee_connection_get_request (cherokee_connection_t *cnt)
{
        /* Get the request     
	 */
	get_uri (cnt);	   
	if (cnt->request == NULL) {
		cnt->error_code = http_bad_request;
		return ret_error;
	}

	/* Look for ".."
	 */
	if (strstr (cnt->request, "..")) {
		cnt->error_code = http_bad_request;
		return ret_error;
	}

	/* Look for starting '/' in the request
	 */
	if (cnt->request[0] != '/') {
		cnt->error_code = http_bad_request;
		return ret_error;		
	}

	cnt->error_code = http_ok;
	return ret_ok;	
}


ret_t 
cherokee_connection_get_plugin_entry (cherokee_connection_t *cnt, cherokee_plugin_table_t *plugins, cherokee_plugin_table_entry_t **plugin_entry)
{
	ret_t ret;

	/* Look for the handler "*_new" function
	 */
	ret = cherokee_plugin_table_get (plugins, cnt->request, plugin_entry, &cnt->web_directory);
	if (ret != ret_ok) {
		cnt->error_code = http_internal_error;
		return ret_error;
	}	

	return ret_ok;
}


ret_t 
cherokee_connection_create_handler (cherokee_connection_t *cnt, cherokee_plugin_table_entry_t *plugin_entry)
{
	ret_t ret;

	/* Create and assign a handler object
	 */
	ret = (plugin_entry->handler_new_func) ((void **)&cnt->handler, cnt, plugin_entry->properties);
	if (cnt->handler == NULL) {
		cnt->error_code = http_internal_error;
		return ret_error;
	}

	/* Check the handler state after execute the constructor
	 */
	if (ret == ret_ok) {
		cherokee_buffer_ensure_size (cnt->buffer, DEFAULT_READ_SIZE);
	}

	return ret_ok;
}


ret_t
cherokee_connection_parse_header (cherokee_connection_t *cnt)
{	   
	char *ptr;

	/* Look for "Connection: Keep-Alive / Close"
	 */
	if ((ptr = strstr (cnt->buffer->buf, "Connection: "))) {
		ptr += 12;
		if (strncasecmp (ptr, "Keep-Alive", 10) == 0) {
			if (cnt->keep_alive == 0) {
				cnt->keep_alive = MAX_KEEP_ALIVE;
			} else {
				cnt->keep_alive--;
			}
		} else if (strncasecmp (ptr, "close", 5) == 0) {
			cnt->keep_alive = 0;
		}
	}
	
	/* Look for "Range:" 
	 */
	if (cnt->handler->support & hsupport_range) {
		if ((ptr = strstr (cnt->buffer->buf, "Range: bytes="))) {
			ptr += 13;
			get_range (cnt, ptr);
		}
	}

	return ret_ok;
}


ret_t
cherokee_connection_open_request (cherokee_connection_t *cnt)
{
	return cherokee_handler_init (cnt->handler, cnt->request, cnt->request_local);
}


ret_t
cherokee_connection_send_response_page (cherokee_connection_t *cnt)
{
	   ret_t  ret;
	   
	   cherokee_buffer_make_empty  (cnt->buffer);
	   
	   cherokee_buffer_add (cnt->buffer, "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">" CRLF, 50);
	   
	   /* Add page title
	    */
	   cherokee_buffer_add (cnt->buffer, "<html><head><title>", 19);
	   add_error_code_string_to_buffer (cnt);

	   /* Add big banner
	    */
	   cherokee_buffer_add (cnt->buffer, "</title></head><body><h1>", 25);
	   add_error_code_string_to_buffer (cnt);
	   cherokee_buffer_add (cnt->buffer, "</h1>", 5);

	   /* Maybe add some info
	    */
	   switch (cnt->error_code) {
	   case http_not_found:
			 if (cnt->request) {
				    cherokee_buffer_add (cnt->buffer, "The requested URL ", 18);
				    cherokee_buffer_add (cnt->buffer, cnt->request, cnt->request_len);
				    cherokee_buffer_add (cnt->buffer, " was not found on this server.", 30);
			 }
			 break;
	   case http_bad_request:
			 cherokee_buffer_add (cnt->buffer, "Your browser sent a request that this server could not understand.", 66);
			 break;
	   case http_moved_permanently:
			 cherokee_buffer_add (cnt->buffer, "The document has moved", 22);
			 break;
	   default:
			 break;
	   }
	   
	   /* Add page foot
	    */
	   cherokee_buffer_add (cnt->buffer, "<p><hr><address>Cherokee web server " VERSION "</address></body></html>" CRLF, 67); 
   
	   /* Send it
	    */
	   ret = write (cnt->socket, cnt->buffer->buf, cnt->buffer->len);
	   cherokee_buffer_make_empty  (cnt->buffer);

	   if (ret > 0)
			 return ret_ok;
	   else
			 return ret_error;
}
