/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "handler_gnomevfs.h"
#include "connection.h"


ret_t 
cherokee_handler_gnomevfs_new (cherokee_handler_t **hdl, void *cnt, cherokee_table_t *properties)
{
	cherokee_handler_gnomevfs_t *n = (cherokee_handler_gnomevfs_t *)malloc(sizeof(cherokee_handler_gnomevfs_t));
	return_if_fail (n != NULL, ret_nomem);

	/* Init the base class object
	 */
	cherokee_handler_init_base(HANDLER(n));
	HANDLER(n)->connection  = cnt;

	HANDLER(n)->support = hsupport_length | hsupport_range;

	HANDLER(n)->init        = (func_init_t) cherokee_handler_gnomevfs_init;
	HANDLER(n)->free        = (func_free_t) cherokee_handler_gnomevfs_free;
	HANDLER(n)->step        = (func_step_t) cherokee_handler_gnomevfs_step;
	HANDLER(n)->add_headers = (func_add_headers_t) cherokee_handler_gnomevfs_add_headers;

	*hdl = HANDLER(n);
	
	/* Init
	 */
	n->filedir = cherokee_table_get (properties, "filedir");
	n->filedir_len = strlen (n->filedir);

	n->vfs_handler = NULL;
	n->openning    = open_none;
	n->reading     = read_none;

	return ret_ok;	
}


static void
open_cb (GnomeVFSAsyncHandle *handle,
	 GnomeVFSResult result,
	 cherokee_handler_gnomevfs_t *hdl)
{	
	if (result != GNOME_VFS_OK) {
		CONN(HANDLER(hdl)->connection)->error_code = http_not_found;
	}

	hdl->openning = open_finished;
}


static gchar *
build_gnomevfs_uri (cherokee_handler_gnomevfs_t *hdl, char *request, char *request_local)
{
	int len;
	gchar *web_directory;
	gchar *begin, *tmp;
	gchar *uri;


	/* Get the requested web_directory
	 */
	if (CONN(HANDLER(hdl)->connection)->web_directory[0] == '/') {
		web_directory = CONN(HANDLER(hdl)->connection)->web_directory + 1;
	} else {
		web_directory = CONN(HANDLER(hdl)->connection)->web_directory;
	}	
	
	
	/* Search the web_directory in the request string
	 */
	begin = strstr (request, web_directory);
	if (begin == NULL) goto error;
	begin += strlen (web_directory) + 1;


	if (! strncmp (begin, "file/", 5)) {
		len = strlen(begin) + 3 + hdl->filedir_len +1;
		uri = (char *)malloc(len);

		if (hdl->filedir[hdl->filedir_len-1] == '/') {
			snprintf (uri, len, "file://%s%s", hdl->filedir, begin+5);
		} else {
			snprintf (uri, len, "file://%s/%s", hdl->filedir, begin+5);
		}

	} else if (! strncmp (begin, "ftp/", 4)) {
		len = strlen(begin) + 3;
		uri = (char *)malloc(len);

		snprintf (uri, len, "ftp://%s", begin+4);

	} else if (! strncmp (begin, "http/", 5)) {
		len = strlen(begin) + 3;
		uri = (char *)malloc(len);

		snprintf (uri, len, "http://%s", begin+5);

	} else if (! strncmp (begin, "smb/", 4)) {
		len = strlen(begin) + 3;
		uri = (char *)malloc(len);

		snprintf (uri, len, "smb://%s", begin+4);
	}


	/* Replace '$' by '#'
	 */
	tmp = uri;
	while (*tmp != '\0') {
		if (*tmp == '$') *tmp = '#';
		tmp++;
	}

	return uri;

error:
	return NULL;
}


ret_t 
cherokee_handler_gnomevfs_init (cherokee_handler_gnomevfs_t *hdl, char *request, char *request_local)
{	
	switch (hdl->openning) {
	case open_none: {
		gchar *uri;

		uri = build_gnomevfs_uri (hdl, request, request_local);
		if (uri == NULL) goto error;

		printf ("uri='%s'\n", uri);
		
		hdl->openning = open_requested;
		gnome_vfs_async_open (&hdl->vfs_handler,
				      uri,
				      GNOME_VFS_OPEN_READ,
				      GNOME_VFS_PRIORITY_DEFAULT,
				      open_cb,
				      hdl);

		g_free (uri);
	}

	case open_requested: {
		int i = 5;

		while (g_main_context_iteration(NULL, FALSE) && i--);
		return ret_eagain;
	}
	case open_finished:
		if (CONN(HANDLER(hdl)->connection)->error_code != http_ok) {
			return ret_error;
		}
	}

	return ret_ok;


error:
	CONN(HANDLER(hdl)->connection)->error_code = http_internal_error;
	return ret_ok;
}

static void
close_cb (GnomeVFSAsyncHandle *handle,
	  GnomeVFSResult result,
	  gpointer callback_data)
{
	printf ("closecb\n");
}


ret_t 
cherokee_handler_gnomevfs_free (cherokee_handler_gnomevfs_t *hdl)
{
	if (hdl->vfs_handler != NULL) {
		gnome_vfs_async_close (hdl->vfs_handler,
				       close_cb, 
				       NULL);
	}

	free (hdl);
	return ret_ok;
}


static void 
read_cb (GnomeVFSAsyncHandle *handle,
	 GnomeVFSResult result,
	 gpointer buffer,
	 GnomeVFSFileSize bytes_requested,
	 GnomeVFSFileSize bytes_read,
	 cherokee_handler_gnomevfs_t *hdl)
{
	hdl->bytes_read = bytes_read;
	hdl->reading = read_finished;
}


ret_t 
cherokee_handler_gnomevfs_step (cherokee_handler_gnomevfs_t *hdl, cherokee_buffer_t *buffer)
{
	switch (hdl->reading) {
	case read_none: 
		hdl->reading = read_requested;
		gnome_vfs_async_read (hdl->vfs_handler,
				      buffer->buf,
				      buffer->size,
				      read_cb,
				      hdl);

	case read_requested: {
		int i = 5;

		while (g_main_context_iteration(NULL, FALSE) && i--);	      
		return ret_eagain;
	} 

	case read_finished:
		hdl->reading = read_none;
		buffer->len = hdl->bytes_read;
	}

	return ret_ok;
}


ret_t 
cherokee_handler_gnomevfs_add_headers (cherokee_handler_gnomevfs_t *hdl, cherokee_buffer_t *buffer)
{
	return ret_ok;
}




/*   Library init function
 */

static int _gnomevfs_is_init = 0;

void 
gnomevfs_init ()
{
	if (_gnomevfs_is_init) {
		return;
	}
	_gnomevfs_is_init = 1;

	gnome_vfs_init();
}
