/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <sys/types.h>
#include <unistd.h>

#include "mime.h"
#include "handler_file.h"
#include "connection.h"


ret_t
cherokee_handler_file_new  (cherokee_handler_t **hdl, void *cnt, cherokee_table_t *properties)
{
	cherokee_handler_file_t *n = (cherokee_handler_file_t *)malloc(sizeof(cherokee_handler_file_t));
	return_if_fail (n != NULL, ret_nomem);
	
	/* Init the base class object
	 */
	cherokee_handler_init_base(HANDLER(n));
	HANDLER(n)->connection  = cnt;

	HANDLER(n)->support = hsupport_length | hsupport_range;

	HANDLER(n)->init        = (func_init_t) cherokee_handler_file_init;
	HANDLER(n)->free        = (func_free_t) cherokee_handler_file_free;
	HANDLER(n)->step        = (func_step_t) cherokee_handler_file_step;
	HANDLER(n)->add_headers = (func_add_headers_t) cherokee_handler_file_add_headers;

	*hdl = HANDLER(n);
	
	/* Init
	 */
	n->offset = 0;
	n->mime   = NULL;

	return ret_ok;
}


ret_t 
cherokee_handler_file_init (cherokee_handler_file_t *n)
{
	int   ret;
	char *ext;
	cherokee_connection_t *conn;

	conn = CONN(HANDLER(n)->connection);

	cherokee_buffer_add (conn->local_directory, conn->request->buf, conn->request->len);
	n->fd = open (conn->local_directory->buf, O_RDONLY);
	cherokee_buffer_drop_endding (conn->local_directory, conn->request->len);

	/* Cannot open the file
	 */
	ret = fstat (n->fd, &n->info);

	if (n->fd == -1) {
		if ((ret == 0) && S_ISREG(n->info.st_mode)) {
			CONN(HANDLER(n)->connection)->error_code = http_access_denied;
		} else {
			CONN(HANDLER(n)->connection)->error_code = http_not_found;
		}
		return ret_error;
	}


	/* Is dir?
	 */
	if (S_ISDIR(n->info.st_mode)) {
		CONN(HANDLER(n)->connection)->error_code = http_access_denied;
		return ret_error;		
	}


	/* Range
	 */
	if (HANDLER_CONN(n)->range_start != 0) {
		n->offset = HANDLER_CONN(n)->range_start;
		lseek (n->fd, n->offset, SEEK_SET);
		HANDLER_CONN(n)->error_code = http_partial_content;
	}

	if (HANDLER_CONN(n)->range_end == 0) {
		HANDLER_CONN(n)->range_end = n->info.st_size;
	}

	/* Look for the mime type
	 */
	ext = rindex (conn->request->buf, '.');
	if (ext != NULL) {
		cherokee_mime_t *m;

		cherokee_mime_get_default (&m);		
		n->mime = cherokee_mime_get (m, ext+1);
	}

	return ret_ok;
}


ret_t
cherokee_handler_file_free (cherokee_handler_file_t *fhdl)
{
	if (fhdl->fd != -1) {
		close (fhdl->fd);
		fhdl->fd = -1;
	}

	free (fhdl);
	
	return ret_ok;
}


ret_t
cherokee_handler_file_step (cherokee_handler_file_t *fhdl, 
			    cherokee_buffer_t       *buffer)
{
	int total;

	total = read (fhdl->fd, buffer->buf, buffer->size);	
	switch (total) {
	case 0:
		return ret_eof;
	case -1:
		return ret_error;
	default:
		buffer->len = total;
		fhdl->offset += total;
	}	

	/* Maybe it was the last file chunk
	 */
	if (fhdl->offset >= CONN(HANDLER(fhdl)->connection)->range_end) {
		return ret_eof_have_data;
	}

	return ret_ok;
}


ret_t
cherokee_handler_file_add_headers (cherokee_handler_file_t *fhdl,
				   cherokee_buffer_t       *buffer)
{
	int length;

	/* Add "Content-length:" header
	 */
	if (fhdl->fd != -1) {
		/* We stat()'ed the file in the handler constructor
		 */
		length = HANDLER_CONN(fhdl)->range_end - HANDLER_CONN(fhdl)->range_start;
		if (length < 0) {
			length = 0;
		}

		length = snprintf (gbl_buffer, gbl_buffer_size, "Content-length: %d"CRLF, length);
		cherokee_buffer_add (buffer, gbl_buffer, length);

	} else {
		/* Can't use Keep-alive w/o "Content-length:", so
		 */
		HANDLER_CONN(fhdl)->keep_alive = 0;
	}

	/* Add "Content-Type:" header
	 */
	if (fhdl->mime != NULL) {
		length = snprintf (gbl_buffer, gbl_buffer_size, "Content-Type: %s"CRLF, fhdl->mime);
		cherokee_buffer_add (buffer, gbl_buffer, length);		
	}

	return ret_ok;
}



/*   Library init function
 */

static int _file_is_init = 0;

void
file_init ()
{
	cherokee_mime_t *mime;

	/* Init flag
	 */
	if (_file_is_init) {
		return;
	}
	_file_is_init = 1;


	/* Library dependences
	 */
	cherokee_mime_init (&mime);
}
