/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004, 2005 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common-internal.h"
#include "admin_client.h"

#include "buffer.h"
#include "downloader.h"
#include "downloader-protected.h"
#include "config_value.h"


typedef enum {
	phase_init,
	phase_stepping
} phase_t;


struct cherokee_admin_client {
	cherokee_config_value_t  *request_conf;
	cherokee_config_value_t  *reply_conf;

	cherokee_buffer_t        *url_ref;
	cherokee_buffer_t         request;

	phase_t                   phase;
	cherokee_fdpoll_t        *poll_ref;
	cherokee_downloader_t     downloader;
};


ret_t 
cherokee_admin_client_new (cherokee_admin_client_t **admin)
{
	CHEROKEE_NEW_STRUCT(n,admin_client);

	/* Init
	 */
	n->phase        = phase_init;

	n->request_conf = NULL;
	n->reply_conf   = NULL;

	n->poll_ref     = NULL;
	n->url_ref      = NULL;

	cherokee_downloader_init (&n->downloader);
	cherokee_buffer_init (&n->request);

	/* Return the object
	 */
	*admin = n;
	return ret_ok;
}


ret_t 
cherokee_admin_client_free (cherokee_admin_client_t *admin)
{
	cherokee_buffer_mrproper (&admin->request);
	cherokee_downloader_mrproper (&admin->downloader);

	free (admin);
	return ret_ok;
}


ret_t 
cherokee_admin_client_set_fdpoll (cherokee_admin_client_t *admin, cherokee_fdpoll_t *poll)
{
	if (admin->poll_ref != NULL) {
		PRINT_ERROR_S ("WARNING: Overwritting poll object\n");
	}

	admin->poll_ref = poll;
	return ret_ok;
}

static ret_t 
cherokee_downloader_finish (void *dload, void *param)
{
	ret_t                    ret;
	cherokee_admin_client_t *admin      = ADMIN_CLIENT(param);
	cherokee_downloader_t   *downloader = DOWNLOADER(dload);
	cherokee_buffer_t       *reply_body = &downloader->body;

	ret = cherokee_config_value_parse (&admin->reply_conf, reply_body->buf);
	if (unlikely (ret != ret_ok)) {
		PRINT_ERROR("ERROR: Couldn't parse: '%s'\n", reply_body->buf);
		return ret_error;
	}

	return ret_ok;
}


ret_t 
cherokee_admin_client_prepare (cherokee_admin_client_t *admin)
{
	ret_t ret;

	/* Serialize configuration
	 */
	if (admin->request_conf != NULL) {
		ret = cherokee_config_value_serialize (admin->request_conf, &admin->request);
		if (unlikely (ret != ret_ok)) return ret_ok;
	}
	
	/* Set up the downloader object properties
	 */
	ret = cherokee_downloader_set_fdpoll (&admin->downloader, admin->poll_ref);
	if (unlikely (ret != ret_ok)) return ret;
	
	ret = cherokee_downloader_set_url (&admin->downloader, admin->url_ref);
	if (unlikely (ret != ret_ok)) return ret;
	
	ret = cherokee_downloader_set_post (&admin->downloader, &admin->request);
	if (unlikely (ret != ret_ok)) return ret;
	
	/* Set up the downloader object signals
	 */
	cherokee_downloader_connect (&admin->downloader, downloader_event_finish, 
				     cherokee_downloader_finish, admin);

	/* Prepare downloader
	 */
	ret = cherokee_downloader_prepare (&admin->downloader);
	if (unlikely (ret != ret_ok)) return ret;	

	return ret_ok;
}


ret_t 
cherokee_admin_client_step (cherokee_admin_client_t *admin)
{
	return cherokee_downloader_step (&admin->downloader);
}


ret_t 
cherokee_admin_client_add_raw (cherokee_admin_client_t *admin, cherokee_buffer_t *thing)
{
	ret_t ret;

	ret = cherokee_buffer_add_buffer (&admin->request, thing);
	if (unlikely (ret != ret_ok)) return ret;
	
	return ret_ok;
}


ret_t 
cherokee_admin_client_get_reply (cherokee_admin_client_t *admin, cherokee_config_value_t **conf)
{
	if (admin->reply_conf == NULL) 
		return ret_not_found;

	*conf = admin->reply_conf;
	return ret_ok;
}


ret_t 
cherokee_admin_client_set_url (cherokee_admin_client_t *admin, cherokee_buffer_t *url)
{
	if (admin->url_ref != NULL) {
		PRINT_ERROR_S("WARNING: Overwritting URL\n");
	}

	admin->url_ref = url;
	return ret_ok;
}
