/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2003 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include "handler_dirlist.h"
#include "connection.h"


ret_t
cherokee_handler_dirlist_new  (cherokee_handler_t **hdl, void *cnt, cherokee_table_t *properties)
{	
	cherokee_handler_dirlist_t *n = (cherokee_handler_dirlist_t *)malloc(sizeof(cherokee_handler_dirlist_t));
	return_if_fail (n != NULL, ret_nomem);
	
	/* Init the base class object
	 */
	cherokee_handler_init_base(HANDLER(n));
	HANDLER(n)->connection  = cnt;

	HANDLER(n)->support = 0;

	HANDLER(n)->init        = (func_init_t) cherokee_handler_dirlist_init;
	HANDLER(n)->free        = (func_free_t) cherokee_handler_dirlist_free;
	HANDLER(n)->step        = (func_step_t) cherokee_handler_dirlist_step;
	HANDLER(n)->add_headers = (func_add_headers_t) cherokee_handler_dirlist_add_headers; 

	*hdl = HANDLER(n);
	
	/* Init
	 */
	n->dir           = NULL;
	n->request       = NULL;
	n->request_len   = 0;
	n->page_begining = 0;

	return ret_ok;
}


ret_t
cherokee_handler_dirlist_init (cherokee_handler_dirlist_t *n, char *request, char *request_local)
{
	n->request = request;
	n->request_len = strlen(request);

	/* If directory name doesn't finish with a slash,
	 * _add_headers() is going to set a: http_moved_permanently
	 */
	if (request[n->request_len-1] != '/') {
		HANDLER(n)->redirect = (char *) malloc (n->request_len+1);
		memcpy (HANDLER(n)->redirect, request, n->request_len);
		
		HANDLER(n)->redirect[n->request_len] = '/';
		HANDLER(n)->redirect[n->request_len+1] = '\0';
		
		CONN(HANDLER(n)->connection)->error_code = http_moved_permanently;

		return ret_error;
	}

	
	/* Open dir
	 */
	n->dir = opendir (request_local);
	if (n->dir == NULL) {
		CONN(HANDLER(n)->connection)->error_code = http_not_found;
		return ret_error;
	}

	return ret_ok;
}


ret_t
cherokee_handler_dirlist_free (cherokee_handler_dirlist_t *dhdl)
{
	
	if (dhdl->dir != NULL) {
		closedir (dhdl->dir);
	}
	
	free (dhdl);
	
	return ret_ok;
}


static inline int
entry_is_dir (char *dir,  int dir_len,
	      char *file, int file_len)
{
	   struct stat f;
	   
	   char *m = (char *) malloc(dir_len + file_len + 1);
	   memcpy (m, dir, dir_len);
	   memcpy (m+dir_len, file, file_len);
	   m[dir_len + file_len] = '\0';
	   
	   stat (m, &f);
	   free (m);
	   
	   return S_ISDIR(f.st_mode);
}


ret_t
cherokee_handler_dirlist_step (cherokee_handler_dirlist_t *dhdl, cherokee_buffer_t *buffer)
{
	struct dirent *entry;

	/*                       	 0        1         2         3         4         5         6         7
					 1234567890123456789012345678901234567890123456789012345678901234567890 */
	
	   
	/* Page header
	 */
	if (dhdl->page_begining == 0) {
		cherokee_buffer_add (buffer, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">" CRLF, 57);
		cherokee_buffer_add (buffer, "<html><head><title>Index of ", 28);
		cherokee_buffer_add (buffer, dhdl->request, dhdl->request_len);
		cherokee_buffer_add (buffer, "</title></head><body><h1>Index of ", 34);
		cherokee_buffer_add (buffer, dhdl->request, dhdl->request_len);
		cherokee_buffer_add (buffer, "</h1><hr><pre>", 14);
		
		dhdl->page_begining = 1;
		return ret_ok;
	}


	/* Page entries
	 */
	while ((entry = readdir (dhdl->dir)) != NULL)
	{ 
		int num = 0;
		int is_a_dir;
 		int entry_len = strlen(entry->d_name);
		
		/* Ignore backup files
		 */
		if ((entry->d_name[0] == '.') ||
		    (entry->d_name[0] == '#') ||
		    (entry->d_name[entry_len-1] == '~')) 
		{
			continue;
		}
		
		/* Add an item
		 */
		is_a_dir = entry_is_dir(dhdl->request, dhdl->request_len, 
					entry->d_name, entry_len);
		
		cherokee_buffer_add (buffer, "<a href=\"", 9); 
		cherokee_buffer_add (buffer, entry->d_name, entry_len);
		if (is_a_dir)
			cherokee_buffer_add (buffer, "/", 1);
		cherokee_buffer_add (buffer, "\">", 2); 
		cherokee_buffer_add (buffer, entry->d_name, entry_len);
		if (is_a_dir)
			cherokee_buffer_add (buffer, "/", 1);
		cherokee_buffer_add (buffer, "</a>\n", 5);			 
		
		/* Add up to 15 each time
		 */
		if (++num >= 15) {
			return ret_ok;
		}
	}

	/* Page ending
	 */
	cherokee_buffer_add (buffer, "</pre><hr><address>Cherokee web server " VERSION "</address></body></html>", 68);
	return ret_eof_have_data;
}


ret_t
cherokee_handler_dirlist_add_headers (cherokee_handler_dirlist_t *dhdl, cherokee_buffer_t *buffer)
{
	if (HANDLER(dhdl)->redirect) {
		cherokee_buffer_add (buffer, "Location: ", 10);
		cherokee_buffer_add (buffer, HANDLER(dhdl)->redirect, strlen(HANDLER(dhdl)->redirect));
		cherokee_buffer_add (buffer, CRLF, 2);
	} else {
		/* This handler works on-the-fly sending some files each 
		 * iteration, so it doesn't know the "Content-length:" 
		 * and the server can't use Keep-alive.
		 */
		CONN(HANDLER(dhdl)->connection)->keep_alive = 0;
	}

	return ret_ok;
}



/* Library init function
 */
void
dirlist_init (void)
{
}
