/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>

#include "plugin_table.h"
#include "handler.h"

ret_t
cherokee_plugin_table_new  (cherokee_plugin_table_t **pt)
{
	cherokee_plugin_table_t *n = (cherokee_plugin_table_t *)malloc(sizeof(cherokee_plugin_table_t));

	cherokee_table_init (&n->table);
	cherokee_table_init (&n->handlers);
	
	*pt = n;
	
	return ret_ok;
}


ret_t
cherokee_plugin_table_free (cherokee_plugin_table_t *pt)
{
	cherokee_table_clean (&pt->table);
	cherokee_table_clean (&pt->handlers);
	
	free (pt);
	
	return ret_ok;
}


void *
cherokee_plugin_table_get_from_url_BAK (cherokee_plugin_table_t *pt, char *dir)
{
	void *ret = NULL;
	int   length;
	char *slash;
	char *url;
	
	url = strdup(dir);
	
	do {
		ret = cherokee_table_get (&pt->table, url);
		
		/* Found
		 */
		if (ret != NULL)
			goto go_out;
		
		length = strlen(url);
		if (length == 0)
			goto go_out;
		
		/* Modify url for next loop
		 */
		if (url[length-1] == '/') {
			url[length-1] = '\0';
		} else {
			slash = rindex (url, '/');
			slash[1] = '\0';
		}
	} while (ret == NULL);


go_out:
	free (url);
	return ret;
}


ret_t 
cherokee_plugin_table_get (cherokee_plugin_table_t *pt, char *requested_url, cherokee_plugin_table_entry_t **plugin_entry, char **web_dir)
{
        /* NOTE: This function get new memory for web_dir parameter. 
	 * Remember to free() it!!
	 */

	ret_t ret;
	int   length;
	char *slash;
	char *url;
	cherokee_plugin_table_entry_t *entry = NULL;

	
	url = strdup(requested_url);
	
	do {
		entry = cherokee_table_get (&pt->table, url);
		
		/* Found
		 */
		if (entry != NULL)
			goto go_out;
 
		length = strlen (url);
		if (length == 0)
			goto go_out;

		/* Modify url for next loop:
		 * Find the last / and finish the string there.
		 */
		if (url[length-1] == '/') {
			url[length-1] = '\0';
		} else {
			slash = rindex (url, '/');
			slash[1] = '\0';
		}
	} while (entry == NULL);


go_out:	
	/* Set the in-out values
	 */
	*plugin_entry = entry;
	
	if (web_dir != NULL) {
		*web_dir = url;
	} else {
		free (url);
	}

	if (entry == NULL) 
		return ret_error;
	return ret_ok;
}


static inline func_new_t
load_plugin_library (char *plugin_name)
{
	void       *lib;
	func_new_t  func_new;
	void      (*func_init) (void);
	int         dl_flags;
	
	/* Load the library
	 */	
#ifdef HAVE_RTLDNOW
	dl_flags = RTLD_NOW;
#else
	dl_flags = RTLD_LAZY;
#endif

#ifdef HAVE_RTLDGLOBAL
	dl_flags |= RTLD_GLOBAL;
#endif

	snprintf (gbl_buffer, gbl_buffer_size, "%s/libcherokee_%s." SO_SUFFIX, CHEROKEE_PLUGINDIR, plugin_name);
	lib = dlopen (gbl_buffer, dl_flags);
	if (lib == NULL) {
		fprintf (stderr, "Can't read the plug-in (%s): %s\n", gbl_buffer, dlerror());
		return NULL;
	}
	

	/* Maybe execute the library init function
	 */
	snprintf (gbl_buffer, gbl_buffer_size, "%s_init", plugin_name);
	func_init = dlsym(lib, gbl_buffer);
	if (func_init != NULL) {
		func_init();
	}


	/* Get the constructor
	 */
	snprintf (gbl_buffer, gbl_buffer_size, "cherokee_handler_%s_new", plugin_name);
	func_new = (func_new_t) dlsym(lib, gbl_buffer);
	if (func_new == NULL) {
		fprintf (stderr, "Can't load plug-in (%s): %s\n", gbl_buffer, dlerror());
		return NULL;
	}
	
	return func_new;
}


static inline func_new_t
get_loaded_function (char *plugin_name)
{
	void (*func_init) (void);

	/* If the library init function exist, it should be
	 * called to, maybe, init the library.
	 */
	snprintf (gbl_buffer, gbl_buffer_size, "%s_init", plugin_name);
	func_init = dlsym(NULL, gbl_buffer);

#if 0
	printf ("get_loaded_function: %s: %p\n", gbl_buffer, func_init);
#endif

	if ((func_init == NULL) || (dlerror() != NULL)) {
		return NULL;
	} 
	func_init();

	
	snprintf (gbl_buffer, gbl_buffer_size, "cherokee_handler_%s_new", plugin_name);
	return (func_new_t) dlsym(NULL, gbl_buffer);	
}


ret_t 
cherokee_plugin_table_set_BAK (cherokee_plugin_table_t *pt, char *dir, char *handler)
{
	ret_t      ret;
	func_new_t func;


	/* Get the cached "handler"_new function
	 */
	func = (func_new_t) cherokee_table_get (TABLE(&pt->handlers), handler);
	if (func == NULL) {

		/* Maybe the plugin is compiled statically
		 */
		func = (func_new_t) get_loaded_function(handler);

		if (func == NULL) {
			func = load_plugin_library (handler);
			if (func == NULL) {
				return ret_error;
			}
		}
		
		/* Add to table
		 */
		ret = cherokee_table_add (&pt->handlers, handler, (void*)func); 
		if (ret != ret_ok) {
			return ret_error;
		}
	}
	
	
	/* Add "handler" <-> handler_new func
	 */
	cherokee_table_add (TABLE(&pt->table), dir, (void*)func);
	
	return ret_ok;
}


ret_t 
cherokee_plugin_table_set (cherokee_plugin_table_t *pt, char *dir,  char *handler_name, cherokee_plugin_table_entry_t *plugin_entry)
{
	ret_t      ret;
	func_new_t func;

	/* Get the cached "handler"_new function
	 */
	func = (func_new_t) cherokee_table_get (TABLE(&pt->handlers), handler_name);
	if (func == NULL) {

		/* Maybe the plugin is compiled statically
		 */
		func = (func_new_t) get_loaded_function (handler_name);

		if (func == NULL) {
			func = load_plugin_library (handler_name);
			if (func == NULL) {
				return ret_error;
			}
		}
		
		
		/* Add to table
		 */
		ret = cherokee_table_add (&pt->handlers, handler_name, (void*)func); 
		if (ret != ret_ok) {
			return ret_error;
		}
	}
	
	plugin_entry->handler_new_func = func;
	
	
	/* Add to "dir <-> plugin_entry" table
	 */
	cherokee_table_add (TABLE(&pt->table), dir, (void*)plugin_entry);
	
	return ret_ok;
}
