/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "thread.h"

#include <signal.h>
#include <errno.h>

#include "server.h"
#include "socket.h"
#include "connection.h"

#define DEBUG_BUFFER(b)  fprintf(stderr, "%s:%d len=%d crc=%d\n", __FILE__, __LINE__, b->len, cherokee_buffer_crc32(b))


static ret_t
set_sigpipe (void)
{
       struct sigaction new_act;
       struct sigaction old_act;

       memset (&new_act, 0, sizeof( new_act ));
       memset (&old_act, 0, sizeof( old_act ));

       new_act.sa_handler = SIG_IGN;

       if (sigaction (SIGPIPE, &new_act, &old_act)) {
               PRINT_ERROR ("sigaction (SIGPIPE, ...): %s (%d)\n", strerror(errno), errno);
	       return ret_error;
       }

       if (sigaction (SIGBUS, &new_act, &old_act)) {
               PRINT_ERROR ("sigaction (SIGBUS, ...): %s (%d)\n", strerror(errno), errno);
	       return ret_error;
       }

       return ret_ok;
}


#ifdef HAVE_PTHREAD
static void *
thread_routine (void *data)
{
        cherokee_thread_t *thread = THREAD(data);
        cherokee_server_t *srv    = THREAD_SRV(data);

	set_sigpipe();	

        while (thread->exit == 0) {
                cherokee_thread_step (thread, false);
        }

        return NULL;
}
#endif


ret_t 
cherokee_thread_new  (cherokee_thread_t **thd, void *server, cherokee_thread_type_t type, 
		      int system_fd_num, int fd_num)
{
	cherokee_server_t *srv = SRV(server);
	CHEROKEE_NEW_STRUCT (n, thread);

	/* Init
	 */
	INIT_LIST_HEAD((list_t*)&n->base);
	INIT_LIST_HEAD((list_t*)&n->active_list);
	INIT_LIST_HEAD((list_t*)&n->reuse_list);
	
	cherokee_fdpoll_new (&n->fdpoll, system_fd_num, fd_num);

	n->exit              = 0;
	n->active_list_num   = 0;
	n->pending_conns_num = 0;
	n->server            = server;
	n->thread_type       = type;

	/* Accepting information
	 */
	n->accept.recalculate    = 0;
	n->accept.continuous     = 0;
	n->accept.continuous_max = 0;
	
	/* Do some related work..
	 */
	if (type == thread_async) {
#ifdef HAVE_PTHREAD
		pthread_attr_t attr;
		struct sched_param param;

		/* Init the thread attributes
		 */
		pthread_attr_init (&attr);
		pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_DETACHED);
		
		/* Maybe set the scheduling policy
		 */
		if (srv->thread_policy != -1) {
			pthread_attr_setschedpolicy (&attr, srv->thread_policy);
		}

		pthread_create (&n->thread, &attr, thread_routine, n);
#else
		SHOULDNT_HAPPEN;
#endif		
	}
	
	/* Return the object
	 */
	*thd = n;
	return ret_ok;
}


static void
conn_set_mode (cherokee_thread_t *thd, cherokee_connection_t *conn, cherokee_socket_status_t s)
{
	cherokee_socket_set_status (conn->socket, s);
	cherokee_fdpoll_set_mode (thd->fdpoll, SOCKET_FD(conn->socket), s);
}


static void
add_connection (cherokee_thread_t *thd, cherokee_connection_t *conn)
{
	list_add ((list_t *)conn, &thd->active_list);
	thd->active_list_num++;
}


static void
del_connection (cherokee_thread_t *thd, cherokee_connection_t *conn)
{
	list_del ((list_t *)conn);
	thd->active_list_num--;
}


static void
purge_closed_connection (cherokee_thread_t *thread, cherokee_connection_t *conn)
{
	/* Delete from file descriptors poll
	 */
	cherokee_fdpoll_del (thread->fdpoll, SOCKET_FD(conn->socket));

	/* Delete from active connections list
	 */
	del_connection (thread, conn);

	/* Maybe have a delayed log
	 */
	cherokee_connection_log_delayed (conn);

	/* Clean the connection and close the socket
	 */
	cherokee_connection_mrproper (conn);

	/* Add it to the rusable list
	 */	
	cherokee_thread_set_reusable_connection (thread, conn);
}


static void
maybe_purge_closed_connection (cherokee_thread_t *thread, cherokee_connection_t *conn)
{
	cherokee_server_t *srv = thread->server;

	/* No-keepalive connection
	 */
	if (conn->keepalive <= 0) {
		purge_closed_connection (thread, conn);
		return;
	} 
	
	conn->keepalive--;

	/* TCP cork
	 */
	if (conn->tcp_cork) {
		cherokee_connection_set_cork (conn, 0);
	}
	
	cherokee_connection_log_delayed (conn);
	cherokee_connection_clean (conn);

	conn_set_mode (thread, conn, socket_reading);
	conn->timeout = srv->bogo_now + srv->timeout;
}


ret_t 
process_active_connections (cherokee_thread_t *thd)
{
	ret_t    ret;
	off_t    len;
	list_t *i, *tmp;
	cherokee_boolean_t process;
	
	cherokee_connection_t *conn = NULL;
	cherokee_server_t     *srv  = thd->server;


	/* Process active connections
	 */
	list_for_each_safe (i, tmp, (list_t*)&thd->active_list) {
		conn = CONN(i);

		/* May the connection was too much time w/o any work
		 */
		if (conn->timeout < srv->bogo_now) {
			purge_closed_connection (thd, conn);
			continue;
		}

		/* Process the connection?
		 * 1.- Are there a pipelined connection?
		 */
		process = ((conn->phase == phase_reading_header) &&
			   (!cherokee_buffer_is_empty (conn->incoming_header)));
			
		/* Process the connection?
		 * 2.- Inspect the file descriptor	
		 */
		if (process == false) {
			ret = cherokee_fdpoll_check (thd->fdpoll, 
						     SOCKET_FD(conn->socket), 
						     SOCKET_STATUS(conn->socket));
			
			switch (ret) {
			case -1:
				purge_closed_connection(thd, conn);
				continue;
			case 0:
				continue;
			}

			process = true;
		}
		
		/* Process the connection?
		 * Finial.- 
		 */
		if (process == false) {
			continue;
		}

		/* The connection has work to do, so..
		 */
		conn->timeout = srv->bogo_now + srv->timeout;

		/* Phases
		 */
		switch (conn->phase) {
		case phase_tls_handshake:
			ret = cherokee_socket_init_tls (CONN_SOCK(conn), CONN_VSRV(conn));
			switch (ret) {
			case ret_eagain:
				continue;
			case ret_error:
				purge_closed_connection (thd, conn);
				break;
			case ret_ok:
				conn->phase = phase_reading_header;
			}
			break;


		case phase_read_post:
			ret = cherokee_connection_recv (conn, conn->post, &len);
			
			switch (ret) {
			case ret_ok:
				/* It's reading the POST info
				 */				
				if (conn->post->len >= conn->post_len) {
					break;
				}
				continue;
				
			case ret_error:
				/* Opss..
				 */
				purge_closed_connection (thd, conn);			       
				continue;
				
			case ret_eof:
				/* Finish..
				 */
				if (conn->post->len < conn->post_len) {
					purge_closed_connection (thd, conn);
					continue;
				}
				break;
				
			default:
				SHOULDNT_HAPPEN;
			}
			
			/* Turn the connection in write mode
			 */
			conn_set_mode (thd, conn, socket_writing);
			conn->phase = phase_setup_connection;
			break;


		case phase_reading_header: 

			/* Maybe the buffer has a request (previous pipelined)
			 */
			if (! cherokee_buffer_is_empty (conn->incoming_header)) {
				if (cherokee_connection_eoh (conn, 
							     conn->incoming_header, 
							     conn->incoming_header->len)) 
				{
					goto phase_reading_header_EXIT;
				}
			}
			       			
			/* Read from the client
			 */
			ret = cherokee_connection_recv (conn, conn->incoming_header, &len);
			switch (ret) {
			case ret_ok:
				break;
			case ret_eagain:
				continue;
			case ret_eof:
			case ret_error:
				purge_closed_connection (thd, conn);
				continue;
			default:
				SHOULDNT_HAPPEN;
			}

			/* Check security after read
			 */
			ret = cherokee_connection_reading_check (conn);
			if (ret != ret_ok) {
				conn_set_mode (thd, conn, socket_writing);
				conn->phase = phase_add_headers;
				continue;
			}

			/* May it already has the full header
			 */
			if (! cherokee_connection_eoh (conn, conn->incoming_header, len+4)) {
				continue;
			}

		phase_reading_header_EXIT:
			conn->phase = phase_processing_header;

		case phase_processing_header:

			/* Get the request
			 */
			ret = cherokee_connection_get_request (conn);
			if (ret != ret_ok) {
				conn_set_mode (thd, conn, socket_writing);
				conn->phase = phase_add_headers;
				continue;
			}

			/* If it's a POST we've to read more data
			 */
			if (HDR_METHOD(conn->header) == http_post) {
				if (conn->post->len < conn->post_len) {
					conn_set_mode (thd, conn, socket_reading);
					conn->phase = phase_read_post;
					continue;
				}
			}

			conn->phase = phase_setup_connection;

		case phase_setup_connection: {

			cherokee_handler_table_entry_t *plugin_entry = NULL;

			/* Turn the connection in write mode
			 */
			conn_set_mode (thd, conn, socket_writing);
			
			/* Get the virtual host and maybe set a new vserver reference
			 */
			if (conn->host->len > 0) {
				cherokee_virtual_server_t *vserver;

				vserver = cherokee_table_get_val (srv->vservers_ref, conn->host->buf);
				if (vserver != NULL) {
					conn->vserver = vserver;
				}
			}

			/* Set the logger of the connection
			 */
			conn->logger_ref = CONN_VSRV(conn)->logger;

			/* Build local request string
			 */
			if ((srv->userdir != NULL) && 
			    (srv->userdir_handler != NULL) && 
			    (cherokee_connection_is_userdir (conn)))
			{
				plugin_entry = srv->userdir_handler;
				ret = cherokee_connection_build_local_directory_userdir (conn, srv->userdir);
				if (ret != ret_ok) {
					conn->phase = phase_add_headers;
					continue;
				}

			} else {
				ret = cherokee_connection_get_plugin_entry (conn, CONN_VSRV(conn)->plugins, &plugin_entry);
				if (ret != ret_ok) {
					conn->phase = phase_add_headers;
					continue;
				}
				ret = cherokee_connection_build_local_directory (conn, CONN_VSRV(conn), plugin_entry);
			}
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}

			/* Check for IP validation
			 */
			ret = cherokee_connection_check_ip_validation (conn, plugin_entry);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}

			/* Check for authentication
			 */
			ret = cherokee_connection_check_authentication (conn, plugin_entry);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}
			
			ret = cherokee_connection_create_handler (conn, plugin_entry);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}
			
			/* Parse the rest of headers
			 */
			ret = cherokee_connection_parse_header (conn, srv->encoders);
			if (ret != ret_ok) {
				conn->phase = phase_add_headers;
				continue;
			}

			conn->phase = phase_init;			
		}
			
		case phase_init: 
			/* Server's "Keep-Alive" can be turned "Off"
			 */
			if (srv->keepalive == false) {
				conn->keepalive = 0;
			}

			/* Look for the request
			 */
			ret = cherokee_connection_open_request (conn);
			if (ret == ret_eagain) {
				break;
			}

			conn->phase = phase_add_headers;

		case phase_add_headers:

			/* If it is an error, maybe we will want to rewrite
			 * the response code to redirect the error page
			 */
			if (http_type_400(conn->error_code) &&
			    cherokee_buffer_is_empty(conn->redirect)) 
			{
				char *url = NULL;

				/* maybe has a custom ErrorPage redirection
				 */
				ret = cherokee_virtual_server_get_error_url (CONN_VSRV(conn), 
									     conn->error_code,
									     &url);

				if (ret == ret_ok) {
					cherokee_buffer_add (conn->redirect, url, strlen(url));
					conn->error_code = http_moved_temporarily;
				}
			}

			/* Build the header
			 */
			ret = cherokee_connection_build_header (conn);
			if (ret == ret_eagain) continue;
			
			/* If it has mmaped content, go to next stage
			 */		     
			if (conn->mmaped != NULL) {
				goto phase_send_headers_EXIT;
			}

			conn->phase = phase_send_headers;

		case phase_send_headers:

			/* Send headers to the client
			 */
			ret = cherokee_connection_send_header (conn);

			switch (ret) {
			case ret_eagain:
				continue;

			case ret_ok:
				if ((conn->header->method == http_head) ||
				    (conn->header->method == http_options)) 
				{
					maybe_purge_closed_connection (thd, conn);
					continue;

				} else if (http_type_300(conn->error_code)) 
				{
					maybe_purge_closed_connection (thd, conn);
					continue;

				} else if (http_type_400(conn->error_code) ||
					   http_type_500(conn->error_code)) 
				{
					char *filename = NULL;

					/* Has a custom ErrorPage
					 */
					ret = cherokee_virtual_server_get_error_page (CONN_VSRV(conn), 
										      conn->error_code,
										      &filename);
					if (ret == ret_ok) {
						cherokee_connection_send_response_page_file (conn, filename);
					} else {
						cherokee_connection_send_response_page_hardcoded (conn);
					}

					/* Log the error
					 */
					if (conn->logger_ref != NULL) {
						cherokee_logger_write_error (conn->logger_ref, conn);
					}
					
					purge_closed_connection (thd, conn);
					continue;
				}
				break;

			case ret_eof:
			case ret_error:
				purge_closed_connection (thd, conn);
				continue;

			default:
				SHOULDNT_HAPPEN; 
			}

			/* Maybe log the connection
			 */
			cherokee_connection_log_or_delay (conn);			

		phase_send_headers_EXIT:
			conn->phase = phase_steping;
			

		case phase_steping: {
			ret_t sent_ret;
			
			/* Special case: 
			 * If the content is mmap()ed, it has to send the header +
			 * the file content and stop processing the connection.
			 */
			if (conn->mmaped != NULL) {
				sent_ret = cherokee_connection_send_header_and_mmaped (conn);
				switch (sent_ret) {
				case ret_eagain:
					continue;
				case ret_eof:
				case ret_error:
					purge_closed_connection (thd, conn);
					continue;
				default:
					maybe_purge_closed_connection (thd, conn);
					continue;
				}
			}

			/* Handler step: read or make new data to send
			 */
			ret = cherokee_connection_step (conn);

			switch (ret) {
			case ret_eof_have_data:
				sent_ret = cherokee_connection_send (conn);

				switch (sent_ret) {
				case ret_eof:
				case ret_error:
					purge_closed_connection (thd, conn);
					continue;
				default:	
					maybe_purge_closed_connection (thd, conn);
					continue;
				}
				break;

			case ret_ok:
				/* We've data, so..
				 */
				sent_ret = cherokee_connection_send (conn);

				if ((sent_ret == ret_eof) ||
				    (sent_ret == ret_error)) 
				{
					purge_closed_connection (thd, conn);					
				}
				break;

			case ret_ok_and_sent:
				break;

			case ret_eof:
				maybe_purge_closed_connection (thd, conn);
				break;
			case ret_error:
				purge_closed_connection (thd, conn);
				break;
			case ret_eagain:
				break;
			default:
				PRINT_ERROR ("Error code unknown: %d\n", ret);
				SHOULDNT_HAPPEN; 
			}
			
		} break;

		default:
 			SHOULDNT_HAPPEN;
		}
	} /* list */	

	return ret_ok;
}


ret_t 
cherokee_thread_free (cherokee_thread_t *thd)
{
	list_t *i, *tmp;

	cherokee_fdpoll_free (thd->fdpoll);
	thd->fdpoll = NULL;

	/* Free the connection
	 */
	list_for_each_safe (i, tmp, &thd->active_list) {
		cherokee_connection_free (CONN(i));
	}

	free (thd);

	return ret_ok;
}


static int
__accept_from_server (cherokee_thread_t *thd, int srv_socket, cherokee_socket_type_t tls)
{
	ret_t                  ret;
	cherokee_connection_t *new_conn;

        /* Return if there're no new connections
         */
        if (cherokee_fdpoll_check (thd->fdpoll, srv_socket, 0) == 0) {
                return 0;
        }

        /* Get a new connection object
         */
	ret = cherokee_thread_get_new_connection (thd, &new_conn);
	if (ret < ret_ok) return 0;
	
	/* Accept the connection
	 */
	ret = cherokee_socket_accept (CONN_SOCK(new_conn), srv_socket);
	if (ret < ret_ok) goto error;
	
	/* Set the connection time-out
	 */
	new_conn->timeout = THREAD_SRV(thd)->bogo_now + THREAD_SRV(thd)->timeout;

	/* May active the TLS support
         */
        if (tls == TLS) {
                new_conn->phase = phase_tls_handshake;
        }
	
	/* Add the new connection to the thread
	 */
	ret = cherokee_thread_add_connection (thd, new_conn);
	if (ret < ret_ok) {
		goto error;
	}

	return 1;

error:
	cherokee_thread_set_reusable_connection (thd, new_conn);
	return 0;
}


static int
__should_accept_more_from_server (cherokee_thread_t *thd, int re)
{
	const uint32_t recalculate_steps = 10;

	/* If it is full, do not accept more!
	 */
	if (cherokee_fdpoll_is_full(thd->fdpoll)) {
		return 0;
	}

	/* Got new connection
	 */
	if (re > 0) {
		thd->accept.continuous++;

		if (thd->accept.recalculate <= 0) {
			thd->accept.continuous_max = thd->accept.continuous;
			return 1;
		}

		if (thd->accept.continuous > thd->accept.continuous_max) {
			thd->accept.continuous_max = thd->accept.continuous;
			thd->accept.recalculate--;
			return 0;
		}
		
		return 1;
	}

	/* Failed to get a new connection
	 */
	thd->accept.continuous  = 0;
	thd->accept.recalculate = recalculate_steps;
	return 0;
}


ret_t 
cherokee_thread_step_SINGLE_THREAD (cherokee_thread_t *thd, cherokee_boolean_t dont_block)
{
	ret_t ret;
	int   re;
	cherokee_server_t *srv           = THREAD_SRV(thd);
	int                fdwatch_msecs = srv->fdwatch_msecs;

	cherokee_fdpoll_reset (thd->fdpoll, srv->socket);

	/* If the thread is full of connections, it should not
	 * get new connections
	 */
	if (cherokee_fdpoll_is_full (thd->fdpoll)) {
		goto out;
	}

	/* If thread has pending connections, it should do a 
	 * faster 'watch' as possible
	 */
	if (thd->pending_conns_num > 0) {
		fdwatch_msecs          = 0;
		thd->pending_conns_num = 0;
	}
	
	re = cherokee_fdpoll_watch (thd->fdpoll, fdwatch_msecs);
	if (re <= 0) return ret_ok;

	do {
		re = __accept_from_server (thd, srv->socket, non_TLS);
	} while (__should_accept_more_from_server (thd, re));

	if (srv->tls_enabled != -1) {
		do {
			re = __accept_from_server (thd, srv->socket_tls, TLS);
		} while (__should_accept_more_from_server (thd, re));
	}
out:
	return process_active_connections (thd);
}


#ifdef HAVE_PTHREAD

static ret_t
step_MULTI_THREAD_block (cherokee_thread_t *thd, int socket, pthread_mutex_t *mutex, int fdwatch_msecs)
{
	int   re;
	ret_t ret;

	CHEROKEE_MUTEX_LOCK (mutex);
	
	ret = cherokee_fdpoll_add (thd->fdpoll, socket, 0);
	if (ret < ret_ok) {
		CHEROKEE_MUTEX_UNLOCK (mutex);
		return ret_error;
	}
	cherokee_fdpoll_reset (thd->fdpoll, socket);
	cherokee_fdpoll_watch (thd->fdpoll, fdwatch_msecs);
	
	do {
		re = __accept_from_server (thd, socket, non_TLS);
	} while (__should_accept_more_from_server (thd, re));
	
	cherokee_fdpoll_del (thd->fdpoll, socket);
	CHEROKEE_MUTEX_UNLOCK (mutex);
	
	return ret_ok;
}


static ret_t
step_MULTI_THREAD_nonblock (cherokee_thread_t *thd, int socket, pthread_mutex_t *mutex, int fdwatch_msecs)
{
	int   re;
	ret_t ret;
	int   unlocked;
	
	/* Try to lock
	 */
	unlocked = CHEROKEE_MUTEX_TRY_LOCK (mutex);
	if (unlocked) {
		cherokee_fdpoll_watch (thd->fdpoll, fdwatch_msecs);
		return ret_ok;
	}
	
	/* Now, it owns the socket..
	 */
	ret = cherokee_fdpoll_add (thd->fdpoll, socket, 0);
	if (ret < ret_ok) {
		CHEROKEE_MUTEX_UNLOCK (mutex);
		return ret_error;
	}
	cherokee_fdpoll_reset (thd->fdpoll, socket);
	cherokee_fdpoll_watch (thd->fdpoll, fdwatch_msecs);
	
	do {
		re = __accept_from_server (thd, socket, non_TLS);
	} while (__should_accept_more_from_server (thd, re));
	
	cherokee_fdpoll_del (thd->fdpoll, socket);
	CHEROKEE_MUTEX_UNLOCK (mutex);
	
	return ret_ok;
}

# ifdef HAVE_TLS

static ret_t
step_MULTI_THREAD_TLS_nonblock (cherokee_thread_t *thd, int fdwatch_msecs, 
				int socket,     pthread_mutex_t *mutex, 
				int socket_tls, pthread_mutex_t *mutex_tls)
{
	int   re;
	ret_t ret;
	int   unlock     = 0;
	int   unlock_tls = 0;

	/* Try to lock the mutex
	 */
	unlock = CHEROKEE_MUTEX_TRY_LOCK (mutex);
	if (!unlock) {
		ret = cherokee_fdpoll_add (thd->fdpoll, socket, 0);
		if (ret < ret_ok) {
			goto error;
		}
		cherokee_fdpoll_reset (thd->fdpoll, socket);
	}

	/* Try to lock the TLS mutex
	 */
	unlock_tls = CHEROKEE_MUTEX_TRY_LOCK (mutex_tls);
	if (!unlock_tls) {
		ret = cherokee_fdpoll_add (thd->fdpoll, socket_tls, 0);
		if (ret < ret_ok) {
			goto error;
		}
		cherokee_fdpoll_reset (thd->fdpoll, socket_tls);
	}

	/* Inspect the fds and maybe sleep
	 */
	cherokee_fdpoll_watch (thd->fdpoll, fdwatch_msecs);
		
	/* Restore..
	 */
	if (!unlock) {
		do {
			re = __accept_from_server (thd, socket, non_TLS);
		} while (__should_accept_more_from_server (thd, re));
		
		cherokee_fdpoll_del (thd->fdpoll, socket);
		CHEROKEE_MUTEX_UNLOCK (mutex);
	}
	
	if (!unlock_tls) {
		do {
			re = __accept_from_server (thd, socket_tls, TLS);
		} while (__should_accept_more_from_server (thd, re));
		
		cherokee_fdpoll_del (thd->fdpoll, socket_tls);
		CHEROKEE_MUTEX_UNLOCK (mutex_tls);
	}
	
	return ret_ok;


error:
	if (!unlock)     CHEROKEE_MUTEX_UNLOCK (mutex);
	if (!unlock_tls) CHEROKEE_MUTEX_UNLOCK (mutex_tls);

	return ret_error;
}

static ret_t
step_MULTI_THREAD_TLS_block (cherokee_thread_t *thd, int fdwatch_msecs, 
			     int socket,     pthread_mutex_t *mutex, 
			     int socket_tls, pthread_mutex_t *mutex_tls)
{
	int                     re;
	ret_t                   ret;
	int                     unlock2 = 0;
	int                     socket1;
	int                     socket2;
	pthread_mutex_t        *mutex1;
	pthread_mutex_t        *mutex2;
	cherokee_socket_type_t  type1;
	cherokee_socket_type_t  type2;

	if (thd->thread_pref == thread_tls_normal) {
		socket1 = socket;
		mutex1  = mutex;
		type1   = non_TLS;

		socket2 = socket_tls;
		mutex2  = mutex_tls;
		type2   = TLS;
	} else {
		socket1 = socket_tls;
		mutex1  = mutex_tls;
		type1   = TLS;

		socket2 = socket;
		mutex2  = mutex;
		type2   = non_TLS;
	}

	/* Lock the main mutex
	 */
	CHEROKEE_MUTEX_LOCK (mutex1);

	ret = cherokee_fdpoll_add (thd->fdpoll, socket1, 0);
	if (ret < ret_ok) {
		CHEROKEE_MUTEX_UNLOCK (mutex1);
		return ret_error;
	}
	cherokee_fdpoll_reset (thd->fdpoll, socket1);


	/* Try to lock the optional groups
	 */
#if 0
	unlock2 = CHEROKEE_MUTEX_TRY_LOCK (mutex2);
	if (!unlock2) {
		ret = cherokee_fdpoll_add (thd->fdpoll, socket2, 0);
		if (ret < ret_ok) {
			CHEROKEE_MUTEX_UNLOCK (mutex1);
			CHEROKEE_MUTEX_UNLOCK (mutex2);
			return ret_error;
		}
		cherokee_fdpoll_reset (thd->fdpoll, socket2);
	}
#endif

	/* Inspect the fds and get new connections
	 */
	cherokee_fdpoll_watch (thd->fdpoll, fdwatch_msecs);
		
	do {
		re = __accept_from_server (thd, socket1, type1);
	} while (__should_accept_more_from_server (thd, re));

	/* Unlock the mail lock
	 */
	cherokee_fdpoll_del (thd->fdpoll, socket1);
	CHEROKEE_MUTEX_UNLOCK (mutex1);

	/* Maybe work with the optional socket
	 */
#if 0
	if (!unlock2) {
		do {
			re = __accept_from_server (thd, socket2, type2);
		} while (__should_accept_more_from_server (thd, re));
		
		cherokee_fdpoll_del (thd->fdpoll, socket2);
		CHEROKEE_MUTEX_UNLOCK (mutex2);
	}
#endif
	
	return ret_ok;
}
# endif /* HAVE_TLS */

ret_t 
cherokee_thread_step_MULTI_THREAD (cherokee_thread_t *thd, cherokee_boolean_t dont_block)
{
	ret_t              ret;
	cherokee_server_t *srv           = THREAD_SRV(thd);
	int                fdwatch_msecs = srv->fdwatch_msecs;

	/* If the thread is full of connections, it should not
	 * get new connections
	 */
	if (cherokee_fdpoll_is_full (thd->fdpoll)) {
		goto out;
	}

	/* If thread has pending connections, it should do a 
	 * faster 'watch' as possible
	 */
	if (thd->pending_conns_num > 0) {
		fdwatch_msecs          = 0;
		thd->pending_conns_num = 0;
	}

	/* Try to get new connections from http
	 */
#ifdef HAVE_TLS
	if (srv->tls_enabled) {
		if ((thd->active_list_num == 0) && (!dont_block)) {
			step_MULTI_THREAD_TLS_block (thd, fdwatch_msecs, 
						     srv->socket, &THREAD_SRV(thd)->accept_mutex, 	
						     srv->socket_tls, &THREAD_SRV(thd)->accept_tls_mutex);
		} else {
			step_MULTI_THREAD_TLS_nonblock (thd, fdwatch_msecs, 
							srv->socket, &THREAD_SRV(thd)->accept_mutex, 	
							srv->socket_tls, &THREAD_SRV(thd)->accept_tls_mutex);
		}
		
		goto out;
	}
#endif	

	if ((thd->active_list_num == 0) && (!dont_block)) {
		step_MULTI_THREAD_block (thd, srv->socket, &THREAD_SRV(thd)->accept_mutex, fdwatch_msecs);
	} else {
		step_MULTI_THREAD_nonblock (thd, srv->socket, &THREAD_SRV(thd)->accept_mutex, fdwatch_msecs);
	}
	
out:
	return process_active_connections (thd);
}

#endif /* HAVE_PTHREAD */



ret_t 
cherokee_thread_get_new_connection (cherokee_thread_t *thd, cherokee_connection_t **conn)
{
	cherokee_connection_t *new_connection;
	cherokee_server_t     *server;

	server = thd->server;

	if (list_empty (&thd->reuse_list)) {
		ret_t ret;

		/* Create new connection object
		 */
		ret = cherokee_connection_new (&new_connection);
		if (ret < ret_ok) return ret;
	} else {
		/* Reuse an old one
		 */
		new_connection = CONN(thd->reuse_list.prev);
		list_del ((list_t *)new_connection);
		INIT_LIST_HEAD((list_t *)new_connection);
	}

	/* Set the basic information to the connection
	 */
	new_connection->thread  = thd;
	new_connection->server  = server;
	new_connection->vserver = server->vserver_default;

	/* Set the keepalive from server
	 */
	new_connection->keepalive = server->keepalive_max;

	*conn = new_connection;
	return ret_ok;
}


ret_t 
cherokee_thread_set_reusable_connection (cherokee_thread_t *thread, cherokee_connection_t *conn)
{
	/* Disable keepalive in the connection
	 */
	conn->keepalive = 0;
	
	/* Add it to the reusable connection list
	 */
	list_add ((list_t *)conn, &thread->reuse_list);
}


ret_t 
cherokee_thread_add_connection (cherokee_thread_t *thd, cherokee_connection_t  *conn)
{
	ret_t ret;

	ret = cherokee_fdpoll_add (thd->fdpoll, SOCKET_FD(conn->socket), 0);
	if (ret < ret_ok) return ret;

	conn_set_mode (thd, conn, socket_reading);
	add_connection (thd, conn);

	return ret_ok;
}


int   
cherokee_thread_connection_num (cherokee_thread_t *thd) 
{
	return thd->active_list_num;
}


ret_t 
cherokee_thread_close_all_connections (cherokee_thread_t *thd)
{
	list_t *i, *tmp;
	list_for_each_safe (i, tmp, &thd->active_list) {
		purge_closed_connection (thd, CONN(i));
	}

	return ret_ok;
}
