/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef __CHEROKEE_HANDLER_H__
#define __CHEROKEE_HANDLER_H__

#include "common.h"

#include "module.h"
#include "buffer.h"
#include "table.h"
#include "http.h"


typedef enum {
	hstat_unset,
	hstat_sending,
	hstat_forbidden,
	hstat_file_not_found,
	hstat_error
} cherokee_handler_status_t;

typedef enum {
	hsupport_nothing         = 0,
	hsupport_length          = 1,         /* Know the length. Eg: for keep-alive     */
	hsupport_range           = 1 << 1,    /* Can handle "Range: bytes=" requests     */
	hsupport_complex_headers = 1 << 2     /* Return headers that should be processed */
} cherokee_handler_support_t;


/* Callback function definitions
 */
typedef ret_t (* handler_func_new_t)         (void **handler, void *cnt, cherokee_table_t *properties);
typedef ret_t (* handler_func_init_t)        (void  *handler);
typedef ret_t (* handler_func_free_t)        (void  *handler);
typedef ret_t (* handler_func_step_t)        (void  *handler, cherokee_buffer_t *buffer);
typedef ret_t (* handler_func_add_headers_t) (void  *handler, cherokee_buffer_t *buffer);


typedef struct {
	cherokee_module_t module;
	
	/* Pure virtual methods
	 */
	handler_func_step_t         step;
	handler_func_add_headers_t  add_headers;

	/* Properties
	 */
	void *connection;
	cherokee_handler_support_t support;
	   
} cherokee_handler_t;

#define HANDLER(x)                         ((cherokee_handler_t *)(x))
#define HANDLER_CONN(h)                    (CONN(HANDLER(h)->connection))
#define HANDLER_SRV(h)                     (CONN_SRV(HANDLER_CONN(h)))
#define HANDLER_SUPPORT_RANGE(h)           (HANDLER(h)->support & hsupport_range)
#define HANDLER_SUPPORT_LENGTH(h)          (HANDLER(h)->support & hsupport_length)
#define HANDLER_SUPPORT_COMPLEX_HEADERS(h) (HANDLER(h)->support & hsupport_complex_headers)

ret_t cherokee_handler_init_base   (cherokee_handler_t  *hdl, void *conn);
ret_t cherokee_handler_free_base   (cherokee_handler_t  *hdl);

ret_t cherokee_handler_init        (cherokee_handler_t  *hdl);
ret_t cherokee_handler_free        (cherokee_handler_t  *hdl);
ret_t cherokee_handler_step        (cherokee_handler_t  *hdl, cherokee_buffer_t *buffer);
ret_t cherokee_handler_add_headers (cherokee_handler_t  *hdl, cherokee_buffer_t *buffer);

#endif /* __CHEROKEE_HANDLER_H__ */
