/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#ifdef HAVE_SYSLOG_H
# include <syslog.h>
#endif

#include <sys/time.h>

#if defined (HAVE_SYS_RESOURCE_H)
# include <sys/resource.h>
#elif defined (HAVE_RESOURCE_H)
# include <unistd.h>
#endif

#include "log.h"


char *cherokee_version = PACKAGE_VERSION;

int
hexit (char c)
{
	if ( c >= '0' && c <= '9' )
		return c - '0';
	if ( c >= 'a' && c <= 'f' )
		return c - 'a' + 10;
	if ( c >= 'A' && c <= 'F' )
		return c - 'A' + 10;

	/* shouldn't happen, we're guarded by isxdigit() */
	return 0;           
}


void 
PRINT_ERROR (const char *format, ...)
{
	va_list arg_list;
	CHEROKEE_TEMP(tmp, 2048);
	
	va_start(arg_list, format);
	vsnprintf (tmp, tmp_size, format, arg_list);
	va_end(arg_list);

	fprintf (stderr, "%s", tmp);
	syslog (LOG_ERR, "%s", tmp);
}


void 
PRINT_DEBUG (const char *format, ...)
{
	va_list arg_list;
	
	va_start(arg_list, format);
	vfprintf(stdout, format, arg_list);
	va_end(arg_list);

	fflush(stdout); 
}


/* This function is licenced under:
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  
 * All rights reserved.
 */
char *
apr_strfsize(unsigned long long size, char *buf)
{
    const char ord[] = "KMGTPE";
    const char *o = ord;
    int remain;

    if (size < 0) {
        return strcpy(buf, "  - ");
    }
    if (size < 973) {
        sprintf(buf, "%3d ", (int) size);
        return buf;
    }
    do {
        remain = (int)(size & 1023);
        size >>= 10;
        if (size >= 973) {
            ++o;
            continue;
        }
        if (size < 9 || (size == 9 && remain < 973)) {
            if ((remain = ((remain * 5) + 256) / 512) >= 10)
                ++size, remain = 0;
            sprintf(buf, "%d.%d%c", (int) size, remain, *o);
            return buf;
        }
        if (remain >= 512)
            ++size;
        sprintf(buf, "%3d%c", (int) size, *o);
        return buf;
    } while (1);
}



char *
min_string_not_null (char *s1, char *s2)
{
	if ((s1 == NULL) && 
	    (s2 == NULL)) return NULL;

	if ((s1 != NULL) && 
	    (s2 == NULL)) return s1;

	if ((s2 != NULL) && 
	    (s1 == NULL)) return s2;
	
	return (s1<s2) ? s1 : s2;
}


ret_t 
get_system_fd_num_limit (int *limit)
{
	int nfiles;

#ifdef USE_SELECT
	/* If we use select(), then we must limit ourselves to FD_SETSIZE. 
	 */
	*limit = FD_SETSIZE;
	return ret_ok;
#endif
	
	/* Figure out how many fd's we can have. 
	 */
	nfiles = getdtablesize();
	
#ifdef RLIMIT_NOFILE
	{
		int           re;
		struct rlimit rl;

		/* If we have getrlimit(), use that, and attempt to raise the limit. 
		 */
		re = getrlimit (RLIMIT_NOFILE, &rl);
		if (re == 0) {
			if (rl.rlim_max == RLIM_INFINITY) {
				return ret_ok;
			}
						
			nfiles = MAX (nfiles, rl.rlim_max);
		}
	}
#endif
	   
	*limit = nfiles;
	return ret_ok;
}


ret_t
set_system_fd_num_limit (int limit)
{
	int           re;
	struct rlimit rl;

	rl.rlim_cur = limit;
	rl.rlim_max = limit;

	re = setrlimit (RLIMIT_NOFILE, &rl);
	if (re != 0) {
		return ret_error;
	}

	return ret_ok;
}


#ifndef HAVE_STRSEP
char *
strsep (char** str, const char* delims)
{
    char* token;

    if (*str == NULL) {
        /* No more tokens 
	 */
        return NULL;
    }

    token = *str;
    while (**str!='\0') {
        if (strchr(delims,**str)!=NULL) {
            **str='\0';
            (*str)++;
            return token;
        }
        (*str)++;
    }

    /* There is no other token 
     */
    *str=NULL;
    return token;
}
#endif
