/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#include <errno.h>

#include "handler_mono.h"
#include "connection.h"

cherokee_module_info_t cherokee_mono_info = {
	cherokee_handler,            /* type     */
	cherokee_handler_mono_new    /* new func */
};


enum Cmd {
        FIRST_COMMAND,
        GET_REQUEST_LINE = 0,
        SEND_FROM_MEMORY,
        GET_PATH_INFO,
        GET_SERVER_VARIABLE,
        GET_PATH_TRANSLATED,
        GET_SERVER_PORT,
        SET_RESPONSE_HEADER,
        GET_FILENAME,
        GET_REMOTE_ADDRESS,
        GET_LOCAL_ADDRESS,
        GET_REMOTE_PORT,
        GET_LOCAL_PORT,
        GET_REMOTE_NAME,
        FLUSH,
        CLOSE,
        SHOULD_CLIENT_BLOCK,
        SETUP_CLIENT_BLOCK,
        GET_CLIENT_BLOCK,
        SET_STATUS_LINE,
        SET_STATUS_CODE,
        DECLINE_REQUEST,
        LAST_COMMAND
};

typedef enum {
	command_ok,
	command_declined,
	command_internal_server_error
} cherokee_command_result_t;


ret_t 
cherokee_handler_mono_new (cherokee_handler_t **hdl, cherokee_connection_t *cnt, cherokee_table_t *properties)
{
	CHEROKEE_NEW_STRUCT (n,handler_mono);
	
	/* Init the base class object
	 */
	cherokee_handler_init_base(HANDLER(n), cnt);
	
	HANDLER(n)->support = hsupport_nothing;
	
	HANDLER(n)->init        = (handler_func_init_t) cherokee_handler_mono_init;
	HANDLER(n)->free        = (handler_func_free_t) cherokee_handler_mono_free;
	HANDLER(n)->step        = (handler_func_step_t) cherokee_handler_mono_step;
	HANDLER(n)->add_headers = (handler_func_add_headers_t) cherokee_handler_mono_add_headers;

	/* Init
	 */
	n->socket      = -1;
	n->socket_path = NULL;
	n->connection  = cnt;

	/* Parse the arguments
	 */
//	cherokee_connection_parse_args (cnt);

	/* Get properties
	 */
	if (properties) {
		n->socket_path = cherokee_table_get_val (properties, "socket");
	}

	*hdl = HANDLER(n);	
	return ret_ok;
}


/* Reading helper functions
 */

static ret_t
buffer_read_string (int fd, cherokee_buffer_t *buf) 
{
	int len, count;

	if (read (fd, &len, sizeof (int)) != sizeof (int)) {
		return ret_error;
	}

	cherokee_buffer_make_empty (buf);
	cherokee_buffer_ensure_size (buf, len);

	count = len;
	while (count > 0) {
		count -= read (fd, buf->buf + len - count, count);
	}
	
	return ret_ok;
}


static ret_t
buffer_send_string (int fd, cherokee_buffer_t *buf) 
{
	int re, count;

	count = 0;
	while (count < buf->len) {
		re = write (fd, buf->buf + count, buf->len - count);
		if (re < 0) return ret_error;
		
		count += re;
	}

	return ret_ok;
}





static int
read_data (int fd, void *ptr, int size)
{
     return (read (fd, ptr, size) == size) ? size : -1;
}


/* Signal sending functions
 */

static int
write_ok (int fd)
{
	ssize_t re;
	uint8_t i = 0;
	
        re = write (fd, &i, 1);
	return (re == 1) ? ret_ok : ret_error;
}

static int
write_err (int fd)
{
	ssize_t re;
        int i = -1;

        re = write (fd, &i, 1);
	return (re == 1) ? ret_ok : ret_error;
}


/* Writing helper functions
 */

static int
write_data_string_no_prefix (int fd, const char *str)
{
        int l;
	
        l = (str == NULL) ? 0 : strlen(str);
        if (write(fd, &l, sizeof (int)) != sizeof (int)) {
                return -1;
	}

        if (l == 0) {
                return 0;
	}

        return write (fd, str, l);
}


static int
write_data_string (int fd, const char *str)
{
        if (write_ok (fd) == -1) {
                return;
	}

        return write_data_string_no_prefix (fd, str);
}

static int
write_data_no_prefix (int fd, const void *str, int size)
{
        return write (fd, str, size);
}


static int
do_command (cherokee_handler_mono_t *hdl, int command, cherokee_command_result_t *result)
{
	int size;
	char *str;
	char *str2;
	int i;
	int status;

	ret_t ret;
	int   fd;
	CHEROKEE_NEW(tmp,buffer);

	fd = hdl->socket;

	*result = ret_ok;
	switch (command) {
	case SEND_FROM_MEMORY:
		printf ("SEND_FROM_MEMORY\n");

		ret = buffer_read_string (fd, tmp);
		if (ret < ret_ok) break;

		ret = buffer_send_string (fd, tmp);
		if (ret < ret_ok) break;

		ret = write_ok (fd);
		break;

	case GET_PATH_INFO:
//		status = write_data_string (fd, r->path_info);
		printf ("GET_PATH_INFO\n");
		break;

	case GET_SERVER_VARIABLE:
		printf ("GET_SERVER_VARIABLE\n");
/*		if (read_data_string (fd, &str, NULL) == NULL) {
			status = -1;
			break;
		}
		str = (char *) request_get_server_variable (r, str);
		status = write_data_string (fd, str);
*/
		break;

	case GET_PATH_TRANSLATED:
		printf ("GET_PATH_TRANSLATED\n");
/*		str = request_get_path_translated (r);
		status = write_data_string (fd, str);
*/
		break;

	case GET_SERVER_PORT:
		printf ("GET_SERVER_PORT\n");
/*		i = request_get_server_port (r);
		status = write_data (fd, &i, sizeof (int));
*/
		break;

	case SET_RESPONSE_HEADER:
		printf ("SET_RESPONSE_HEADER\n");
/*		if (read_data_string (fd, &str, NULL) == NULL) {
			status = -1;
			break;
		}
		if (read_data_string (fd, &str2, NULL) == NULL) {
			status = -1;
			break;
		}
		set_response_header (r, str, str2);
		status = write_ok (fd);
*/
		break;
	case GET_FILENAME:
		printf ("GET_FILENAME\n");
//		status = write_data_string (fd, r->filename);
		break;

	case GET_REMOTE_ADDRESS:
		printf ("GET_REMOTE_ADDRESS\n");
//		status = write_data_string (fd, r->connection->remote_ip);
		break;

	case GET_LOCAL_ADDRESS:
		printf ("GET_LOCAL_ADDRESS\n");
//		status = write_data_string (fd, r->connection->local_ip);
		break;

	case GET_REMOTE_PORT:
		printf ("GET_REMOTE_PORT\n");
/*		i = connection_get_remote_port (r->connection);
		status = write_data (fd, &i, sizeof (int));
*/
		break;

	case GET_LOCAL_PORT:
		printf ("GET_LOCAL_PORT\n");
/*		i = connection_get_local_port (r);
		status = write_data (fd, &i, sizeof (int));
*/
		break;

	case GET_REMOTE_NAME:
		printf ("GET_REMOTE_NAME\n");
/*		str = (char *) connection_get_remote_name (r);
		status = write_data_string (fd, str);
*/
		break;

	case FLUSH:
		printf ("FLUSH\n");
/*		connection_flush (r);
		status = write_ok (fd);
*/
		break;

	case CLOSE:
		printf ("CLOSE\n");
//		status = write_ok (fd);
		goto error;

	case SHOULD_CLIENT_BLOCK:
		printf ("SHOULD_CLIENT_BLOCK\n");
//		size = ap_should_client_block (r);
//		status = write_data (fd, &size, sizeof (int));
		break;

	case SETUP_CLIENT_BLOCK:
		printf ("SETUP_CLIENT_BLOCK\n");
/*		if (setup_client_block (r) != APR_SUCCESS) {
			size = -1;
			status = write_data (fd, &size, sizeof (int));
			break;
		}

		size = 0;
		status = write_data (fd, &size, sizeof (int));
*/
		break;

	case GET_CLIENT_BLOCK:
		printf ("GET_CLIENT_BLOCK\n");
/*		status = read_data (fd, &i, sizeof (int));
		if (status == -1)
			break;

		str = apr_pcalloc (i);
		i = ap_get_client_block (r, str, i);
		status = write_data (fd, &i, sizeof (int));
		status = write_data_no_prefix (fd, str, i);
*/
		break;

	case SET_STATUS_LINE:
		printf ("SET_STATUS_LINE\n");
/*		if (read_data_string (fd, &str, NULL) == NULL) {
			status = -1;
			break;
		}
		status = write_ok (fd);
		r->status_line = apr_pstrdup (str);
*/
		break;

	case SET_STATUS_CODE:
		printf ("SET_STATUS_CODE\n");
/*		status = read_data (fd, &i, sizeof (int));
		if (status == -1)
			break;

		r->status = i;
		status = write_ok (fd);
*/
		break;

	case DECLINE_REQUEST:
		printf ("DECLINE_REQUEST\n");
/*		status = write_ok (fd);
		*result = DECLINED;
*/		
		goto error;

	default:
/*		*result = HTTP_INTERNAL_SERVER_ERROR;
		write_err (fd);
*/
		SHOULDNT_HAPPEN;
		goto error;
	}

	if (status == -1) {
		*result = command_internal_server_error;
		goto error;
	}

	cherokee_buffer_free (tmp);
	return ret_ok;

error:
	cherokee_buffer_free (tmp);
	return ret_error;
}


static ret_t
send_headers_callback (cherokee_buffer_t *header, cherokee_header_t *content, 
		       cherokee_buffer_t *name, cherokee_buffer_t *cont, void *data)
{
	cherokee_handler_mono_t *hdl = HANDLER_MONO(data);

	printf ("name='%s' content='%s'\n", name->buf, cont->buf);
//	write_data_string_no_prefix

	return ret_ok;
}

static ret_t
send_headers (cherokee_handler_mono_t *hdl)
{
	int   num;
	ret_t ret;
	cherokee_connection_t *conn   = CONN(hdl->connection);
	cherokee_header_t     *header = CONN_HDR(conn);
	CHEROKEE_NEW2 (header_name, header_cont, buffer);

	ret = cherokee_header_get_number (header, &num);
	if (ret < ret_ok) goto error;

	write_data_no_prefix (hdl->socket, &num, sizeof (int));

	ret = cherokee_header_foreach (header, send_headers_callback, header_name, header_cont, hdl);
	if (ret < ret_ok) goto error;
	

	cherokee_buffer_free (header_name);
	cherokee_buffer_free (header_cont);       
	return ret_ok;

error:
	cherokee_buffer_free (header_name);
	cherokee_buffer_free (header_cont);
	return ret_error;
}


static ret_t
init_protocol (cherokee_handler_mono_t *hdl)
{
	ret_t                  ret;
	ssize_t                readed;
	cherokee_buffer_t     *tmp;
	char                  *tmp2;
	cherokee_connection_t *conn   = CONN(hdl->connection);
	cherokee_header_t     *header = CONN_HDR(conn);


	/* Send the method: GET, POST, ..
	 */
	cherokee_buffer_new (&tmp);
	cherokee_header_copy_method (header, tmp);
        if (write_data_string_no_prefix (hdl->socket, tmp->buf) <= 0) {
		goto error;
	}

	/* Send the request
	 */
        if (write_data_string_no_prefix (hdl->socket, conn->request->buf) <= 0) {	
		goto error;
	}

	/* Send the arguments
	 */
	if (! cherokee_buffer_is_empty (conn->query_string)) {
		tmp2 = conn->query_string->buf;
	} else { 
		tmp2 = "";
	}

	if (write_data_string_no_prefix (hdl->socket, tmp2) < 0) {
		goto error;
	}

	/* The protocol version: HTTP/1.1
	 */
	cherokee_buffer_make_empty (tmp);
	cherokee_header_copy_version (header, tmp);
        if (write_data_string_no_prefix (hdl->socket, tmp->buf) <= 0) {
		goto error;
	}

	/* Send headers
	 */
	ret = send_headers (hdl);
	if (ret < ret_ok) return ret;

	/* Read the response code
	 */
	do {
		int                       command;
		cherokee_command_result_t status;

		readed = read (hdl->socket, &command, sizeof(int));
		if (readed > 0) {
			ret = do_command (hdl, command, &status);
		}
	} while ((readed > 0) && (ret == ret_ok));


	cherokee_buffer_free (tmp);
	return ret_ok;

error:
	cherokee_buffer_free (tmp);	
	return ret_error;
}


ret_t 
cherokee_handler_mono_init (cherokee_handler_mono_t *hdl)
{
	int                    re;
	cherokee_connection_t *conn = CONN(hdl->connection);

	/* Create the socket
	 */
	hdl->socket = socket (PF_UNIX, SOCK_STREAM, 0);
	if (hdl->socket < 0) {
		PRINT_ERROR ("Can not create the socket to xsp\n");
		return ret_error;
	}
	
	/* Checks for connect
	 */
	if (hdl->socket_path == NULL) {
		PRINT_ERROR ("You have to provide the xsp socket path (\"socket\" property)\n");
		return ret_error;
	}
	
	hdl->address.sun_family = PF_UNIX;
        memcpy (hdl->address.sun_path, hdl->socket_path, strlen(hdl->socket_path)+1);

	/* Connect!
	 */
        re = connect (hdl->socket, (struct sockaddr *)&hdl->address, sizeof(struct sockaddr_un));
	if (re < 0) {
		PRINT_ERROR ("Can not connect to XSP server (%s): %s\n", hdl->socket_path, strerror(errno));

                close (hdl->socket);
		hdl->socket = -1;

		conn->error_code = http_internal_error;
                return ret_error;
        }

	return init_protocol (hdl);
}


ret_t 
cherokee_handler_mono_free (cherokee_handler_mono_t *hdl)
{
	if (hdl->socket != -1) {
		close (hdl->socket);
		hdl->socket = -1;
	}

	free (hdl);
	return ret_ok;
}


ret_t 
cherokee_handler_mono_step (cherokee_handler_mono_t *hdl, cherokee_buffer_t *buffer)
{
	   return ret_ok;
}


ret_t 
cherokee_handler_mono_add_headers (cherokee_handler_mono_t *hdl, cherokee_buffer_t *buffer)
{
	   return ret_ok;
}


/*   Library init function
 */
static int _mono_is_init = 0;

void 
mono_init ()
{
	/* Init flag
	 */
	if (_mono_is_init) {
		return;
	}
	_mono_is_init = 1;

	/* Init the socket to xsp
	 */
	
}
