/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#include <time.h>
#include <stdio.h>
#include <stdlib.h>

#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "bench.h"


/* Exit codes
 */
#define ERROR_OK      0
#define ERROR_NEW     1
#define ERROR_INIT    2
#define ERROR_RUN     3
#define ERROR_FREE    4
#define ERROR_READING 5
#define ERROR_URL     6


static void
print_help (void)
{
	printf ("Cherokee Benchmarker %s\n"
		"Usage: cherokee-bench [options] URL\n\n"
		"  -h,  --help                   Print this help\n"
		"  -V,  --version                Print version and exit\n"
		"  -t,  --threads                Set threads number\n"
		"  -P,  --pipeline               Set pipelining requests per connection\n"
		"  -c,  --concurrency            Set concurrency per thread\n"
		"  -n,  --number                 Set number of connections\n"
		"  -u,  --urls=FILE              Read URLs from file\n"
		"  -K,  --not-keepalive          Don't use keepalive\n"
		"\n"
		"Report bugs to alvaro@gnu.org\n", VERSION);
}

static void
print_usage (void)
{
	printf ("Cherokee Benchmarker %s\n"
		"Usage: cherokee-bench [options] URL\n\n"
		"Try `cherokee-bench --help' for more options.\n", VERSION);
}


int
main (int argc, char *argv[])
{
	cherokee_bench_t *bench;

	struct option long_options[] = {
		/* Options without arguments */
		{"help",          no_argument,       NULL, 'h'},
		{"version",       no_argument,       NULL, 'V'},
		{"threads",       required_argument, NULL, 't'},
		{"pipeline",      required_argument, NULL, 'P'},
		{"concurrency",   required_argument, NULL, 'c'},
		{"number",        required_argument, NULL, 'n'},
		{"urls",          required_argument, NULL, 'u'},
		{"not-keepalive", no_argument,       NULL, 'K'},
		{NULL, 0, NULL, 0}
	};

	int   val;
	ret_t ret;
	int   param_num;

	/* Create the benchmark object
	 */
	ret = cherokee_bench_new (&bench);
	if (ret < ret_ok) return ERROR_NEW;
	

	/* Parse the paramaters
	 */
	while ((val = getopt_long (argc, argv, "Vhp:t:P:c:n:u:K", long_options, NULL)) != -1) {
		switch (val) {
		case 'V':
			printf ("Cherokee Benchmarker %s\n"
				"Written by Alvaro Lopez Ortega <alvaro@gnu.org>\n\n"
				"Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega.\n"
				"This is free software; see the source for copying conditions.  There is NO\n"
				"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n",
				VERSION);
			exit(0);
		case 't':
			BENCH_THREADS(bench) = atoi(optarg);
			break;
		case 'P':
			BENCH_PIPELINE(bench) = atoi(optarg);
			break;
		case 'c':
			BENCH_CONCU(bench) = atoi(optarg);
			break;
		case 'n':
			BENCH_NUMBER(bench) = atoi(optarg);
			break;
		case 'u':
			ret = cherokee_bench_read_url_file (bench, optarg);
			if (ret < ret_ok)  return ERROR_READING;
		case 'K':
			BENCH_KEEPALIVE(bench) = keepalive_disable;
			break;
		case 'h':
		case '?':
		default:
			print_help();
			exit(0);
		}
	}

	/* Add URLs..
	 */
	param_num = argc - optind;

	if (param_num > 0) {
		for (val=optind; val<optind+param_num; val++) {
			ret = cherokee_bench_add_url (bench, argv[val]);
			if (ret < ret_ok) {
				PRINT_ERROR ("Error adding URL '%s'\n", argv[val]);
				return ERROR_URL;
			}
		}
	}

	/* Init the benchmark
	 */
	ret = cherokee_bench_init (bench);
	if (ret < ret_ok) return ERROR_INIT;

	/* Print info..
	 */
	cherokee_bench_print_info (bench);
	
	/* Run the benchmark!
	 */
	ret = cherokee_bench_run (bench);
	if (ret < ret_ok) return ERROR_RUN;

	ret = cherokee_bench_free (bench);
	if (ret < ret_ok) return ERROR_FREE;

	return ERROR_OK;
}
