/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <fcntl.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/tcp.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>	
#include <sys/wait.h>

#include "common.h"
#include "server.h"
#include "socket.h"
#include "connection.h"
#include "log.h"
#include "ncpus.h"
#include "mime.h"

#ifdef HAVE_PTHREAD
#include "thread.h"
#endif

static ret_t init_tls (void);


ret_t
cherokee_server_new  (cherokee_server_t **srv)
{
	ret_t ret;

	/* Get memory
	 */
	CHEROKEE_NEW_STRUCT(n, server);
	
	n->wanna_exit      = 0;
	n->config_filename = NULL;
	
	/* Sockets
	 */
	n->socket     = -1;
	n->socket_tls = -1;
	n->ipv6       =  1;

	CHEROKEE_MUTEX_INIT (&n->accept_mutex, NULL);
//	CHEROKEE_MUTEX_INIT (&n->socket_tls_mutex, NULL);
	
	/* Threads
	 */
	INIT_LIST_HEAD(&n->thread_list);
	cherokee_thread_new (&n->main_thread, n, 0);

	/* Default Index files list
	 */
	INIT_LIST_HEAD(&n->index_list);

	/* Server config
	 */
	n->port           = 80;
	n->port_tls       = 443;
	n->listen_to      = NULL;
	n->fdwatch_msecs  = 999;
	n->hideversion    = 0;
	n->hideservername = 0;
	n->userdir        = NULL;
	n->userdir_handler= NULL;
	n->start_time     = time(NULL);

	n->icons_file     = NULL;
	n->mime_file      = NULL;

	n->keepalive      = 1;
	n->keepalive_max  = MAX_KEEP_ALIVE;

	n->ncpus          = -1;
	n->thread_num     = 0;

	n->chroot         = NULL;
	n->chrooted       = 0;

	n->user_orig      = getuid();
	n->user           = n->user_orig;
	n->group_orig     = getgid();
	n->group          = n->group_orig;

	n->timeout        = 15;
	cherokee_buffer_new (&n->timeout_header);
	cherokee_buffer_add (n->timeout_header, "Keep-Alive: timeout=15"CRLF, 48);


	cherokee_buffer_new (&n->bogo_now_string);
	cherokee_buffer_ensure_size (n->bogo_now_string, 100);

	/* Virtual servers table
	 */
	INIT_LIST_HEAD (&n->vservers);

	cherokee_table_new (&n->vservers_ref);
	return_if_fail (n->vservers_ref!=NULL, ret_nomem);

	cherokee_virtual_server_new (&n->vserver_default);
	return_if_fail (n->vserver_default!=NULL, ret_nomem);

	/* Module loader
	 */
	cherokee_module_loader_new (&n->loader);
	return_if_fail (n->loader != NULL, ret_nomem);	
		
	/* Encoders 
	 */
	cherokee_encoder_table_new (&n->encoders);
	return_if_fail (n->encoders != NULL, ret_nomem);

	/* Icons 
	 */
	cherokee_icons_new (&n->icons);
	return_if_fail (n->icons != NULL, ret_nomem);

	/* Loggers
	 */
	n->log_flush_next   = 0;
	n->log_flush_elapse = 10;

	cherokee_logger_table_new (&n->loggers);
	return_if_fail (n->loggers != NULL, ret_nomem);

	ret = init_tls();
	if (ret < ret_ok) {
		return ret;
	}

	/* Return the object
	 */
	*srv = n;

	return ret_ok;
}


static ret_t
init_tls (void)
{
#ifdef HAVE_TLS
	int rc;

	rc = gnutls_global_init();
	if (rc < 0) {
		PRINT_ERROR ("Global TLS state initialisation failed.\n");
		return ret_error;
	}

	rc = gnutls_global_init_extra();
	if (rc < 0) {
		PRINT_ERROR ("Global TLS state initialisation `extra' failed.\n");
		return ret_error;
	}	
#endif

	return ret_ok;
}


static inline ret_t
change_execution_user (cherokee_server_t *srv)
{
	int error;

	/* Change of group requested
	 */
	if (srv->group != srv->group_orig) {
		error = setgid (srv->group);
		if (error != 0) {
			PRINT_ERROR ("Can't change group to GID %d, running with GID=%d\n",
				     srv->group, srv->group_orig);
		}
	}

	/* Change of user requested
	 */
	if (srv->user != srv->user_orig) {
		error = setuid (srv->user);		
		if (error != 0) {
			PRINT_ERROR ("Can't change user to UID %d, running with UID=%d\n",
				     srv->user, srv->user_orig);
		}
	}

	return ret_ok;
}


void  
cherokee_server_set_min_latency (cherokee_server_t *srv, int msecs)
{
	srv->fdwatch_msecs = msecs;
}


static ret_t
set_server_socket_opts (int socket)
{
	int re;
	int on;

	/* Set 'close-on-exec'
	 */
	fcntl (socket, F_SETFD, FD_CLOEXEC);
		
	/* To re-bind without wait to TIME_WAIT
	 */
	on = 1;
	re = setsockopt (socket, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
	if (re != 0) return ret_error;

	/* TCP_MAXSEG:
	 * The maximum size of a TCP segment is based on the network MTU for des-
	 * tinations on local networks or on a default MTU of 576 bytes for desti-
	 * nations on nonlocal networks.  The default behavior can be altered by
	 * setting the TCP_MAXSEG option to an integer value from 1 to 65,535.
	 * However, TCP will not use a maximum segment size smaller than 32 or
	 * larger than the local network MTU.  Setting the TCP_MAXSEG option to a
	 * value of zero results in default behavior.  The TCP_MAXSEG option can
	 * only be set prior to calling listen or connect on the socket.  For pas-
	 * sive connections, the TCP_MAXSEG option value is inherited from the
	 * listening socket. This option takes an int value, with a range of 0 to
	 * 65535.
	 */
	on = 64000;
	re = setsockopt (socket, SOL_SOCKET, TCP_MAXSEG, &on, sizeof(on));
	if (re != 0) return ret_error;
	

	/* TCP_DEFER_ACCEPT:
	 * Allows a listener to be awakened only when data arrives on the socket.
	 * Takes an integer value (seconds), this can bound the maximum number of
	 * attempts TCP will make to complete the connection. This option should 
	 * not be used in code intended to be portable.
	 */
	on = 5;
	re = setsockopt (socket, SOL_SOCKET, TCP_DEFER_ACCEPT, &on, sizeof(on));
	if (re != 0) return ret_error;

	return ret_ok;
}


static ret_t
initialize_server_socket4 (cherokee_server_t *srv, unsigned short port, int *srv_socket_ret)
{
	int re;
	int srv_socket;
	struct sockaddr_in  sockaddr;
	
	srv_socket = socket (AF_INET, SOCK_STREAM, 0);
	if (srv_socket <= 0) {
		PRINT_ERROR ("Error creating IPv4 server socket\n");
		exit(EXIT_CANT_CREATE_SERVER_SOCKET4);
	}

	*srv_socket_ret = srv_socket;
	set_server_socket_opts(srv_socket);

	/* Bind the socket
	 */
	sockaddr.sin_port   = htons (port);
	sockaddr.sin_family = AF_INET;

	if (srv->listen_to == NULL) {
		sockaddr.sin_addr.s_addr = INADDR_ANY; 
	} else {
#ifdef HAVE_INET_PTON
		inet_pton (sockaddr.sin_family, srv->listen_to, &sockaddr.sin_addr);
#else
		/* IPv6 needs inet_pton. inet_addr just doesn't work without
		 * it. We'll suppose then that we haven't IPv6 support 
		 */
		sockaddr.sin_addr.s_addr = inet_addr (srv->listen_to);
#endif
	}

	re = bind (srv_socket, (struct sockaddr *)&sockaddr, sizeof(sockaddr));
	return (re == 0) ? ret_ok : ret_error;
}


static ret_t
initialize_server_socket6 (cherokee_server_t *srv, unsigned short port, int *srv_socket_ret)
{
#ifdef HAVE_IPV6
	int re;
	int srv_socket;
	struct sockaddr_in6 sockaddr;


	/* Create the socket
	 */
	srv_socket = socket (AF_INET6, SOCK_STREAM, 0);
	if (srv_socket <= 0) {
		PRINT_ERROR ("Error creating IPv6 server socket.. switching to IPv4\n");
		srv->ipv6 = 0;
		return ret_error;
	}


	*srv_socket_ret = srv_socket;
	set_server_socket_opts(srv_socket);

	/* Bind the socket
	 */
	sockaddr.sin6_port   = htons (port);  /* Transport layer port #   */
	sockaddr.sin6_family = AF_INET6;      /* AF_INET6                 */

	if (srv->listen_to == NULL) {
		sockaddr.sin6_addr = in6addr_any; 
	} else {
		inet_pton (sockaddr.sin6_family, srv->listen_to, &sockaddr.sin6_addr);
	}

	re = bind (srv_socket, (struct sockaddr *)&sockaddr, sizeof(sockaddr));
	return (re == 0) ? ret_ok : ret_error;
#endif
}


static ret_t
print_banner (cherokee_server_t *srv)
{
	   char *method;
	   CHEROKEE_NEW(n,buffer);
	

	   cherokee_buffer_add_va (n, "Cherokee Web Server %s: ", VERSION);
	   cherokee_buffer_add_va (n, "Listening on port %d", srv->port);
	   if (srv->thread_num < 1) {
			 cherokee_buffer_add (n, ", single thread", 15);
	   } else {
			 cherokee_buffer_add_va (n, ", %d threads", srv->thread_num+1);
	   }
	   
	   cherokee_fdpoll_get_method (srv->main_thread->fdpoll, &method);
	   cherokee_buffer_add_va (n, ", using %s", method);
						  
	   printf ("%s\n", n->buf);
	   cherokee_buffer_free (n);

	   return ret_ok;
}


static ret_t
initialize_server_socket (cherokee_server_t *srv, unsigned short port, int *srv_socket_ptr)
{
	ret_t ret;
	int   flags;
	int   srv_socket;

#ifdef HAVE_IPV6
	if (srv->ipv6) {
		ret = initialize_server_socket6 (srv, port, srv_socket_ptr);
	} 
#endif
	if ((ret != ret_ok) || (srv->ipv6 == 0)) {
		ret = initialize_server_socket4 (srv, port, srv_socket_ptr);
	}

	srv_socket = *srv_socket_ptr;

	if (ret != 0) {
		uid_t uid = getuid();
		gid_t gid = getgid();

		PRINT_ERROR ("Can't bind() socket (port=%d, UID=%d, GID=%d)\n", port, uid, gid);
		return ret_error;
	}
	   
	/* Set no-delay mode
	 */
	flags = fcntl (srv_socket, F_GETFL, 0);
	return_if_fail (flags != -1, ret_error);
	
	ret = fcntl (srv_socket, F_SETFL, flags | O_NDELAY);
	return_if_fail (ret >= 0, ret_error);
	
	/* Listen
	 */
	ret = listen (srv_socket, 1024);
	if (ret < 0) {
		close (srv_socket);
		return_if_fail (0, ret_error);
	}

	return ret_ok;
}




static ret_t
initialize_server_threads (cherokee_server_t *srv)
{
#ifdef HAVE_PTHREAD
	int i;

	for (i=0; i<srv->ncpus - 1; i++) {
		ret_t ret;
		cherokee_thread_t *thread;

		ret = cherokee_thread_new (&thread, srv, 1);
		if (ret != ret_ok) {
			return ret;
		}
		
		list_add ((list_t *)thread, &srv->thread_list);
		srv->thread_num++;
	}
#endif

	return ret_ok;
}


#ifdef HAVE_TLS
static void
for_each_vserver_init_tls_func (const char *key, void *value)
{
	cherokee_virtual_server_init_tls (VSERVER(value));
}
#endif


static void
add_socket_to_thread_fdpolls (cherokee_server_t *srv, int srv_socket)
{
	list_t *i;

	/* Add the main thread
	 */
	cherokee_fdpoll_add (srv->main_thread->fdpoll, srv_socket, 0);

	/* The rest
	 */
	list_for_each (i, &srv->thread_list) {
		cherokee_fdpoll_add (THREAD(i)->fdpoll, srv_socket, 0);		
	}
}


ret_t
cherokee_server_init (cherokee_server_t *srv) 
{   
	ret_t ret;

	/* If the server has a previous server socket opened, Eg:
	 * because a SIGHUP, it shouldn't init the server socket.
	 */
	if (srv->socket == -1) {
		ret = initialize_server_socket (srv, srv->port, &srv->socket);
		if (ret != ret_ok) return ret;
	}

#ifdef HAVE_TLS
	/* Initialize the server TLS socket
	 */
	if (srv->socket_tls == -1) {
		ret = initialize_server_socket (srv, srv->port_tls, &srv->socket_tls);
		if (ret != ret_ok) return ret;
	}

	/* Initialize TLS in all the virtual servers
	 */
	cherokee_virtual_server_init_tls (srv->vserver_default);
	cherokee_table_foreach (srv->vservers_ref, for_each_vserver_init_tls_func);
#endif

        /* Threads / CPUs
	 */
	if (srv->ncpus == -1) {
		dcc_ncpus (&srv->ncpus);
	}
	
	if (srv->ncpus != 1) {
		ret = initialize_server_threads (srv);
		if (ret != ret_ok) return ret;
	}

	/* Add the server socket to fdpolls
	 */
	add_socket_to_thread_fdpolls (srv, srv->socket);
#ifdef HAVE_TLS
	add_socket_to_thread_fdpolls (srv, srv->socket_tls);
#endif

	/* Change the user
	 */
	ret = change_execution_user (srv);
	if (ret != ret_ok) {
		return ret;
	}
	
	/* Chroot
	 */
	if (srv->chroot) {
		srv->chrooted = (chroot (srv->chroot) == 0);
		if (srv->chrooted == 0) {
			PRINT_ERROR ("Cannot chroot() to '%s'\n", srv->chroot);
		}
	} 

	chdir ("/");

	print_banner (srv);

	return ret_ok;
}


static inline void
close_all_connections (cherokee_server_t *srv)
{
	list_t *i;

	cherokee_thread_close_all_connections (srv->main_thread);
	
	list_for_each (i, &srv->thread_list) {
		cherokee_thread_close_all_connections (THREAD(i));
	}
}


static void
for_each_func_free_vserver (const char *key, void *vserver)
{	
	cherokee_virtual_server_free (vserver);
}


ret_t 
cherokee_server_clean (cherokee_server_t *srv)
{
	/* Close all active connections
	 */
	close_all_connections (srv);

	/* Clean the configuration
	 */
	if (srv->listen_to != NULL) {
		free (srv->listen_to);
		srv->listen_to = NULL;
	}

	if (srv->userdir != NULL) {
		free (srv->userdir);
		srv->userdir = NULL;
	}

	if (srv->userdir_handler != NULL) {
		cherokee_handler_table_entry_free (srv->userdir_handler);
		srv->userdir_handler = NULL;
	}

	if (srv->chroot != NULL) {
		free (srv->chroot);
		srv->chroot = NULL;
	}

	if (srv->icons_file != NULL) {
		free (srv->icons_file);
		srv->icons_file = NULL;
	}

	if (srv->mime_file != NULL) {
		   free (srv->mime_file);
		   srv->mime_file = NULL;
	}
	
	/* Clean the loader
	 */
	cherokee_module_loader_free (srv->loader);
	srv->loader = NULL;

	/* Clean the encoders table
	 */
	cherokee_encoder_table_clean (srv->encoders);

	/* Clean the loggers table
	 */
	cherokee_logger_table_clean (srv->loggers);

	/* Clean the virtual servers table
	 */
	cherokee_virtual_server_clean (srv->vserver_default);

	/* Clean the virtual servers table
	 */	
	cherokee_list_free (srv->vservers, cherokee_virtual_server_free);
	cherokee_table_clean (srv->vservers_ref);

	/* Icons 
	 */
	cherokee_icons_clean (srv->icons);

	/* Time-out pre-builded header
	 */
	cherokee_buffer_clean (srv->timeout_header);

	/* Clean the index list
	 */
	cherokee_list_free (&srv->index_list, free);
	INIT_LIST_HEAD(&srv->index_list);

	CHEROKEE_MUTEX_DESTROY (&srv->accept_mutex);
//	CHEROKEE_MUTEX_DESTROY (&srv->socket_tls_mutex);

 	srv->wanna_exit     = 0;
	srv->hideversion    = 0;
	srv->hideservername = 0;
	srv->keepalive_max  = MAX_KEEP_ALIVE;

	return ret_ok;
}


ret_t
cherokee_server_free (cherokee_server_t *srv)
{
	close (srv->socket);

	if (srv->socket_tls != -1) {
		close (srv->socket_tls);		
	}

	cherokee_encoder_table_free (srv->encoders);
	cherokee_logger_table_free (srv->loggers);

	cherokee_list_free (&srv->index_list, free);

	/* Virtual servers
	 */
	cherokee_virtual_server_free (srv->vserver_default);
	srv->vserver_default = NULL;

	cherokee_list_free (srv->vservers, cherokee_virtual_server_free);
	cherokee_table_free (srv->vservers_ref);

	cherokee_module_loader_free (srv->loader);

	cherokee_icons_free (srv->icons);
	cherokee_buffer_free (srv->bogo_now_string);

	cherokee_buffer_free (srv->timeout_header);

	if (srv->listen_to != NULL) {
		free (srv->listen_to);
		srv->listen_to = NULL;
	}

	if (srv->userdir != NULL) {
		free (srv->userdir);
		srv->userdir = NULL;
	}

	if (srv->userdir_handler != NULL) {
		cherokee_handler_table_entry_free (srv->userdir_handler);
		srv->userdir_handler = NULL;
	}
	
	if (srv->chroot != NULL) {
		free (srv->chroot);
		srv->chroot = NULL;
	}

	if (srv->config_filename != NULL) {
		free (srv->config_filename);
		srv->config_filename = NULL;
	}

	if (srv->icons_file != NULL) {
		free (srv->icons_file);
		srv->icons_file = NULL;
	}

	if (srv->mime_file != NULL) {
		   free (srv->mime_file);
		   srv->mime_file = NULL;
	}

	free (srv);
	
	return ret_ok;
}


void static
flush_vserver (const char *key, void *value)
{
	/* There's no logger in this virtual server
	 */
	if ((value == NULL) || (VSERVER_LOGGER(value) == NULL))
		return;

	cherokee_logger_flush (VSERVER_LOGGER(value));
}


void static 
flush_logs (cherokee_server_t *srv)
{
	flush_vserver (NULL, srv->vserver_default);
	cherokee_table_foreach (srv->vservers_ref, flush_vserver);
}


ret_t 
cherokee_server_reinit (cherokee_server_t *srv)
{
	ret_t  ret;
	
	if (srv->chrooted) {
		fprintf (stderr, 
			 "WARNING: Chrooted cherokee cannot be reloaded. "
			 "Please, stop and restart it again.\n");
		return ret_ok;
	}


	ret = cherokee_server_clean (srv);
	if (ret != ret_ok) {
		exit(EXIT_SERVER_CLEAN);
	}

	ret = cherokee_server_read_config_file (srv, NULL);
	if (ret != ret_ok) {
		exit(EXIT_SERVER_READ_CONFIG);
	}

	ret = cherokee_server_init (srv);
	if (ret != ret_ok) {
		exit(EXIT_SERVER_INIT);	
	}

	return ret_ok;
}


static void
update_bogo_now (cherokee_server_t *srv)
{
	static unsigned char to_wait = 1;

 	srv->bogo_now = time (NULL);
	
	if (--to_wait == 0) {
		srv->bogo_now_string->len = strftime (srv->bogo_now_string->buf, srv->bogo_now_string->size,
						      "%a, %d %b %Y %H:%M:%S GMT", gmtime(&srv->bogo_now));
		to_wait = 1000 / srv->fdwatch_msecs;
	}
}


void
cherokee_server_step (cherokee_server_t *srv)
{
	ret_t ret;

	/* Process the new connections
	 */
//	process_new_connections (srv, srv->socket, 0);

#ifdef HAVE_TLS	
//	process_new_connections (srv, srv->socket_tls, 1);
#endif	

	/* Get the time
	 */
	update_bogo_now (srv);
	ret = cherokee_thread_step (srv->main_thread);

	/* Logger flush 
	 */
	if (srv->log_flush_next < srv->bogo_now) {
		flush_logs (srv);
		srv->log_flush_next = srv->bogo_now + srv->log_flush_elapse;
	}

	/* Wanna exit?
	 */
	if (srv->wanna_exit) {
		cherokee_server_reinit (srv);
		srv->wanna_exit = 0;
		printf ("Cherokee reloaded\n");
	}
}


ret_t 
cherokee_server_read_config_file (cherokee_server_t *srv, char *filename)
{
	ret_t ret;
	int   error;
	void *bufstate;
	extern FILE *yyin;

	extern int  yyparse             (void *);
	extern void yy_switch_to_buffer (void *);
	extern void yy_delete_buffer    (void *);


	/* Maybe set a default configuration filename
	 */
	if (filename == NULL) {
		filename = (srv->config_filename)? srv->config_filename : CHEROKEE_CONFDIR"/cherokee.conf";

	} else {
		/* Maybe free an old filename
		 */
		if (srv->config_filename != NULL) {
			free (srv->config_filename);
		}
		
		/* Store last configuration filename
		 */
		srv->config_filename = strdup(filename);
	}

	/* Set the file to read
	 */
	yyin = fopen (filename, "r");
	if (yyin == NULL) {
		PRINT_ERROR("Can't read the configuration file: '%s'\n", filename);
		return ret_error;
	}

	/* Cooooome on :-)
	 */
	yyrestart(yyin);

	bufstate = (void *) yy_create_buffer (yyin, 65535);
	yy_switch_to_buffer (bufstate);
	error = yyparse ((void *)srv);
	yy_delete_buffer (bufstate);

	fclose (yyin);

	ret = (error == 0) ? ret_ok : ret_error;
	if (ret < ret_ok) {
		return ret;
	}

	/* Maybe read the Icons file
	 */
	if (srv->icons_file != NULL) {
		ret = cherokee_icons_read_config_file (srv->icons, srv->icons_file);
		if (ret < ret_ok) {
			PRINT_ERROR ("Cannot read the icons file\n");
		}
	}

	/* Maybe read the MIME file
	 */
	if (srv->mime_file != NULL) {
		   cherokee_mime_t *mime;

		   ret = cherokee_mime_get_default (&mime);
		   if (ret < ret_ok) {
				 PRINT_ERROR("Can not get default MIME configuration file\n");				 
				 goto out;
		   }
		   
		   ret = cherokee_mime_load (mime, srv->mime_file);
		   if (ret < ret_ok) {
				 PRINT_ERROR("Can not load MIME configuration file %s\n", srv->mime_file);
		   }
	out:
	}

	return ret;
}


ret_t 
cherokee_server_read_config_string (cherokee_server_t *srv, char *config_string)
{
	ret_t ret;
	int   error;
	void *bufstate;

	extern int  yyparse             (void *);
	extern int  yy_scan_string      (void *);
	extern void yy_switch_to_buffer (void *);
	extern void yy_delete_buffer    (void *);

	bufstate = (void *) yy_scan_string (config_string);
	yy_switch_to_buffer(bufstate);

	error = yyparse((void *)srv);

	yy_delete_buffer (bufstate);

	ret = (error == 0) ? ret_ok : ret_error;
	if (ret < ret_ok) {
		return ret;
	}
	
	/* Maybe read the Icons file
	 */
	if (srv->icons_file != NULL) {
		ret = cherokee_icons_read_config_file (srv->icons, srv->icons_file);
		if (ret < ret_ok) {
			PRINT_ERROR ("Cannot read the icons file\n");
		}
	}

	/* Maybe read the MIME file
	 */
	if (srv->mime_file != NULL) {
		   cherokee_mime_t *mime;

		   ret = cherokee_mime_get_default (&mime);
		   if (ret < ret_ok) {
				 PRINT_ERROR("Can not get default MIME configuration file\n");				 
				 goto out;
		   }
		   
		   ret = cherokee_mime_load (mime, srv->mime_file);
		   if (ret < ret_ok) {
				 PRINT_ERROR("Can not load MIME configuration file %s\n", srv->mime_file);
		   }
	out:
	}

	return ret;
}


ret_t 
cherokee_server_handle_HUP (cherokee_server_t *srv)
{
	srv->wanna_exit = 1;
	return ret_ok;
}


static void 
exit_parent (int sig)
{
        /* parent simply exits when child says go... 
	    */
        exit(0);
}

ret_t 
cherokee_server_daemonize (cherokee_server_t *srv)
{
	/* Function Copy&Pasted from 'SMBFS mount program'
	 * Copyright (C) Andrew Tridgell 1999
	 */

	int j, status;
        pid_t child_pid;
	
        signal (SIGTERM, exit_parent);

        if ((child_pid = fork()) < 0) {
                PRINT_ERROR ("Could not fork\n");
        }

        if (child_pid > 0) {
                while (1) {
                        j = waitpid( child_pid, &status, 0 );
                        if (j < 0) {
                                if (EINTR == errno) {
                                        continue;
                                }
                                status = errno;
                        }
                        break;
                }
			 
                /* If we get here - the child exited with some error status 
			  */
                if (WIFSIGNALED(status))
                        exit(128 + WTERMSIG(status));
                else
                        exit(WEXITSTATUS(status));
        }

        signal (SIGTERM, SIG_DFL);
        chdir("/");

	return ret_ok;
}


ret_t 
cherokee_server_get_active_conns (cherokee_server_t *srv, int *num)
{
	int     active = 0;
	list_t *thread, *i;

	/* Active connections number
	 */
	list_for_each (thread, &srv->thread_list) {
		active += THREAD(thread)->active_list_num;
	}
	
	active += srv->main_thread->active_list_num;

	/* Return out parameters
	 */
	*num = active;

	return ret_ok;
}


ret_t 
cherokee_server_get_reusable_conns (cherokee_server_t *srv, int *num)
{
	int     reusable = 0;
	list_t *thread, *i;

	/* Reusable connections
	 */
	list_for_each (thread, &srv->thread_list) {
		list_for_each (i, &THREAD(thread)->reuse_list) {
			reusable++;
		}
	}
	list_for_each (i, &THREAD(srv->main_thread)->reuse_list) {
		reusable++;
	}

	/* Return out parameters
	 */
	*num = reusable;

	return ret_ok;
}


ret_t 
cherokee_server_get_total_traffic (cherokee_server_t *srv,  unsigned long int *rx, unsigned long int *tx)
{
	list_t *i;

	*rx = srv->vserver_default->data.rx;
	*tx = srv->vserver_default->data.tx;

	list_for_each (i, &srv->vservers) {
		*rx += VSERVER(i)->data.rx;
		*tx += VSERVER(i)->data.tx;
	}

	return ret_ok;
}
