/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee Benchmarker
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "bench.h"


ret_t 
cherokee_bench_new  (cherokee_bench_t **bench)
{
	CHEROKEE_NEW_STRUCT(n,bench);

	/* Init lists
	 */
	INIT_LIST_HEAD((list_t*)&n->thread_list);
	INIT_LIST_HEAD((list_t*)&n->request_list);

	/* Init mutexes
	 */
	CHEROKEE_MUTEX_INIT (&n->request_list_mutex, NULL);

	BENCH_THREADS(n)   = 5;
	BENCH_PIPELINE(n)  = 3;
	BENCH_CONCU(n)     = 10;
	BENCH_NUMBER(n)    = 1000;
	BENCH_KEEPALIVE(n) = keepalive_enable;

	n->request_list_current = NULL;

	/* Return the object
	 */
	*bench = n;
	return ret_ok;
}


ret_t 
cherokee_bench_free (cherokee_bench_t *bench)
{
	list_t *i, *tmp;
	
	/* Free URL list
	 */
	list_for_each_safe (i, tmp, &bench->request_list) {
		list_del (i);
		cherokee_bench_request_free (REQUEST(i));
	}

	/* Free thread list
	 */
	list_for_each_safe (i, tmp, &bench->thread_list) {
		list_del (i);
		cherokee_bench_thread_free (THREAD(i));
	}

	/* Free mutexes
	 */
	CHEROKEE_MUTEX_DESTROY (&bench->request_list_mutex);

	free (bench);
	return ret_ok;
}


static ret_t
init_threads (cherokee_bench_t *bench)
{
	int   i;
	ret_t ret;
	cherokee_bench_thread_t *thread        = NULL;
	int                      thread_number = BENCH_THREADS(bench);
	   

	/* Grab starting time
	 */
	gettimeofday (&bench->time.start, NULL);

	/* Create the threads
	 */
	for (i=0; i<thread_number; i++) 
	{
		/* Construct the object
		 */
		ret = cherokee_bench_thread_new (&thread, bench);
		if (ret < ret_ok) return ret;

		/* Configure the object
		 */
		THREAD_CONNS_TODO(thread) = BENCH_NUMBER(bench)/thread_number;
		
		/* Add to the threads list
		 */
		list_add ((list_t *)thread, &bench->thread_list);
		
		/* Run it!
		 */
		cherokee_bench_thread_run (thread);

		/* Check for errors
		 */
		if (THREAD_ERROR(thread) < thread_ok) {
			return ret_error;
		}
	}

	return ret_ok;
}


ret_t 
cherokee_bench_init (cherokee_bench_t *bench)
{
	ret_t   ret;
	list_t *i;

	/* Check for URLs
	 */
	if (list_empty(&bench->request_list)) {
		PRINT_ERROR ("Error: No URL to request\n"
			     "Try `cherokee-bench --help' for more options.\n");
		return ret_error;
	}

	/* Initialize threads
	 */
	ret = init_threads (bench);
	if (ret < ret_ok) return ret;

	return ret_ok;
}


static int
timeval_subtract (struct timeval *result, struct timeval *x, struct timeval *y)
{
	/* Perform the carry for the later subtraction by updating y. 
	 */
	if (x->tv_usec < y->tv_usec) {
		int nsec = (y->tv_usec - x->tv_usec) / 1000000 + 1;
		y->tv_usec -= 1000000 * nsec;
		y->tv_sec += nsec;
	}

	if (x->tv_usec - y->tv_usec > 1000000) {
		int nsec = (x->tv_usec - y->tv_usec) / 1000000;
		y->tv_usec += 1000000 * nsec;
		y->tv_sec -= nsec;
	}

	/* Compute the time remaining to wait. tv_usec is certainly positive. 
	 */
	result->tv_sec = x->tv_sec - y->tv_sec;
	result->tv_usec = x->tv_usec - y->tv_usec;

	/* Return 1 if result is negative. 	
	 */
	return x->tv_sec < y->tv_sec;
}


static ret_t
collect_bench_information (cherokee_bench_t *bench)
{
	list_t *i;
	struct timeval elapse;
	float          elapse_usec;
	cherokee_bench_info_t info;

	/* Calculate the total amount..
	 */
	cherokee_bench_info_init (&info);

	list_for_each (i, &bench->thread_list) {
		cherokee_bench_info_add (&info, &THREAD(i)->info);
	}

	/* Grab ending time
	 */
	gettimeofday (&bench->time.end, NULL);

	timeval_subtract (&elapse, &bench->time.end, &bench->time.start);
	elapse_usec = (elapse.tv_sec * 1000000) + elapse.tv_usec;

	/* Print it!
	 */
	cherokee_bench_info_print (&info);
	printf ("Time        %8.2f secs\n"
		"Speed       %8.2f conns/sec\n", 
		elapse_usec/1000000, info.transfers/(elapse_usec/1000000));

	return ret_ok;
}


ret_t 
cherokee_bench_run (cherokee_bench_t *bench)
{
	int     re;
	ret_t   ret;
	list_t *i;
	void   *thread_ret;

	/* Waif for the threads
	 */
	list_for_each (i, &bench->thread_list) {
		re = pthread_join (THREAD(i)->thread, &thread_ret);
		if (re < 0) {
			PRINT_ERROR ("Error %d waiting for thread\n", re);
		}
	}

	/* Collect thread information
	 */
	ret = collect_bench_information (bench);
	if (ret < ret_ok) return ret;

	return ret_ok;
}




ret_t 
cherokee_bench_add_url (cherokee_bench_t *bench, char *url_string)
{
	ret_t  ret;
	char  *server;
	CHEROKEE_NEW (entry,bench_request);

	/* Parse the URL string
	 */	
	ret = cherokee_url_parse (REQUEST_URL(entry), url_string);
	if (ret < ret_ok) return ret;
	
	/* Add the URL object to the list
	 */
	list_add ((list_t *)entry, &bench->request_list);

	return ret_ok;
}


ret_t 
cherokee_bench_read_url_file (cherokee_bench_t *bench, char *filename)
{
	ret_t ret;
	int   error;
	void *bufstate;
	extern FILE *yy_url_in;

	extern int  yy_url_parse        (void *);
	extern void yy_switch_to_buffer (void *);
	extern void yy_delete_buffer    (void *);

	/* Set the file to read
	 */
	yy_url_in = fopen (filename, "r");
	if (yy_url_in == NULL) {
		PRINT_ERROR("Can't read the url file: '%s'\n", filename);
		return ret_error;
	}

	/* Parse the file
	 */
	yy_url_restart (yy_url_in);
	bufstate = (void *) yy_url__create_buffer (yy_url_in, 65535);
	yy_url__switch_to_buffer (bufstate);
	error = yy_url_parse ((void *)bench);

	/* Finish
	 */
	yy_url__delete_buffer (bufstate);
	fclose (yy_url_in);

	/* Check the error value
	 */
	ret = (error == 0) ? ret_ok : ret_error;
	if (ret < ret_ok) {
		return ret;
	}

	return ret_ok;
}


ret_t 
cherokee_bench_print_info (cherokee_bench_t *bench)
{
	ret_t ret;

	printf ("Threads            %d\n", BENCH_THREADS(bench));

	return ret_ok;
}


ret_t 
_cherokee_bench_get_request (cherokee_bench_t *bench, cherokee_bench_request_t **req)
{
	CHEROKEE_MUTEX_LOCK (&bench->request_list_mutex);

	if (bench->request_list_current == NULL) {
		/* It is the first time..
		 */
		bench->request_list_current = bench->request_list.next;
       
	} else {
		/* Move to the next list node
		 */
		if (bench->request_list_current->next != &bench->request_list) {
			bench->request_list_current = bench->request_list_current->next;
		} else {
			bench->request_list_current = bench->request_list.next;
		}
	}

	/* Return the current URL object
	 */
	*req = bench->request_list_current;

	CHEROKEE_MUTEX_UNLOCK (&bench->request_list_mutex);	
	return ret_ok;
}


ret_t 
_cherokee_bench_add_request (cherokee_bench_t *bench, cherokee_bench_request_t *req)
{
	list_add ((list_t *)req, &bench->request_list);
	return ret_ok;
}
