/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include "mime.h"
#include "handler_file.h"
#include "connection.h"

ret_t
cherokee_handler_file_new  (cherokee_handler_t **hdl, char *request)
{
	int   ret;
	char *ext;

	cherokee_handler_file_t *n = (cherokee_handler_file_t *)malloc(sizeof(cherokee_handler_file_t));
	return_if_fail (n != NULL, ret_nomem);

	/* Init the base class object
	 */
	cherokee_handler_init(HANDLER(n));
	HANDLER(n)->free        = (func_free_t) cherokee_handler_file_free;
	HANDLER(n)->step        = (func_step_t) cherokee_handler_file_step;
	HANDLER(n)->add_headers = (func_add_headers_t) cherokee_handler_file_add_headers;
	
	*hdl = HANDLER(n);
	
	/* Init
	 */
	n->offset = 0;
	n->mime   = NULL;
	n->fd     = open (request, O_RDONLY);

	ret = fstat (n->fd, &n->info);

	if (n->fd == -1) {
		if ((ret == 0) && S_ISREG(n->info.st_mode)) {
			HANDLER(n)->status = http_access_denied;
		} else {
			HANDLER(n)->status = http_not_found;
		}
		return ret_error;
	}


	/* Look for the mime type
	 */
	ext = rindex (request, '.');
	if (ext != NULL) {
		cherokee_mime_t *m;

		cherokee_mime_get_default (&m);		
		n->mime = cherokee_mime_get (m, ext+1);
	}


	return ret_ok;
}


static inline void
close_file (cherokee_handler_file_t *fhdl)
{
	if (fhdl->fd != -1) {
		close (fhdl->fd);
		fhdl->fd = -1;
	}
}


ret_t
cherokee_handler_file_free (cherokee_handler_file_t *fhdl)
{
	close_file (fhdl);
	free (fhdl);
	
	return ret_ok;
}


ret_t
cherokee_handler_file_step (cherokee_handler_file_t *fhdl, 
			    cherokee_buffer_t       *buffer)
{
	int total;

	total = read (fhdl->fd, buffer->buf, buffer->size);	
	switch (total) {
	case 0:
		return ret_eof;
	case -1:
		return ret_error;
	default:
		buffer->len = total;
		fhdl->offset += total;
	}	

	/* Maybe it was the last file chunk
	 */
	if (fhdl->offset >= fhdl->info.st_size) {
		return ret_eof_have_data;
	}

	return ret_ok;
}


ret_t
cherokee_handler_file_add_headers (cherokee_handler_file_t *fhdl,
				   cherokee_buffer_t       *buffer)
{
	int length;

	/* Add "Content-length:" header
	 */
	if (fhdl->fd != -1) {
		/* We stat()'ed the file in the handler constructor
		 */
		length = snprintf (gbl_buffer, gbl_buffer_size, "Content-length: %d"CRLF, (int)fhdl->info.st_size);
		cherokee_buffer_add (buffer, gbl_buffer, length);

	} else {
		/* Can't use Keep-alive w/o "Content-length:", so
		 */
		HANDLER(fhdl)->keep_alive = 0;
	}

	/* Add "Content-type:" header
	 */
	if (fhdl->mime != NULL) {
		length = snprintf (gbl_buffer, gbl_buffer_size, "Content-type: %s"CRLF, fhdl->mime);
		cherokee_buffer_add (buffer, gbl_buffer, length);		
	}

	return ret_ok;
}



/*   Library init function
 */

static int _file_is_init = 0;

void
file_init (void)
{
	if (_file_is_init)
		return;

	mime_init();
	_file_is_init = 1;
}
