/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include "server.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#include <signal.h>
#include <unistd.h>	
#include <sys/types.h>


static void
just_exit (int sig)
{
	/* Call exit, to make profiling works
	 */
	exit(sig);
}

static void
print_help (void)
{
	printf ("Cherokee %s\n"
		"Usage: cherokee [options]\n\n"
		"  -b,  --background               Go to background after startup.\n"
		"  -p,  --port=NUM                 Port number.\n"
		"  -d,  --htdocs=DIR               Set 'htdocs' directory.\n"
		"  -u,  --uid=UID                  Set UID.\n"
		"  -g,  --gid=GID                  Set GID.\n"
		"  -l,  --log                      Use syslog.\n"
		"  -h,  --help                     Print this help.\n"
		"\n"
		"Report bugs to alvaro@alobbs.com\n", VERSION);
}

int
main (int argc, char *argv[])
{
	ret_t ret;
	cherokee_server_t *server;

	/* Parameters
	 */
	int   port           = 80;
	int   work_as_daemon =  0;
	int   uid;
	int   set_uid        = 0;
	int   set_gid        = 0;
	int   log            = 0;
	char *chroot_dir     = "./";

	   
	/* Command line options
	 */
	extern char* optarg;

#ifdef HAVE_GETOPT_LONG
	struct option long_options[] = {
		{"help",       no_argument,       NULL, 'h'},
		{"background", no_argument,       NULL, 'b'},
		{"log",        no_argument,       NULL, 'l'},
		{"port",       required_argument, NULL, 'p'},
		{"htdocs",     required_argument, NULL, 'd'},
		{"uid",        required_argument, NULL, 'u'},
		{"gid",        required_argument, NULL, 'g'},
		{NULL, 0, NULL, 0}
	};
#endif
	   
	/* getopt_long
	 */
	int val;

#ifdef HAVE_GETOPT_LONG
	while ((val = getopt_long (argc, argv, "hlbp:d:u:d:", long_options, NULL)) != -1) {
#else
	while ((val = getopt (argc, argv, "hlbp:d:u:d:")) != -1) {
#endif
		switch (val) {
		case 'l':
			log = 1;
			break;
		case 'b':
			work_as_daemon = 1;
			break;
		case 'p':
			port = atoi(optarg);
			break;
		case 'd':
			chroot_dir = optarg;
			break;
		case 'u':
			set_uid = atoi(optarg);
			break;
		case 'g':
			set_gid = atoi(optarg);
			break;
		case 'h':
		case '?':
		default:
			print_help();
			exit(0);
		}
	}

	   
	/* To avoid problems
	 */
	signal(SIGPIPE, SIG_IGN);
	signal(SIGINT,  just_exit);

	   
	/* Get UID
	 */
	uid = getuid();

	   
        /* Daemon mode?
	 */
	if (work_as_daemon == 1)
		if (fork()) return 0;
	   
	/* Create server
	 */
	ret = cherokee_server_new (&server, port);
	if (ret != ret_ok) return(1);


	/* Maybe start the log
	 */
	cherokee_server_log (server, log);


	/* Use default handler
	 */
	cherokee_server_set (server, "/", "common");
	cherokee_server_set (server, "/hidden/", "file"); 

	/* Chroot
	 */
	if (chdir (chroot_dir) == -1) {
		fprintf (stderr, "ERROR: Can't chdir(\"%s\"): ", chroot_dir);
		perror("");
		exit(1);
	}
	
	if (uid == 0) {
		if (chroot (chroot_dir) == -1) {
			fprintf (stderr, "ERROR: Can't chroot(\"%s\"): ", chroot_dir);
			perror("");
			exit(1);
		}
		
	} else {
		fprintf (stderr, "*WARNING*: Can't chroot() because you are not root\n");
	}

	   
	/* Set UID/GID?
	 * IMPORTANT: After create the server
	 */
	if (set_uid != 0) {
		if (setuid (set_uid)) {
			perror("setuid");
			exit(1);
		}
	}

	if (set_gid != 0) {
		if (setgid (set_gid)) {
			perror("setgid");
			exit(1);
		}
	}

	   
	/* Run forever
	 */
	for (;;) {
		cherokee_server_step (server);
	}

	   
	/* Free the memory
	 */
	cherokee_server_free (server);
	   
	return 0;
}
