/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include "handler_dirlist.h"


ret_t
cherokee_handler_dirlist_new  (cherokee_handler_t **hdl, char *request)
{
	int   request_len;
	
	cherokee_handler_dirlist_t *n = (cherokee_handler_dirlist_t *)malloc(sizeof(cherokee_handler_dirlist_t));
	return_if_fail (n != NULL, ret_nomem);
	
	/* Init the base class object
	 */
	cherokee_handler_init(HANDLER(n));
	HANDLER(n)->free        = (func_free_t) cherokee_handler_dirlist_free;
	HANDLER(n)->step        = (func_step_t) cherokee_handler_dirlist_step;
	HANDLER(n)->add_headers = (func_add_headers_t) cherokee_handler_dirlist_add_headers; 

	*hdl = HANDLER(n);
	
	/* Init
	 */
	n->request       = NULL;
	n->page_begining = 0;
	n->page_ending   = 0;
	n->request       = request;

	/* If directory name doesn't finish with a slash,
	 * _add_headers() is going to set a: http_moved_permanently
	 */
	request_len = strlen(request);
	if (request[request_len-1] != '/') {
		HANDLER(n)->redirect = (char *) malloc (request_len+1);
		memcpy (HANDLER(n)->redirect, request, request_len);
		
		HANDLER(n)->redirect[request_len] = '/';
		HANDLER(n)->redirect[request_len+1] = '\0';
		
		HANDLER(n)->status = http_moved_permanently;

		return ret_error;
	}
	
	/* Open dir
	 */
	if (request[0] == '/') {
		char *my_request = (char *) malloc (request_len + 2);
		sprintf (my_request, ".%s", request);
		n->dir = opendir (my_request);
		free (my_request);
	} else {
		n->dir = opendir (request);
	}

	if (n->dir == NULL) {
		HANDLER(n)->status = http_not_found;
		return ret_error;
	}
	
	return ret_ok;
}

ret_t
cherokee_handler_dirlist_free (cherokee_handler_dirlist_t *dhdl)
{
	if (dhdl->dir) {
		closedir (dhdl->dir);
	}
	
	free (dhdl);
	
	return ret_ok;
}


static inline int
entry_is_dir (char *dir, struct dirent *file)
{
	   struct stat f;
	   int dir_len = strlen(dir);
	   
	   char *m = (char *) malloc(dir_len + file->d_reclen + 1);
	   memcpy (m, dir, dir_len);
	   memcpy (m+dir_len, file->d_name, file->d_reclen);
	   m[dir_len + file->d_reclen] = '\0';
	   
	   stat (m, &f);
	   free (m);
	   
	   return S_ISDIR(f.st_mode);
}


ret_t
cherokee_handler_dirlist_step (cherokee_handler_dirlist_t *dhdl, cherokee_buffer_t *buffer)
{
	struct dirent *entry;

	/*                       	 0        1         2         3         4         5         6         7
					 1234567890123456789012345678901234567890123456789012345678901234567890 */
	
	   
	/* Page header
	 */
	if (dhdl->page_begining == 0) {
		cherokee_buffer_add (buffer, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">" CRLF, 57);
		cherokee_buffer_add (buffer, "<html><head><title>Index of ", 28);
		cherokee_buffer_add (buffer, dhdl->request, strlen(dhdl->request));
		cherokee_buffer_add (buffer, "</title></head><body><h1>Index of ", 34);
		cherokee_buffer_add (buffer, dhdl->request, strlen(dhdl->request));
		cherokee_buffer_add (buffer, "</h1><hr><pre>", 14);
		
		dhdl->page_begining = 1;
		return ret_ok;
	}


	/* Page entries
	 */
	while ((entry = readdir (dhdl->dir)) != NULL)
	{ 
		int num = 0;
		int is_a_dir;
		int entry_len = strlen(entry->d_name);
		
		/* Ignore backup files
		 */
		if ((entry->d_name[0] == '.') ||
		    (entry->d_name[0] == '#') ||
		    (entry->d_name[entry_len] == '~')) 
		{
			continue;
		}
		
		/* Add an item
		 */
		is_a_dir = entry_is_dir(dhdl->request, entry);
		
		cherokee_buffer_add (buffer, "<a href=\"", 9); 
		cherokee_buffer_add (buffer, entry->d_name, entry_len);
		if (is_a_dir)
			cherokee_buffer_add (buffer, "/", 1);
		cherokee_buffer_add (buffer, "\">", 2); 
		cherokee_buffer_add (buffer, entry->d_name, entry_len);
		if (is_a_dir)
			cherokee_buffer_add (buffer, "/", 1);
		cherokee_buffer_add (buffer, "</a>\n", 5);			 
		
		/* Add up to 5 each time
		 */
		if (++num >= 5) {
			return ret_ok;
		}
	}

	/* Page ending
	 */
	if (dhdl->page_ending == 0) {
		cherokee_buffer_add (buffer, "</pre><hr><address>Cherokee web server</address></body></html>", 62);
		
		dhdl->page_ending = 1;			 
		return ret_ok;
	}
	
	return ret_eof;
}


ret_t
cherokee_handler_dirlist_add_headers (cherokee_handler_dirlist_t *dhdl, cherokee_buffer_t *buffer)
{
	if (HANDLER(dhdl)->redirect) {
		cherokee_buffer_add (buffer, "Location: ", 10);
		cherokee_buffer_add (buffer, HANDLER(dhdl)->redirect, strlen(HANDLER(dhdl)->redirect));
		cherokee_buffer_add (buffer, CRLF, 2);
	}

	return ret_ok;
}



/* Library init function
 */
void
dirlist_init (void)
{
}
