/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "common.h"
#include "server.h"
#include "connection.h"
#include "log.h"


ret_t
cherokee_server_new  (cherokee_server_t **srv,
		      unsigned int        port)
{
	cherokee_server_t  *n;
	struct sockaddr_in  sockaddr;
	int                 ret;
	int                 flags;
	int                 on = 1;
	
	
	/* Get memory
	 */
	n = (cherokee_server_t*) malloc (sizeof(cherokee_server_t));
	return_if_fail (n!=NULL, ret_nomem);
	
	
	/* Lists
	 */
	INIT_LIST_HEAD(&n->active_list);
	INIT_LIST_HEAD(&n->reuse_list);

	
	/* Options
	 */
	n->log = 0;


	/* Create the socket
	 */
	n->socket = socket (AF_INET, SOCK_STREAM, 0);
	return_if_fail (n->socket > 0, ret_error);
	
	
	/* File descriptors poll
	 */
	cherokee_fdpoll_new(&n->fdpoll);
	cherokee_fdpoll_add(n->fdpoll, n->socket, 0);
	
	
	/* Plugin table
	 */
	cherokee_plugin_table_new (&n->plugins);
	return_if_fail (n->plugins!=NULL, ret_nomem);
	
	
	/* Set 'close-on-exec'
	 */
	fcntl(n->socket, F_SETFD, FD_CLOEXEC);
	
	
	/* To re-bind without wait to TIME_WAIT
	 */
	ret = setsockopt (n->socket, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
	return_if_fail (ret == 0, ret_error);
	
	   
	/* Bind the socket
	 */
	sockaddr.sin_port        = htons (port);
	sockaddr.sin_family      = AF_INET;
	sockaddr.sin_addr.s_addr = INADDR_ANY;
	
	ret = bind (n->socket, (struct sockaddr *)&sockaddr, sizeof(sockaddr));
	if (ret != 0) {
		fprintf (stderr, "Can't bind() socket (port %d)\n", port);
		return ret_error;
	}

	   
	/* Set no-delay mode
	 */
	flags = fcntl (n->socket, F_GETFL, 0);
	return_if_fail (flags != -1, ret_error);
	
	ret = fcntl (n->socket, F_SETFL, flags | O_NDELAY);
	return_if_fail (ret >= 0, ret_error);
	
	
	/* Listen
	 */
	ret = listen (n->socket, 5);
	if (ret < 0) {
		close (n->socket);
		return_if_fail (0, ret_error);
	}
	
	   
	/* Return the object
	 */
	*srv = n;
	
	return ret_ok;
}


ret_t
cherokee_server_free (cherokee_server_t *srv)
{
	cherokee_fdpoll_free (srv->fdpoll);
	cherokee_plugin_table_free (srv->plugins);
	
	free (srv);
	
	return ret_ok;
}


ret_t 
cherokee_server_log (cherokee_server_t *srv, int active)
{
	/* Shutdown loggin
	 */
	if (srv->log && !active) {
		srv->log = active;
		cherokee_log (LOG_NOTICE, "Closing log");
		return cherokee_log_close();
	}
	
	/* Active logging
	 */
	if (active && !srv->log) {
		ret_t ret;
		srv->log = active;
		ret = cherokee_log_init();
		cherokee_log (LOG_NOTICE, "Starting log");
		return ret;
	}

	return ret_error;
}


static inline int
look_for_new_connection (cherokee_server_t *srv)
{
        /* Look for new connections
	 */
	if (cherokee_fdpoll_check (srv->fdpoll, srv->socket))
	{
		int new_socket;
		cherokee_connection_t *new_connection;
		
		
		/* Get the new socket connection
		 */
		new_socket = accept (srv->socket, NULL, NULL);
		cherokee_fdpoll_add (srv->fdpoll, new_socket, 0);
		
		
		/* Look for a reusable connection
		 */
		if (list_empty (&srv->reuse_list)) {
			cherokee_connection_new (&new_connection);
		} else {
			new_connection = CONN(srv->reuse_list.next);
			list_del ((list_t *)new_connection);
		}
		
		
		/* Initial state is "reading"
		 */
		new_connection->status = stat_reading;
		
		
		/* Set connection info
		 */
		cherokee_connection_set_socket (new_connection, new_socket);
		
		
		/* Add to active connection list
		 */
		list_add ((list_t *)new_connection, &srv->active_list);
		
		return 1;
	}
	
	return 0;
}


static void
purge_closed_connection (cherokee_server_t *srv, cherokee_connection_t *conn)
{
	   /* Delete from File descriptors poll
	    */
	   cherokee_fdpoll_del (srv->fdpoll, conn->socket);

	   /* Close
	    */
	   conn->status = stat_closed;
	   cherokee_connection_close (conn);
	   
	   /* Delete from active connections list
	    */
	   list_del ((list_t *)conn);
	   
	   /* Add to reusable connections list
	    */
	   cherokee_connection_clean (conn);
	   list_add ((list_t *)conn, &srv->reuse_list);
}


static inline void
process_active_connections (cherokee_server_t *srv)
{
	ret_t  ret;
	cherokee_connection_t *conn;
	list_t *i, *tmp;
	
	/* Process active connections
	 */
	list_for_each_safe (i, tmp, (list_t*)&srv->active_list) {
		conn = CONN(i);			 											   
		
		if (cherokee_fdpoll_check (srv->fdpoll, conn->socket) == 0)
			continue;
		
		/* Reading
		 */
		switch (conn->status) {
		case stat_reading:
		{	
			cherokee_connection_recv (conn);
			
			if (cherokee_connection_eoh (conn)) {
				func_new_t handler_new_func;
				cherokee_handler_t *hdl;
				
				/* Turn the connection in write mode
				 */
				conn->status = stat_writing;
				ret = cherokee_fdpoll_set_mode (srv->fdpoll, conn->socket, 1);
				if (ret != ret_ok) {
					SHOULDNT_HAPPEN;
				}
				
				/* Parse headers
				 */
				ret = cherokee_connection_parse_header (conn);
				if (ret != ret_ok) {
					continue;
				}
				

				/* Maybe log it
				 */
				cherokee_log (LOG_INFO, conn->request);


				/* Look for the handler "*_new" function
				 */
				handler_new_func = (func_new_t) cherokee_plugin_table_get_from_url (srv->plugins, conn->request);
				if (handler_new_func == NULL) {
					ret = ret_error;
					continue;
				}
				
				/* Look for the request
				 */
				ret = cherokee_connection_open_request (conn, handler_new_func);
			}
			
			/* Check security
			 */
			ret = cherokee_connection_reading_check (conn);
			if (ret != ret_ok) {
				conn->status = stat_writing;
				continue;
			}
		}
		break;
		
		/* Writing
		 */
		case stat_writing:
		{
			/* Header
			 */
			if (conn->header_sent == 0) {
				ret = cherokee_connection_send_header (conn);
				
				if (ret == ret_ok) {
					conn->header_sent = 1;
					
					if (conn->method == http_head) {
						purge_closed_connection (srv, conn);
					}
					
					if (conn->error_code != http_ok) {
						cherokee_connection_send_response_page (conn);
						purge_closed_connection (srv, conn);
					}
				}
				
			/* Body
			 */
			} else {
				ret = cherokee_connection_step (conn);
				
				switch (ret) {
				case ret_ok:
					ret = cherokee_connection_send (conn);
					
					if (ret == ret_eof)
						purge_closed_connection (srv, conn);
					break;
				case ret_eof:
					purge_closed_connection (srv, conn);
					break;
				case ret_error:
					purge_closed_connection (srv, conn);
					break;
				default:
					SHOULDNT_HAPPEN;
				}
			}
		}
		break;
		
		default:
			SHOULDNT_HAPPEN;
			
		} /* switch (conn->staus) */
	} /* list_for_each */
}



ret_t 
cherokee_server_set (cherokee_server_t *srv, char *dir, char *handler)
{
	   ret_t ret;

	   ret = cherokee_plugin_table_set (srv->plugins, dir, handler);

	   if (ret != ret_ok) {
			 fprintf (stderr, "Can't load handler '%s': ", handler);
			 switch (ret) {
			 case ret_file_not_found:
				    fprintf (stderr, "file not found\n");
				    break;
			 default:
				    fprintf (stderr, "unknown error\n"); 
			 }
	   }

	   return ret;
}


void
cherokee_server_step (cherokee_server_t *srv)
{
	int ret;
	
	ret = cherokee_fdpoll_watch (srv->fdpoll, 999);
	
	if (ret > 0) {
		/* Look for new connections
		 */	   
		ret = look_for_new_connection(srv);
		if (ret) return;
		
		/* Active connections
		 */
		process_active_connections(srv);
	}	   
}


