/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001-2002 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include "fdpoll.h"

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#ifdef HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#else
#include <resource.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif


/*
** Copyright (c) 1999,2000 by Jef Poskanzer <jef@acme.com>
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
*/

static inline int
get_fd_limit (void)
{
	   int nfiles;
	   struct rlimit rl;

	   if (getrlimit (RLIMIT_NOFILE, &rl) == 0)
	   {
			 nfiles = rl.rlim_cur;

			 if (rl.rlim_max == RLIM_INFINITY) {
                        /* arbitrary */
				    rl.rlim_cur = 8192;  
			 } else if (rl.rlim_max > rl.rlim_cur) {
				    rl.rlim_cur = rl.rlim_max;
			 }
			 
			 if (setrlimit( RLIMIT_NOFILE, &rl ) == 0) {
				    nfiles = rl.rlim_cur;
			 }

			 return nfiles;
	   }

	   return getdtablesize();
}


ret_t
cherokee_fdpoll_new (cherokee_fdpoll_t **fdp)
{
	   cherokee_fdpoll_t *n;

	   /* Get memory
	    */
	   n = (cherokee_fdpoll_t *)malloc(sizeof(cherokee_fdpoll_t));
	   return_if_fail (n != NULL, ret_nomem);


	   /* Look for max fd limit
	    */
	   n->nfiles = get_fd_limit();
	   
	   
	   /* Get memory
	    */
	   n->fd_rw = (int*) malloc (sizeof(int) * n->nfiles);
	   return_if_fail (n->fd_rw, ret_nomem);
	   
	   n->pollfds = (struct pollfd*) malloc (sizeof(struct pollfd) * n->nfiles);
	   return_if_fail (n->pollfds, ret_nomem);
	   
	   n->poll_fdidx = (int*) malloc (sizeof(int) * n->nfiles);
	   return_if_fail (n->poll_fdidx, ret_nomem);
	   
	   n->poll_rfdidx = (int*) malloc (sizeof(int) * n->nfiles);
	   return_if_fail (n->poll_rfdidx, ret_nomem);


	   /* Return it
	    */
	   *fdp = n;

	   return ret_ok;
}


ret_t
cherokee_fdpoll_free (cherokee_fdpoll_t *fdp)
{
	   free (fdp->fd_rw);
	   free (fdp->pollfds);
	   free (fdp->poll_fdidx);
	   free (fdp->poll_rfdidx);

	   free (fdp);

	   return ret_ok;
}


ret_t
cherokee_fdpoll_add (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	   fdp->pollfds[fdp->npollfds].fd = fd;

	   switch (rw) {
	   case fdp_read:
			 fdp->pollfds[fdp->npollfds].events = POLLIN;
			 break;
	   case fdp_write:
			 fdp->pollfds[fdp->npollfds].events = POLLOUT;
			 break;
	   default: break;
	   }
	   
	   fdp->poll_fdidx[fd] = fdp->npollfds;
	   fdp->npollfds++;

	   fdp->fd_rw[fd] = rw;

	   return ret_ok;
}


ret_t
cherokee_fdpoll_set_mode (cherokee_fdpoll_t *fdp, int fd, int rw)
{
	   int fdidx = fdp->poll_fdidx[fd];

	   if (fdidx < 0 || fdidx >= fdp->nfiles)
			 return ret_error;

	   fdp->pollfds[fdidx].events = (rw ? POLLOUT: POLLIN);
	   fdp->fd_rw[fd] = (rw ? fdp_write : fdp_read);
	  
	   return ret_ok;
}


ret_t
cherokee_fdpoll_del (cherokee_fdpoll_t *fdp, int fd)
{
	   int idx = fdp->poll_fdidx[fd];

	   if (idx < 0 || idx >= fdp->nfiles)
			 return ret_error;

	   fdp->npollfds--;
	   fdp->pollfds[idx] = fdp->pollfds[fdp->npollfds];
	   fdp->poll_fdidx[fdp->pollfds[idx].fd] = idx;

	   return ret_ok;
}


ret_t
cherokee_fdpoll_check (cherokee_fdpoll_t *fdp, int fd)
{
	   int fdidx = fdp->poll_fdidx[fd];
	   
	   if (fdidx < 0 || fdidx >= fdp->nfiles)
			 return ret_error;

	   if (fdp->pollfds[fdidx].revents & POLLERR)
			 return ret_error;
	   
	   switch (fdp->fd_rw[fd]) {
	   case fdp_read:
			 return fdp->pollfds[fdidx].revents & (POLLIN | POLLERR | POLLHUP | POLLNVAL);
	   case fdp_write:
			 return fdp->pollfds[fdidx].revents & (POLLOUT | POLLERR | POLLHUP | POLLNVAL);
	   default:
			 return ret_error;
	   }

	   return ret_ok;
}


int
cherokee_fdpoll_watch (cherokee_fdpoll_t *fdp, int timeout_msecs)
{
	   int r, ridx, i;
	   
	   r = poll (fdp->pollfds, fdp->npollfds, timeout_msecs);
	   if (r == -1)
			 return -1;

	   ridx = 0;
	   for (i = 0; i < fdp->npollfds; ++i)
			 if (fdp->pollfds[i].revents &
				(POLLIN | POLLOUT | POLLERR | POLLHUP | POLLNVAL))
				    fdp->poll_rfdidx[ridx++] = fdp->pollfds[i].fd;

        /* not r */
	   return ridx;        
}

