/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.encrypt;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.deprecation.log.DeprecationLogger;
import org.apache.nifi.deprecation.log.DeprecationLoggerFactory;
import org.apache.nifi.encrypt.EncryptionException;
import org.apache.nifi.encrypt.PropertyEncryptionMethod;
import org.apache.nifi.encrypt.PropertySecretKeyProvider;
import org.apache.nifi.security.util.KeyDerivationFunction;
import org.apache.nifi.security.util.crypto.Argon2SecureHasher;
import org.apache.nifi.security.util.crypto.KeyDerivationBcryptSecureHasher;
import org.apache.nifi.security.util.crypto.PBKDF2SecureHasher;
import org.apache.nifi.security.util.crypto.ScryptSecureHasher;
import org.apache.nifi.security.util.crypto.SecureHasher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class StandardPropertySecretKeyProvider
implements PropertySecretKeyProvider {
    private static final Logger LOGGER = LoggerFactory.getLogger(StandardPropertySecretKeyProvider.class);
    private static final Charset PASSWORD_CHARSET = StandardCharsets.UTF_8;
    private static final int MINIMUM_PASSWORD_LENGTH = 12;
    private static final String PASSWORD_LENGTH_MESSAGE = String.format("Key Password length less than required [%d]", 12);
    private static final String SECRET_KEY_ALGORITHM = "AES";

    StandardPropertySecretKeyProvider() {
    }

    @Override
    public SecretKey getSecretKey(PropertyEncryptionMethod propertyEncryptionMethod, String password) {
        Objects.requireNonNull(propertyEncryptionMethod, "Property Encryption Method is required");
        Objects.requireNonNull(password, "Password is required");
        if (StringUtils.length((CharSequence)password) < 12) {
            throw new EncryptionException(PASSWORD_LENGTH_MESSAGE);
        }
        KeyDerivationFunction keyDerivationFunction = propertyEncryptionMethod.getKeyDerivationFunction();
        int keyLength = propertyEncryptionMethod.getKeyLength();
        LOGGER.debug("Generating [{}-{}] Secret Key using [{}]", new Object[]{SECRET_KEY_ALGORITHM, keyLength, keyDerivationFunction.getKdfName()});
        StandardPropertySecretKeyProvider.logDeprecated(propertyEncryptionMethod);
        SecureHasher secureHasher = StandardPropertySecretKeyProvider.getSecureHasher(propertyEncryptionMethod);
        byte[] passwordBinary = password.getBytes(PASSWORD_CHARSET);
        byte[] hash = secureHasher.hashRaw(passwordBinary);
        return new SecretKeySpec(hash, SECRET_KEY_ALGORITHM);
    }

    private static SecureHasher getSecureHasher(PropertyEncryptionMethod propertyEncryptionMethod) {
        KeyDerivationFunction keyDerivationFunction = propertyEncryptionMethod.getKeyDerivationFunction();
        int hashLength = propertyEncryptionMethod.getHashLength();
        if (KeyDerivationFunction.ARGON2.equals((Object)keyDerivationFunction)) {
            return new Argon2SecureHasher(Integer.valueOf(hashLength));
        }
        if (KeyDerivationFunction.BCRYPT.equals((Object)keyDerivationFunction)) {
            return new KeyDerivationBcryptSecureHasher(hashLength);
        }
        if (KeyDerivationFunction.PBKDF2.equals((Object)keyDerivationFunction)) {
            return new PBKDF2SecureHasher(hashLength);
        }
        if (KeyDerivationFunction.SCRYPT.equals((Object)keyDerivationFunction)) {
            return new ScryptSecureHasher(hashLength);
        }
        String message = String.format("Key Derivation Function [%s] not supported", keyDerivationFunction.getKdfName());
        throw new EncryptionException(message);
    }

    private static void logDeprecated(PropertyEncryptionMethod method) {
        DeprecationLogger deprecationLogger = DeprecationLoggerFactory.getLogger(StandardPropertySecretKeyProvider.class);
        PropertyEncryptionMethod recommendedMethod = PropertyEncryptionMethod.NIFI_PBKDF2_AES_GCM_256;
        KeyDerivationFunction keyDerivationFunction = method.getKeyDerivationFunction();
        int keyLength = method.getKeyLength();
        if (KeyDerivationFunction.BCRYPT == keyDerivationFunction || KeyDerivationFunction.SCRYPT == keyDerivationFunction) {
            deprecationLogger.warn("Sensitive Properties Algorithm [{}] is deprecated in favor of [{}]", new Object[]{method, recommendedMethod});
        } else if (keyLength == 128) {
            deprecationLogger.warn("Sensitive Properties Algorithm [{}] Key Length [{}] should be upgraded to Key Length [{}]", new Object[]{method, keyLength, recommendedMethod.getKeyLength()});
        }
    }
}

