/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.myfaces.orchestra.viewController;

/**
 * A {@link org.apache.myfaces.orchestra.viewController.ViewControllerManager} implementation which uses
 * annotations on backing beans to determine the beans responsible for a given view and execute
 * the appropriate annotated methods.
 *
 * <p>When using Spring, every bean declaration in the spring config files is checked to see if the
 * referenced class has annotations, and if so that information is cached. Here, that information is
 * then used to locate a bean which has a ViewController annotation that references the current view.</p>
 * 
 * <p>See also org.apache.myfaces.orchestra.viewController.annotations.*.</p>
 *
 * <p>
 * Notice: For backward compatiblity with the Orchestra-core module this ViewControllerManager
 * will take the Orchestra-core naming strategy
 * into account and uses reflection if no annotated method were found.
 * </p>
 * 
 * @see org.apache.myfaces.orchestra.viewController.ViewControllerManager
 * @see org.apache.myfaces.orchestra.viewController.PlainAnnotationsViewControllerManager
 */
public class AnnotationsViewControllerManager extends AbstractAnnotationsViewControllerManager
{
    private ViewControllerNameMapper viewControllerNameMapper;
    private ViewControllerExecutor viewControllerExecutor;

    public AnnotationsViewControllerManager()
    {
    }

    public void initManager()
    {
        // Set things up so that finding a bean-name for a viewId looks first using an
        // AnnotationsViewControllerNameMapper, then a DefaultViewControllerNameMapper.
        viewControllerNameMapper =
            new CompositeViewControllerNameMapper(
                new ViewControllerNameMapper[]
                    {
                        new AnnotationsViewControllerNameMapper(getAnnotationInfoManager()),
                        new DefaultViewControllerNameMapper()
                    });

        // Set things up so that executing a method first using an
        // AnnotationsViewControllerExecutor, then a ReflectiveViewControllerExecutor.
        viewControllerExecutor =
            new CompositeViewControllerExecutor(
                new ViewControllerExecutor[]
                    {
                        new AnnotationsViewControllerExecutor(getAnnotationInfoManager()),
                        new ReflectiveViewControllerExecutor()
                    });
    }

    protected ViewControllerNameMapper getViewControllerNameMapper()
    {
        return viewControllerNameMapper;
    }

    protected ViewControllerExecutor getViewControllerExecutor()
    {
        return viewControllerExecutor;
    }
}
