/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.frameworkAdapter.jsf;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.myfaces.orchestra.conversation.ConversationMessager;
import org.apache.myfaces.orchestra.conversation.jsf.JsfConversationMessager;
import org.apache.myfaces.orchestra.frameworkAdapter.FrameworkAdapter;
import org.apache.myfaces.orchestra.frameworkAdapter._FrameworkAdapterUtils;
import org.apache.myfaces.orchestra.lib.OrchestraException;

import javax.faces.component.UIViewRoot;
import javax.faces.context.FacesContext;

import java.io.IOException;

/**
 * An implementation of the FrameworkAdapter for JSF environments.
 * <p>
 * This class defaults to using a JsfConversationMessager instance. 
 */
public class JsfFrameworkAdapter extends FrameworkAdapter
{
    private final Log log = LogFactory.getLog(JsfFrameworkAdapter.class);

    public JsfFrameworkAdapter(String conversationMessagerClass)
    {
        ConversationMessager cm = _FrameworkAdapterUtils.createConversationMessager(conversationMessagerClass,
                JsfConversationMessager.class);
        setConversationMessager(cm);
    }

    /**
     * This method should be invoked at the start of each JSF request cycle, before any Orchestra
     * functionality is invoked.
     * 
     * @since 1.1
     */
    public void beginRequest()
    {
        log.debug("Beginning request");
        FrameworkAdapter.setCurrentInstance(this);
    }

    /**
     * This method should be invoked at the end of each JSF request cycle, after the last
     * Orchestra functionality is invoked.
     * 
     * @since 1.1
     */
    public void endRequest()
    {
        log.debug("Ending request");
        FrameworkAdapter.setCurrentInstance(null);
    }

    protected ConversationMessager createDefaultConversationMessager()
    {
        return new JsfConversationMessager();
    }

    protected FacesContext getFacesContext()
    {
        FacesContext fc = FacesContext.getCurrentInstance();
        if (fc == null)
        {
            throw new OrchestraException("Missing FacesContext");
        }
        return fc;
    }

    public String getInitParameter(String key)
    {
        FacesContext context = getFacesContext();

        return context.getExternalContext().getInitParameter(key);
    }

    public Object getRequestParameterAttribute(String key)
    {
        FacesContext context = getFacesContext();
        return context.getExternalContext().getRequestParameterMap().get(key);
    }

    public boolean containsRequestParameterAttribute(String key)
    {
        FacesContext context = getFacesContext();
        return context.getExternalContext().getRequestParameterMap().containsKey(key);
    }

    public Object getRequestAttribute(String key)
    {
        FacesContext context = getFacesContext();
        return context.getExternalContext().getRequestMap().get(key);
    }

    public void setRequestAttribute(String key, Object value)
    {
        FacesContext context = getFacesContext();
        context.getExternalContext().getRequestMap().put(key, value);
    }

    public boolean containsRequestAttribute(String key)
    {
        FacesContext context = getFacesContext();
        return context.getExternalContext().getRequestMap().containsKey(key);
    }

    public Object getSessionAttribute(String key)
    {
        FacesContext context = getFacesContext();
        return context.getExternalContext().getSessionMap().get(key);
    }

    public void setSessionAttribute(String key, Object value)
    {
        FacesContext context = getFacesContext();
        context.getExternalContext().getSessionMap().put(key, value);
    }

    public boolean containsSessionAttribute(String key)
    {
        FacesContext context = getFacesContext();
        return context.getExternalContext().getSessionMap().containsKey(key);
    }

    protected String getRequestContextPath()
    {
        FacesContext context = getFacesContext();
        return context.getExternalContext().getRequestContextPath();
    }

    public void redirect(String url) throws IOException
    {
        StringBuffer redir = new StringBuffer();
        if (url.startsWith("/"))
        {
            redir.append(getRequestContextPath());
        }
        redir.append(url);


        FacesContext context = getFacesContext();

        String actionUrl = context.getExternalContext().encodeActionURL(redir.toString());
        context.getExternalContext().redirect(actionUrl);
        context.responseComplete();
    }

    public Object getBean(String name)
    {
        FacesContext context = getFacesContext();
        if (context == null)
        {
            throw new IllegalStateException("getBean invoked before FacesServlet");
        }

        try
        {
            return context.getApplication()
                .getVariableResolver().resolveVariable(context, name);
        }
        catch(RuntimeException e)
        {
            // Some libraries don't always report the name of the bean when a lookup
            // failure occurs. In particular, Spring does not do this when the associated
            // scope does not exist.
            log.error("Failed to resolve variable [" + name + "]", e);
            throw e;
        }
    }

    public void invokeNavigation(String navigationName)
    {
        FacesContext context = getFacesContext();

        context.getApplication().getNavigationHandler().handleNavigation(context, null, navigationName);
    }

    /**
     * Return the current JSF viewId.
     * <p>
     * Null is returned if there is not yet a UIViewRoot set up for the current request.
     */
    public String getCurrentViewId()
    {
        UIViewRoot vr = getFacesContext().getViewRoot();
        if (vr == null)
        {
            return null;
        }
        
        return vr.getViewId();
    }
}
