"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Subject_1 = require("rxjs/Subject");
const empty_1 = require("rxjs/observable/empty");
const of_1 = require("rxjs/observable/of");
const throw_1 = require("rxjs/observable/throw");
const exception_1 = require("../../exception/exception");
const path_1 = require("../path");
class SimpleMemoryHost {
    constructor() {
        this._cache = new Map();
        this._watchers = new Map();
    }
    _isDir(path) {
        for (const p of this._cache.keys()) {
            if (p.startsWith(path + path_1.NormalizedSep)) {
                return true;
            }
        }
        return false;
    }
    _updateWatchers(path, type) {
        const time = new Date();
        let currentPath = path;
        let parent = null;
        if (this._watchers.size == 0) {
            // Nothing to do if there's no watchers.
            return;
        }
        const maybeWatcher = this._watchers.get(currentPath);
        if (maybeWatcher) {
            maybeWatcher.forEach(watcher => {
                const [options, subject] = watcher;
                subject.next({ path, time, type });
                if (!options.persistent && type == 2 /* Deleted */) {
                    subject.complete();
                    this._watchers.delete(currentPath);
                }
            });
        }
        do {
            currentPath = parent !== null ? parent : currentPath;
            parent = path_1.dirname(currentPath);
            const maybeWatcher = this._watchers.get(currentPath);
            if (maybeWatcher) {
                maybeWatcher.forEach(watcher => {
                    const [options, subject] = watcher;
                    if (!options.recursive) {
                        return;
                    }
                    subject.next({ path, time, type });
                    if (!options.persistent && type == 2 /* Deleted */) {
                        subject.complete();
                        this._watchers.delete(currentPath);
                    }
                });
            }
        } while (parent != currentPath);
    }
    get capabilities() {
        return { synchronous: true };
    }
    write(path, content) {
        if (this._isDir(path)) {
            return throw_1._throw(new exception_1.PathIsDirectoryException(path));
        }
        const existed = this._cache.has(path);
        this._cache.set(path, content);
        this._updateWatchers(path, existed ? 0 /* Changed */ : 1 /* Created */);
        return empty_1.empty();
    }
    read(path) {
        if (this._isDir(path)) {
            return throw_1._throw(new exception_1.PathIsDirectoryException(path));
        }
        const maybeBuffer = this._cache.get(path);
        if (!maybeBuffer) {
            return throw_1._throw(new exception_1.FileDoesNotExistException(path));
        }
        else {
            return of_1.of(maybeBuffer);
        }
    }
    delete(path) {
        if (this._isDir(path)) {
            for (const [cachePath, _] of this._cache.entries()) {
                if (path.startsWith(cachePath + path_1.NormalizedSep)) {
                    this._cache.delete(cachePath);
                }
            }
        }
        else {
            this._cache.delete(path);
        }
        this._updateWatchers(path, 2 /* Deleted */);
        return empty_1.empty();
    }
    rename(from, to) {
        if (!this._cache.has(from)) {
            return throw_1._throw(new exception_1.FileDoesNotExistException(from));
        }
        else if (this._cache.has(to)) {
            return throw_1._throw(new exception_1.FileAlreadyExistException(from));
        }
        if (this._isDir(from)) {
            for (const path of this._cache.keys()) {
                if (path.startsWith(from + path_1.NormalizedSep)) {
                    const content = this._cache.get(path);
                    if (content) {
                        this._cache.set(path_1.join(to, path_1.NormalizedSep, path.slice(from.length)), content);
                    }
                }
            }
        }
        else {
            const content = this._cache.get(from);
            if (content) {
                this._cache.delete(from);
                this._cache.set(to, content);
            }
        }
        this._updateWatchers(from, 3 /* Renamed */);
        return empty_1.empty();
    }
    list(path) {
        if (this._cache.has(path)) {
            return throw_1._throw(new exception_1.PathIsFileException(path));
        }
        const fragments = path_1.split(path);
        const result = new Set();
        for (const p of this._cache.keys()) {
            if (p.startsWith(path + path_1.NormalizedSep)) {
                result.add(path_1.split(p)[fragments.length]);
            }
        }
        return of_1.of([...result]);
    }
    exists(path) {
        return of_1.of(this._cache.has(path) || this._isDir(path));
    }
    isDirectory(path) {
        return of_1.of(this._isDir(path));
    }
    isFile(path) {
        return of_1.of(this._cache.has(path));
    }
    stats(_path) {
        return null;
    }
    watch(path, options) {
        const subject = new Subject_1.Subject();
        let maybeWatcherArray = this._watchers.get(path);
        if (!maybeWatcherArray) {
            maybeWatcherArray = [];
            this._watchers.set(path, maybeWatcherArray);
        }
        maybeWatcherArray.push([options || {}, subject]);
        return subject.asObservable();
    }
}
exports.SimpleMemoryHost = SimpleMemoryHost;
//# sourceMappingURL=data:application/json;base64,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