/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.rfc3986;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.jena.rfc3986.AlgResolveIRI;
import org.apache.jena.rfc3986.Chars3986;
import org.apache.jena.rfc3986.IRI;
import org.apache.jena.rfc3986.IRIParseException;
import org.apache.jena.rfc3986.Issue;
import org.apache.jena.rfc3986.LibParseIRI;
import org.apache.jena.rfc3986.ParseDID;
import org.apache.jena.rfc3986.ParseErrorIRI3986;
import org.apache.jena.rfc3986.ParseIPv6Address;
import org.apache.jena.rfc3986.ParseOID;
import org.apache.jena.rfc3986.ParseURN;
import org.apache.jena.rfc3986.ParseURNComponents;
import org.apache.jena.rfc3986.RFC3986;
import org.apache.jena.rfc3986.Severity;
import org.apache.jena.rfc3986.URIScheme;
import org.apache.jena.rfc3986.Violation;
import org.apache.jena.rfc3986.Violations;

public class IRI3986
implements IRI {
    private final String iriStr;
    private final int length;
    private int scheme0 = -1;
    private int scheme1 = -1;
    private String scheme = null;
    private int authority0 = -1;
    private int authority1 = -1;
    private String authority = null;
    private int userinfo0 = -1;
    private int userinfo1 = -1;
    private int host0 = -1;
    private int host1 = -1;
    private String host = null;
    private int port0 = -1;
    private int port1 = -1;
    private String port = null;
    private int path0 = -1;
    private int path1 = -1;
    private String path = null;
    private int query0 = -1;
    private int query1 = -1;
    private String query = null;
    private int fragment0 = -1;
    private int fragment1 = -1;
    private String fragment = null;
    private List<Violation> reports = null;
    private static final boolean strictResolver = false;
    private static final Pattern authorityRegex = Pattern.compile("(([^/?#@]*)@)?(\\[[^/?#]*\\]|([^/?#:]*))?(:([^/?#]*)?)?");
    private static Pattern UUID_PATTERN_LC = Pattern.compile("^uuid:[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$");
    private static Pattern URN_UUID_PATTERN_LC = Pattern.compile("^urn:uuid:[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$");
    private static Pattern UUID_PATTERN_AnyCase_PREFIX = Pattern.compile("^(?:urn:uuid|uuid):[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}", 2);
    private final int UUID_length = 36;
    private final int UUID_scheme_path_length = 36;
    private final int URN_UUID_scheme_path_length = 36 + "uuid:".length();

    static void checkSyntax(String iristr) {
        IRI3986.newAndParseEx(iristr);
    }

    public static IRI3986 create(String iristr) {
        IRI3986 iri = IRI3986.newAndParseAndCheck(iristr);
        return iri;
    }

    public static IRI3986 createSyntax(String iristr) {
        IRI3986 iri = IRI3986.newAndParseAndCheck(iristr);
        return iri;
    }

    public static IRI3986 createAny(String iristr) {
        IRI3986 iri = IRI3986.newAndCheck(iristr);
        return iri;
    }

    private static IRI3986 newAndParseEx(String iristr) {
        IRI3986 iri = new IRI3986(iristr);
        iri.parse();
        return iri;
    }

    private static IRI3986 newAndParseAndCheck(String iristr) {
        IRI3986 iri = new IRI3986(iristr);
        iri.parse();
        iri.schemeSpecificRulesInternal();
        return iri;
    }

    private static IRI3986 newAndCheck(String iriStr) {
        IRI3986 iri = new IRI3986(iriStr);
        try {
            iri.parse();
            iri.schemeSpecificRulesInternal();
        }
        catch (IRIParseException ex) {
            String msg = ex.getMessage();
            IRI3986.addReportParseError(iri, iriStr, ex.getMessage());
        }
        return iri;
    }

    private IRI3986(String iriStr) {
        this.iriStr = iriStr;
        this.length = iriStr.length();
    }

    @Override
    public final String str() {
        if (this.iriStr != null) {
            return this.iriStr;
        }
        return this.rebuild();
    }

    public boolean hasViolations() {
        return this.reports != null && !this.reports.isEmpty();
    }

    public boolean hasViolations(Severity levelSeverity) {
        if (!this.hasViolations()) {
            return false;
        }
        for (Violation violation : this.reports) {
            Severity severity = Violations.getSeverity(violation.issue());
            if (severity.level() <= levelSeverity.level()) continue;
            return true;
        }
        return false;
    }

    public void forEachViolation(Consumer<Violation> action) {
        if (this.reports == null) {
            return;
        }
        this.reports.forEach(action);
    }

    public List<Violation> violations() {
        if (this.reports == null) {
            return List.of();
        }
        return this.reports;
    }

    @Override
    public String toString() {
        return this.str();
    }

    @Override
    public boolean hasScheme() {
        return this.scheme0 != -1;
    }

    @Override
    public String scheme() {
        if (this.hasScheme() && this.scheme == null) {
            this.scheme = IRI3986.part(this.iriStr, this.scheme0, this.scheme1);
        }
        return this.scheme;
    }

    @Override
    public boolean hasAuthority() {
        return this.authority0 != -1;
    }

    @Override
    public String authority() {
        if (this.hasAuthority() && this.authority == null) {
            this.authority = IRI3986.part(this.iriStr, this.authority0, this.authority1);
        }
        return this.authority;
    }

    @Override
    public boolean hasUserInfo() {
        return this.userinfo0 != -1;
    }

    @Override
    public String userInfo() {
        return IRI3986.part(this.iriStr, this.userinfo0, this.userinfo1);
    }

    @Override
    public boolean hasHost() {
        return this.host0 != -1;
    }

    @Override
    public String host() {
        if (this.hasHost() && this.host == null) {
            this.host = IRI3986.part(this.iriStr, this.host0, this.host1);
        }
        return this.host;
    }

    @Override
    public boolean hasPort() {
        return this.port0 != -1;
    }

    @Override
    public String port() {
        if (this.hasPort() && this.port == null) {
            this.port = IRI3986.part(this.iriStr, this.port0, this.port1);
        }
        return this.port;
    }

    @Override
    public boolean hasPath() {
        return true;
    }

    @Override
    public String path() {
        if (this.hasPath() && this.path == null) {
            this.path = IRI3986.part(this.iriStr, this.path0, this.path1);
        }
        if (this.path == null) {
            this.path = "";
        }
        return this.path;
    }

    @Override
    public String[] pathSegments() {
        String x = this.path();
        if (x == null) {
            return null;
        }
        return x.split("/");
    }

    @Override
    public boolean hasQuery() {
        return this.query0 != -1;
    }

    @Override
    public String query() {
        if (this.hasQuery() && this.query == null) {
            this.query = IRI3986.part(this.iriStr, this.query0, this.query1);
        }
        return this.query;
    }

    @Override
    public boolean hasFragment() {
        return this.fragment0 != -1;
    }

    @Override
    public String fragment() {
        if (this.hasFragment() && this.fragment == null) {
            this.fragment = IRI3986.part(this.iriStr, this.fragment0, this.fragment1);
        }
        return this.fragment;
    }

    @Override
    public boolean isAbsolute() {
        return this.hasScheme() && !this.hasFragment();
    }

    @Override
    public boolean isRelative() {
        return !this.hasScheme();
    }

    @Override
    public boolean isRootless() {
        return this.hasScheme() && !this.hasAuthority() && this.rootlessPath();
    }

    @Override
    public boolean isHierarchical() {
        return this.hasScheme() && this.hasAuthority() && this.hierarchicalPath();
    }

    private boolean hierarchicalPath() {
        return this.hasPath() && this.firstChar(this.path0, this.path1) == '/';
    }

    private boolean rootlessPath() {
        return this.hasPath() && this.firstChar(this.path0, this.path1) != '/';
    }

    private static String part(String str, int start, int finish) {
        if (start >= 0) {
            if (finish > str.length()) {
                return str.substring(start);
            }
            return str.substring(start, finish);
        }
        return null;
    }

    private static int contains(String str, char character, int start, int finish) {
        for (int i = start; i < finish; ++i) {
            char ch = str.charAt(i);
            if (ch != character) continue;
            return i;
        }
        return -1;
    }

    private char firstChar(int x0, int x1) {
        if (x0 < 0) {
            return '\uffff';
        }
        if (x1 < 0) {
            return '\uffff';
        }
        if (x0 > x1) {
            return '\uffff';
        }
        return this.charAt(x0);
    }

    public boolean isRFC3986() {
        return IRI3986.isASCII(this.iriStr, 0, this.iriStr.length());
    }

    @Override
    public IRI3986 normalize() {
        String scheme = this.scheme();
        String authority = this.authority();
        String path = this.path();
        String query = this.query();
        String fragment = this.fragment();
        scheme = this.toLowerCase(scheme);
        authority = this.toLowerCase(authority);
        authority = this.normalizePercent(authority);
        path = this.normalizePercent(path);
        query = this.normalizePercent(query);
        fragment = this.normalizePercent(fragment);
        if (path != null) {
            path = AlgResolveIRI.remove_dot_segments(path);
        }
        if (path == null || path.isEmpty()) {
            path = "/";
        }
        if (authority != null && authority.endsWith(":")) {
            authority = authority.substring(0, authority.length() - 1);
        }
        if (Objects.equals("http", scheme)) {
            if (authority != null && authority.endsWith(":80")) {
                authority = authority.substring(0, authority.length() - 3);
            }
        } else if (Objects.equals("https", scheme) && authority != null && authority.endsWith(":443")) {
            authority = authority.substring(0, authority.length() - 4);
        }
        if (Objects.equals(scheme, this.scheme()) && Objects.equals(authority, this.authority()) && Objects.equals(path, this.path()) && Objects.equals(query, this.query()) && Objects.equals(fragment, this.fragment())) {
            return this;
        }
        String s = IRI3986.rebuild(scheme, authority, path, query, fragment);
        return IRI3986.newAndCheck(s);
    }

    private String normalizePercent(String str) {
        if (str == null) {
            return str;
        }
        int idx = str.indexOf(37);
        if (idx < 0) {
            return str;
        }
        int len = str.length();
        StringBuilder sb = new StringBuilder(len);
        for (int i = 0; i < len; ++i) {
            char ch = str.charAt(i);
            if (!Chars3986.isPctEncoded(ch, str, i)) {
                sb.append(ch);
                continue;
            }
            char ch1 = this.toUpperASCII(str.charAt(i + 1));
            char ch2 = this.toUpperASCII(str.charAt(i + 2));
            i += 2;
            char x = (char)(Chars3986.hexValue(ch1) * 16 + Chars3986.hexValue(ch2));
            if (Chars3986.unreserved(x)) {
                sb.append(x);
                continue;
            }
            sb.append('%');
            sb.append(ch1);
            sb.append(ch2);
        }
        return sb.toString();
    }

    private char toUpperASCII(char ch) {
        if (ch >= 'a' && ch <= 'z') {
            ch = (char)(ch + -32);
        }
        return ch;
    }

    private String toLowerCase(String string) {
        if (string == null) {
            return null;
        }
        return string.toLowerCase(Locale.ROOT);
    }

    public IRI3986 relativize(IRI iri) {
        return AlgResolveIRI.relativize(this, iri);
    }

    public IRI3986 resolve(IRI3986 other) {
        IRI3986 iri = AlgResolveIRI.resolve(this, other);
        if (iri != other) {
            iri.schemeSpecificRulesInternal();
        }
        return iri;
    }

    public static IRI3986 build(String scheme, String authority, String path, String query, String fragment) {
        String s = IRI3986.rebuild(scheme, authority, path, query, fragment);
        return IRI3986.newAndParseEx(s);
    }

    public String rebuild() {
        return IRI3986.rebuild(this.scheme(), this.authority(), this.path(), this.query(), this.fragment());
    }

    private static String rebuild(String scheme, String authority, String path, String query, String fragment) {
        StringBuilder result = new StringBuilder();
        if (scheme != null) {
            result.append(scheme);
            result.append(":");
        }
        if (authority != null) {
            result.append("//");
            result.append(authority);
        }
        if (path != null) {
            result.append(path);
        }
        if (query != null) {
            result.append("?");
            result.append(query);
        }
        if (fragment != null) {
            result.append("#");
            result.append(fragment);
        }
        return result.toString();
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.iriStr);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IRI3986)) {
            return false;
        }
        IRI3986 other = (IRI3986)obj;
        return Objects.equals(this.iriStr, other.iriStr);
    }

    boolean identical(IRI3986 other, boolean includeComponentStrings) {
        if (this == other) {
            return true;
        }
        if (other == null) {
            return false;
        }
        return !(!Objects.equals(this.iriStr, other.iriStr) || this.length != other.length || this.scheme0 != other.scheme0 || this.scheme1 != other.scheme1 || includeComponentStrings && !Objects.equals(this.scheme(), other.scheme()) || this.authority0 != other.authority0 || this.authority1 != other.authority1 || includeComponentStrings && !Objects.equals(this.authority(), other.authority()) || this.userinfo0 != other.userinfo0 || this.userinfo1 != other.userinfo1 || includeComponentStrings && !Objects.equals(this.userInfo(), other.userInfo()) || this.host0 != other.host0 || this.host1 != other.host1 || includeComponentStrings && !Objects.equals(this.host(), other.host()) || this.port0 != other.port0 || this.port1 != other.port1 || includeComponentStrings && !Objects.equals(this.port(), other.port()) || this.path0 != other.path0 || this.path1 != other.path1 || includeComponentStrings && !Objects.equals(this.path(), other.path()) || this.query0 != other.query0 || this.query1 != other.query1 || includeComponentStrings && !Objects.equals(this.query(), other.query()) || this.fragment0 != other.fragment0 || this.fragment1 != other.fragment1 || includeComponentStrings && !Objects.equals(this.fragment(), other.fragment()));
    }

    static IRI3986 createByRegex(String iriStr) {
        Objects.requireNonNull(iriStr);
        Pattern pattern = RFC3986.rfc3986regex;
        Matcher m = pattern.matcher(iriStr);
        if (!m.matches()) {
            throw ParseErrorIRI3986.parseError(iriStr, "String does not match the regular expression for IRIs");
        }
        int length = iriStr.length();
        int schemeGroup = 2;
        int authorityGroup = 4;
        int pathGroup = 5;
        int queryGroup = 7;
        int fragmentGroup = 9;
        IRI3986 iri = new IRI3986(iriStr);
        iri.scheme0 = m.start(2);
        iri.scheme1 = m.end(2);
        iri.scheme = m.group(2);
        iri.authority0 = m.start(4);
        iri.authority1 = m.end(4);
        iri.authority = m.group(4);
        iri.userinfo0 = -1;
        iri.userinfo1 = -1;
        iri.host0 = -1;
        iri.host1 = -1;
        iri.host = null;
        iri.port0 = -1;
        iri.port1 = -1;
        iri.port = null;
        iri.path0 = m.start(5);
        iri.path1 = m.end(5);
        iri.path = m.group(5);
        if (iri.path.isEmpty()) {
            iri.path0 = -1;
            iri.path1 = -1;
        }
        iri.query0 = m.start(7);
        iri.query1 = m.end(7);
        iri.query = m.group(7);
        iri.fragment0 = m.start(9);
        iri.fragment1 = m.end(9);
        iri.fragment = m.group(9);
        m = null;
        if (iri.authority != null) {
            int userinfoGroup = 2;
            int hostGroup = 3;
            int portGroup = 6;
            int offset = iri.authority0;
            Matcher m2 = authorityRegex.matcher(iri.authority);
            if (m2.matches()) {
                iri.userinfo0 = IRI3986.offset(offset, m2.start(2));
                iri.userinfo1 = IRI3986.offset(offset, m2.end(2));
                iri.host = m2.group(3);
                iri.host0 = IRI3986.offset(offset, m2.start(3));
                iri.host1 = IRI3986.offset(offset, m2.end(3));
                iri.port = m2.group(6);
                iri.port0 = IRI3986.offset(offset, m2.start(6));
                iri.port1 = IRI3986.offset(offset, m2.end(6));
            }
        }
        return iri;
    }

    private static int offset(int offset, int index) {
        return index < 0 ? index : offset + index;
    }

    private IRI3986 parse() {
        int x = this.scheme(0);
        if (x > 0) {
            this.scheme0 = 0;
            this.scheme1 = x;
            x = this.withScheme(x + 1);
        } else {
            x = this.withoutScheme(0);
        }
        if (x != this.length) {
            String label = this.fragment0 >= 0 ? "fragment" : (this.query0 >= 0 ? "query" : "path");
            throw ParseErrorIRI3986.parseError(this.iriStr, "Bad character in " + label + " component: " + Chars3986.displayChar(this.charAt(x)));
        }
        return this;
    }

    private int scheme(int start) {
        int end = this.length;
        for (int p = start; p < end; ++p) {
            char c = this.charAt(p);
            if (c == ':') {
                return p;
            }
            if (Chars3986.isAlpha(c)) continue;
            if (p == start) {
                return -1;
            }
            if (Chars3986.isDigit(c) || c == '+' || c == '-' || c == '.') continue;
            return -1;
        }
        return 0;
    }

    private int withScheme(int start) {
        int p = this.maybeAuthority(start);
        return this.pathQueryFragment(p, true);
    }

    private int withoutScheme(int start) {
        char ch = this.charAt(start);
        if (ch == ':') {
            throw ParseErrorIRI3986.parseError(this.iriStr, "A URI without a scheme can't start with a ':'");
        }
        int p = this.maybeAuthority(start);
        return this.pathQueryFragment(p, false);
    }

    private int maybeAuthority(int start) {
        int p = start;
        char ch1 = this.charAt(p);
        char ch2 = this.charAt(p + 1);
        if (ch1 == '/' && ch2 == '/') {
            p += 2;
            p = this.authority(p);
        }
        return p;
    }

    private int authority(int start) {
        int p;
        int end = this.length;
        int endUserInfo = -1;
        int lastColon = -1;
        int countColon = 0;
        int startIPv6 = -1;
        int endIPv6 = -1;
        for (p = start; p < end; ++p) {
            char ch = this.charAt(p);
            if (ch == ':') {
                ++countColon;
                lastColon = p;
                continue;
            }
            if (ch == '/') {
                if (startIPv6 < 0 || endIPv6 != -1) break;
                throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, "Bad IPv6 address - No closing ']'");
            }
            if (ch == '@') {
                if (endUserInfo != -1) {
                    throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, "Bad authority segment - multiple '@'");
                }
                if (startIPv6 != -1 || endIPv6 != -1) {
                    throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, "Bad authority segment - contains '[' or ']'");
                }
                endUserInfo = p;
                countColon = 0;
                lastColon = -1;
                continue;
            }
            if (ch == '[') {
                if (startIPv6 >= 0) {
                    throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, "Bad IPv6 address - multiple '['");
                }
                startIPv6 = p;
                continue;
            }
            if (ch == ']') {
                if (startIPv6 == -1) {
                    throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, "Bad IPv6 address - No '[' to match ']'");
                }
                if (endIPv6 >= 0) {
                    throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, "Bad IPv6 address - multiple ']'");
                }
                endIPv6 = p;
                countColon = 0;
                lastColon = -1;
                continue;
            }
            if (!this.isIPChar(ch, p)) break;
        }
        if (startIPv6 != -1) {
            if (endIPv6 == -1) {
                throw ParseErrorIRI3986.parseError(this.iriStr, startIPv6, "Bad IPv6 address - missing ']'");
            }
            char ch1 = this.iriStr.charAt(startIPv6);
            char ch2 = this.iriStr.charAt(endIPv6);
            ParseIPv6Address.checkIPv6(this.iriStr, startIPv6, endIPv6 + 1);
        }
        this.authority0 = start;
        this.authority1 = p;
        int endAuthority = p;
        if (endUserInfo != -1) {
            this.userinfo0 = start;
            this.userinfo1 = endUserInfo;
            this.host0 = endUserInfo + 1;
            if (lastColon != -1 && lastColon < endUserInfo) {
                lastColon = -1;
            }
        } else {
            this.host0 = start;
        }
        if (countColon > 1) {
            throw ParseErrorIRI3986.parseError(this.iriStr, -1, "Multiple ':' in host:port section");
        }
        if (lastColon != -1) {
            char ch;
            int x;
            this.host1 = lastColon;
            this.port0 = lastColon + 1;
            this.port1 = endAuthority;
            for (x = this.port0; x < this.port1 && Chars3986.isDigit(ch = this.charAt(x)); ++x) {
            }
            if (x != this.port1) {
                throw ParseErrorIRI3986.parseError(this.iriStr, -1, "Bad port");
            }
        } else {
            this.host1 = endAuthority;
        }
        return endAuthority;
    }

    private int pathQueryFragment(int start, boolean withScheme) {
        int x2;
        int x1 = this.path(start, withScheme);
        if (x1 < 0) {
            x1 = start;
        }
        if ((x2 = this.query(x1)) < 0) {
            x2 = x1;
        }
        int x3 = this.fragment(x2);
        return x3;
    }

    private int path(int start, boolean withScheme) {
        if (start == this.length) {
            return start;
        }
        int segStart = start;
        int p = start;
        boolean allowColon = withScheme;
        while (p < this.length) {
            char ch = this.charAt(p);
            int charLen = this.isIPCharLen(ch, p);
            if (charLen == 1) {
                if (!allowColon && ch == ':') {
                    throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, "':' in initial segment of a scheme-less IRI");
                }
                ++p;
                continue;
            }
            if (charLen == 3) {
                p += 3;
                continue;
            }
            if (ch != '/') {
                if (ch == ' ') {
                    throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, "Space found in IRI");
                }
                if (ch == '?' || ch == '#') break;
                throw ParseErrorIRI3986.parseError(this.iriStr, p + 1, String.format("Bad character in IRI path: '%s' (U+%04X)", Character.toString((int)ch), (int)ch));
            }
            allowColon = true;
            segStart = p + 1;
            ++p;
        }
        if (p > start) {
            this.path0 = start;
            this.path1 = Math.min(p, this.length);
        }
        return p;
    }

    private int query(int start) {
        int x = this.trailer('?', start, true);
        if (x >= 0 && x != start) {
            this.query0 = start + 1;
            this.query1 = x;
        }
        if (x < 0) {
            x = start;
        }
        return x;
    }

    private int fragment(int start) {
        int x = this.trailer('#', start, false);
        if (x >= 0 && x != start) {
            this.fragment0 = start + 1;
            this.fragment1 = x;
        }
        if (x < 0) {
            x = start;
        }
        return x;
    }

    private int trailer(char startChar, int start, boolean allowPrivate) {
        if (start >= this.length) {
            return -1;
        }
        if (this.charAt(start) != startChar) {
            return -1;
        }
        int p = start + 1;
        while (p < this.length) {
            char ch = this.charAt(p);
            int charLen = this.isIPCharLen(ch, p);
            if (charLen == 1 || charLen == 3) {
                p += charLen;
                continue;
            }
            if (ch == '/' || ch == '?') {
                ++p;
                continue;
            }
            if (allowPrivate && Chars3986.isIPrivate(ch)) {
                ++p;
                continue;
            }
            return p;
        }
        return p;
    }

    private char charAt(int x) {
        if (x < 0) {
            throw new IllegalArgumentException("Negative index");
        }
        if (x >= this.length) {
            return '\uffff';
        }
        return this.iriStr.charAt(x);
    }

    private boolean isPctEncoded(char ch, int idx) {
        if (ch != '%') {
            return false;
        }
        char ch1 = this.charAt(idx + 1);
        char ch2 = this.charAt(idx + 2);
        return Chars3986.percentCheck(ch1, ch2, this.iriStr, idx);
    }

    private boolean isPChar(char ch, int posn) {
        return Chars3986.unreserved(ch) || this.isPctEncoded(ch, posn) || Chars3986.subDelims(ch) || ch == ':' || ch == '@';
    }

    private int isPCharLen(char ch, int posn) {
        if (Chars3986.unreserved(ch) || Chars3986.subDelims(ch) || ch == ':' || ch == '@') {
            return 1;
        }
        if (this.isPctEncoded(ch, posn)) {
            return 3;
        }
        return -1;
    }

    private boolean isIPChar(char ch, int posn) {
        return this.isPChar(ch, posn) || Chars3986.isUcsChar(ch);
    }

    private int isIPCharLen(char ch, int posn) {
        if (Chars3986.unreserved(ch) || Chars3986.subDelims(ch) || ch == ':' || ch == '@' || Chars3986.isUcsChar(ch)) {
            return 1;
        }
        if (this.isPctEncoded(ch, posn)) {
            return 3;
        }
        return -1;
    }

    private static boolean isASCII(String string, int start, int finish) {
        for (int i = 0; i < finish; ++i) {
            char ch = string.charAt(i);
            if (ch <= '\u007f') continue;
            return false;
        }
        return true;
    }

    private IRI3986 schemeSpecificRulesInternal() {
        if (this.reports != null) {
            return this;
        }
        this.checkGeneral();
        if (!this.hasScheme()) {
            return this;
        }
        if (URIScheme.fromScheme(this.iriStr, URIScheme.HTTPS)) {
            this.checkHTTPS();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.HTTP)) {
            this.checkHTTP();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.URN_UUID)) {
            this.checkURN_UUID();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.URN_OID)) {
            this.checkURN_OID();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.URN)) {
            this.checkURN();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.FILE)) {
            this.checkFILE();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.UUID)) {
            this.checkUUID();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.DID)) {
            this.checkDID();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.OID)) {
            this.checkOID();
        } else if (URIScheme.fromScheme(this.iriStr, URIScheme.EXAMPLE)) {
            this.checkExample();
        }
        if (this.reports != null) {
            this.reports = List.copyOf(this.reports);
        }
        return this;
    }

    private void checkGeneral() {
        boolean good;
        if (this.hasUserInfo()) {
            this.schemeReport(this, Issue.iri_user_info_present, URIScheme.GENERAL, "Use of user info is deprecated");
            int idx = IRI3986.contains(this.iriStr, ':', this.userinfo0, this.userinfo1);
            if (idx >= 0 && idx < this.userinfo1 - 1) {
                this.schemeReport(this, Issue.iri_password, URIScheme.GENERAL, "Non-empty password");
            }
        }
        if (this.hasHost() && this.containsUppercase(this.iriStr, this.host0, this.host1)) {
            this.schemeReport(this, Issue.iri_host_not_lowercase, URIScheme.GENERAL, "Host name should be lowercase");
        }
        this.checkPercent();
        if (this.hasPath() && !(good = LibParseIRI.checkDotSegments(this.iriStr, this.path0, this.path1))) {
            this.schemeReport(this, Issue.iri_bad_dot_segments, URIScheme.GENERAL, "Dot segments should only appear at the start of a relative IRI");
        }
    }

    private void checkPercent() {
        if (this.path0 < 0) {
            return;
        }
        int N = this.iriStr.length();
        for (int i = this.path0; i < N; ++i) {
            char ch = this.iriStr.charAt(i);
            if (ch != '%') continue;
            if (i + 2 > N) {
                // empty if block
            }
            char ch1 = this.iriStr.charAt(i + 1);
            char ch2 = this.iriStr.charAt(i + 2);
            if (Chars3986.isHexDigitLC(ch1) || Chars3986.isHexDigitLC(ch2)) {
                this.schemeReport(this, Issue.iri_percent_not_uppercase, URIScheme.GENERAL, "Percent encoding should be uppercase");
            }
            i += 2;
        }
    }

    private void checkSchemeName(URIScheme scheme) {
        String correctSchemeName = scheme.getSchemeName();
        if (!this.hasScheme()) {
            this.schemeReport(this, Issue.iri_scheme_expected, scheme, "No scheme name");
            return;
        }
        if (!URIScheme.matchesExact(this.iriStr, scheme)) {
            if (URIScheme.matchesIgnoreCase(this.iriStr, scheme)) {
                this.schemeReport(this, Issue.iri_scheme_name_is_not_lowercase, scheme, "Scheme name should be lowercase");
            } else {
                this.schemeReport(this, Issue.iri_scheme_unexpected, scheme, "Scheme name should be '" + correctSchemeName + "'");
            }
        }
    }

    private boolean containsUppercase(String string, int start, int finish) {
        for (int i = start; i < finish; ++i) {
            char ch = string.charAt(i);
            if (!Character.isUpperCase(ch)) continue;
            return true;
        }
        return false;
    }

    private void checkHTTP() {
        this.checkSchemeName(URIScheme.HTTP);
        this.checkHTTPx(URIScheme.HTTP);
    }

    private void checkHTTPS() {
        this.checkSchemeName(URIScheme.HTTPS);
        this.checkHTTPx(URIScheme.HTTPS);
    }

    private void checkHTTPx(URIScheme scheme) {
        if (!this.hasHost()) {
            this.schemeReport(this, Issue.http_no_host, scheme, "http and https URI schemes require //host/");
        } else if (this.host0 == this.host1) {
            this.schemeReport(this, Issue.http_empty_host, scheme, "http and https URI schemes do not allow the host to be empty");
        }
        if (this.hasPort()) {
            if (this.port0 == this.port1) {
                this.schemeReport(this, Issue.http_empty_port, scheme, "Port is empty - omit the ':'");
            } else {
                int port = Integer.parseInt(this.port());
                switch (scheme) {
                    case HTTP: {
                        if (port == 80) {
                            this.schemeReport(this, Issue.http_omit_well_known_port, scheme, "Default port 80 should be omitted");
                            break;
                        }
                        if (port >= 1024 || port == 80) break;
                        this.schemeReport(this, Issue.http_port_not_advised, scheme, "An HTTP port under 1024 should only be 80, not " + port);
                        break;
                    }
                    case HTTPS: {
                        if (port == 443) {
                            this.schemeReport(this, Issue.http_omit_well_known_port, scheme, "Default port 443 should be omitted");
                            break;
                        }
                        if (port >= 1024 || port == 443) break;
                        this.schemeReport(this, Issue.http_port_not_advised, scheme, "An HTTPS ports under 1024 should only be 443, not " + port);
                        break;
                    }
                    default: {
                        throw new IllegalStateException();
                    }
                }
            }
        }
    }

    private void checkFILE() {
        this.checkSchemeName(URIScheme.FILE);
        if (!this.hasAuthority()) {
            if (this.path().startsWith("/")) {
                this.schemeReport(this, Issue.file_bad_form, URIScheme.FILE, "file: URLs are of the form file:///path/...");
            } else {
                this.schemeReport(this, Issue.file_relative_path, URIScheme.FILE, "file: URLs are of the form file:///path/..., not file:filename");
            }
        } else if (this.authority0 != this.authority1) {
            this.schemeReport(this, Issue.file_bad_form, URIScheme.FILE, "file: URLs are of the form file:///path/..., not file://path");
        } else if (this.path0 == this.path1) {
            this.schemeReport(this, Issue.file_bad_form, URIScheme.FILE, "file: URLs are of the form file:///path/..., not file://path");
        }
    }

    private void checkURN() {
        this.checkSchemeName(URIScheme.URN);
        BiConsumer<Issue, String> handler = (issue, msg) -> this.schemeReport(this, (Issue)((Object)issue), URIScheme.URN, (String)msg);
        int finishURN = ParseURN.validateAssignedName(this.iriStr, handler);
        if (finishURN == -1) {
            return;
        }
        this.checkURNComponents(URIScheme.URN, handler);
        if (this.hasQuery()) {
            this.urnCharCheck("URN components", this.iriStr, this.query0, this.iriStr.length());
        } else if (this.hasFragment()) {
            this.urnCharCheck("URN components", this.iriStr, this.fragment0, this.iriStr.length());
        }
    }

    private void urnCharCheck(String urnPart, String string, int start, int finish) {
    }

    private void checkURNComponents(URIScheme scheme, BiConsumer<Issue, String> handler) {
        if (!this.hasQuery() && !this.hasFragment()) {
            return;
        }
        if (!this.hasQuery()) {
            return;
        }
        int idx = this.query0 - 1;
        ParseURNComponents.validateURNComponents(this.iriStr, idx, handler);
    }

    private void checkURN_UUID() {
        this.checkSchemeName(URIScheme.URN_UUID);
        boolean matches = URN_UUID_PATTERN_LC.matcher(this.iriStr).matches();
        if (matches) {
            return;
        }
        this.checkUUID(URIScheme.URN_UUID, this.iriStr, this.URN_UUID_scheme_path_length);
        BiConsumer<Issue, String> handler = (issue, msg) -> this.schemeReport(this, (Issue)((Object)issue), URIScheme.URN, (String)msg);
        this.checkURNComponents(URIScheme.URN_UUID, handler);
    }

    private void checkUUID() {
        this.checkSchemeName(URIScheme.UUID);
        this.schemeReport(this, Issue.uuid_scheme_not_registered, URIScheme.UUID, "Use urn:uuid: -  'uuid:' is not a registered URI scheme.");
        boolean matches = UUID_PATTERN_LC.matcher(this.iriStr).matches();
        if (matches) {
            return;
        }
        this.checkUUID(URIScheme.UUID, this.iriStr, 36);
        if (this.hasQuery()) {
            this.schemeReport(this, Issue.uuid_has_query, URIScheme.UUID, "query component not allowed");
        }
        if (this.hasFragment()) {
            this.schemeReport(this, Issue.uuid_has_fragment, URIScheme.UUID, "fragment not allowed");
        }
    }

    private void checkUUID(URIScheme scheme, String iriStr, int uriPathLen) {
        boolean matchesAnyCase;
        int actualPathLen = this.path1 - this.path0;
        if (actualPathLen != uriPathLen) {
            this.schemeReport(this, Issue.uuid_bad_pattern, scheme, "Bad UUID string (wrong length)");
            return;
        }
        if (scheme == URIScheme.URN_UUID && this.containsHexUC(iriStr, this.path0, this.path0 + "uuid".length())) {
            this.schemeReport(this, Issue.uuid_not_lowercase, scheme, "Lowercase recommended for urn UUID namspace");
        }
        if (!(matchesAnyCase = UUID_PATTERN_AnyCase_PREFIX.matcher(iriStr).find())) {
            this.schemeReport(this, Issue.uuid_bad_pattern, scheme, "Not a valid UUID string");
            return;
        }
        int uuidStart = this.path1 - 36;
        int uuidFinish = this.path1;
        if (this.containsHexUC(iriStr, uuidStart, uuidFinish)) {
            this.schemeReport(this, Issue.uuid_not_lowercase, scheme, "Lowercase recommended for UUID string");
        }
    }

    private boolean containsHexUC(String iriStr2, int uuidStart, int uuidFinish) {
        for (int i = uuidStart; i < uuidFinish; ++i) {
            char ch = this.charAt(i);
            if (!Chars3986.range(ch, 65, 70)) continue;
            return true;
        }
        return false;
    }

    private void checkDID() {
        this.checkSchemeName(URIScheme.DID);
        try {
            ParseDID.parse(this.iriStr, true);
        }
        catch (RuntimeException ex) {
            this.schemeReport(this, Issue.did_bad_syntax, URIScheme.DID, "Invalid DID: " + ex.getMessage());
        }
    }

    private void checkURN_OID() {
        this.checkSchemeName(URIScheme.URN_OID);
        this.checkOID(URIScheme.URN_OID, this.iriStr);
    }

    private void checkOID() {
        this.checkSchemeName(URIScheme.OID);
        this.schemeReport(this, Issue.oid_scheme_not_registered, URIScheme.OID, "Use 'urn:oid:' - 'oid:' is not a registered URI scheme.");
        this.checkOID(URIScheme.OID, this.iriStr);
    }

    private void checkOID(URIScheme scheme, String iriStr) {
        try {
            ParseOID.parse(iriStr);
        }
        catch (RuntimeException ex) {
            this.schemeReport(this, Issue.oid_bad_syntax, scheme, "Invalid OID: " + ex.getMessage());
        }
    }

    private void checkExample() {
        this.checkSchemeName(URIScheme.EXAMPLE);
    }

    private void schemeReport(IRI3986 iri, Issue issue, URIScheme scheme, String msg) {
        Objects.requireNonNull(issue);
        if (issue == Issue.ParseError) {
            throw ParseErrorIRI3986.parseError(iri.str(), msg);
        }
        IRI3986.addReport(iri, iri.str(), scheme, issue, msg);
    }

    private static void addReport(IRI3986 iri, String iriStr, URIScheme uriScheme, Issue issue, String message) {
        String msg = "<" + iriStr + "> " + message;
        Violation v = new Violation(iriStr, uriScheme, issue, msg);
        IRI3986.addReport(iri, v);
    }

    private static void addReportParseError(IRI3986 iri, String iriStr, String message) {
        Object msg = message;
        if (!message.startsWith("<" + iriStr + ">")) {
            msg = "'" + iriStr + "' : " + message;
        }
        Violation v = new Violation(iriStr, null, Issue.ParseError, (String)msg);
        IRI3986.addReport(iri, v);
    }

    private static void addReport(IRI3986 iri, Violation report) {
        if (iri.reports == null) {
            iri.reports = new ArrayList<Violation>(4);
        }
        iri.reports.add(report);
    }
}

