/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hertzbeat.collector.collect.redis;

import io.lettuce.core.RedisClient;
import io.lettuce.core.RedisConnectionException;
import io.lettuce.core.RedisURI;
import io.lettuce.core.api.StatefulConnection;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.cluster.RedisClusterClient;
import io.lettuce.core.cluster.api.StatefulRedisClusterConnection;
import io.lettuce.core.cluster.models.partitions.Partitions;
import io.lettuce.core.cluster.models.partitions.RedisClusterNode;
import io.lettuce.core.resource.ClientResources;
import io.lettuce.core.resource.DefaultClientResources;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.apache.hertzbeat.collector.collect.AbstractCollect;
import org.apache.hertzbeat.collector.collect.common.cache.AbstractConnection;
import org.apache.hertzbeat.collector.collect.common.cache.CacheIdentifier;
import org.apache.hertzbeat.collector.collect.common.cache.GlobalConnectionCache;
import org.apache.hertzbeat.collector.collect.common.cache.RedisConnect;
import org.apache.hertzbeat.collector.collect.common.ssh.SshTunnelHelper;
import org.apache.hertzbeat.collector.util.CollectUtil;
import org.apache.hertzbeat.common.entity.job.Metrics;
import org.apache.hertzbeat.common.entity.job.SshTunnel;
import org.apache.hertzbeat.common.entity.job.protocol.RedisProtocol;
import org.apache.hertzbeat.common.entity.message.CollectRep;
import org.apache.hertzbeat.common.util.CommonUtil;
import org.apache.hertzbeat.common.util.MapCapUtil;
import org.apache.sshd.common.SshException;
import org.apache.sshd.common.channel.exception.SshChannelOpenException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

public class RedisCommonCollectImpl
extends AbstractCollect {
    private static final Logger log = LoggerFactory.getLogger(RedisCommonCollectImpl.class);
    private static final String CLUSTER = "3";
    private static final String CLUSTER_INFO = "cluster";
    private static final String UNIQUE_IDENTITY = "identity";
    private final ClientResources defaultClientResources;
    private final GlobalConnectionCache connectionCache = GlobalConnectionCache.getInstance();

    public RedisCommonCollectImpl() {
        this.defaultClientResources = DefaultClientResources.create();
    }

    public void preCheck(Metrics metrics) throws IllegalArgumentException {
        Assert.noNullElements((Object[])new Object[]{metrics, metrics.getRedis()}, (String)"Redis collect must has redis params");
        RedisProtocol redisProtocol = metrics.getRedis();
        Assert.hasText((String)redisProtocol.getHost(), (String)"Redis Protocol host is required.");
        Assert.hasText((String)redisProtocol.getPort(), (String)"Redis Protocol port is required.");
        SshTunnelHelper.checkTunnelParam((SshTunnel)metrics.getRedis().getSshTunnel());
    }

    public void collect(CollectRep.MetricsData.Builder builder, Metrics metrics) {
        try {
            if (Objects.nonNull(metrics.getRedis().getPattern()) && Objects.equals(metrics.getRedis().getPattern(), CLUSTER)) {
                List<Map<String, String>> redisInfoList = this.getClusterRedisInfo(metrics);
                this.doMetricsDataList(builder, redisInfoList, metrics);
            } else {
                Map<String, String> redisInfo = this.getSingleRedisInfo(metrics);
                this.doMetricsData(builder, redisInfo, metrics);
            }
        }
        catch (RedisConnectionException connectionException) {
            String errorMsg = CommonUtil.getMessageFromThrowable((Throwable)connectionException);
            log.info("[redis connection] error: {}", (Object)errorMsg);
            builder.setCode(CollectRep.Code.UN_CONNECTABLE);
            builder.setMsg(errorMsg);
        }
        catch (SshException sshException) {
            Throwable throwable = sshException.getCause();
            if (throwable instanceof SshChannelOpenException) {
                log.warn("[redis collect] Remote ssh server no more session channel, please increase sshd_config MaxSessions.");
            }
            String errorMsg = CommonUtil.getMessageFromThrowable((Throwable)sshException);
            builder.setCode(CollectRep.Code.UN_CONNECTABLE);
            builder.setMsg("Peer ssh connection failed: " + errorMsg);
        }
        catch (Exception e) {
            String errorMsg = CommonUtil.getMessageFromThrowable((Throwable)e);
            log.warn("[redis collect] error: {}", (Object)e.getMessage(), (Object)e);
            builder.setCode(CollectRep.Code.FAIL);
            builder.setMsg(errorMsg);
        }
    }

    private Map<String, String> getSingleRedisInfo(Metrics metrics) throws GeneralSecurityException, IOException {
        StatefulRedisConnection<String, String> connection = this.getSingleConnection(metrics.getRedis());
        String info = connection.sync().info(metrics.getName());
        Map<String, String> valueMap = this.parseInfo(info, metrics);
        if (log.isDebugEnabled()) {
            log.debug("[RedisSingleCollectImpl] fetch redis info");
            valueMap.forEach((k, v) -> log.debug("{} : {}", k, v));
        }
        return valueMap;
    }

    private List<Map<String, String>> getClusterRedisInfo(Metrics metrics) throws GeneralSecurityException, IOException {
        Map<String, StatefulRedisClusterConnection<String, String>> connectionMap = this.getConnectionList(metrics.getRedis());
        ArrayList<Map<String, String>> list = new ArrayList<Map<String, String>>(connectionMap.size());
        connectionMap.forEach((identity, connection) -> {
            String info = connection.sync().info(metrics.getName());
            Map<String, String> valueMap = this.parseInfo(info, metrics);
            valueMap.put(UNIQUE_IDENTITY, (String)identity);
            if (Objects.equals(metrics.getName(), CLUSTER_INFO)) {
                String clusterNodes = connection.sync().clusterInfo();
                valueMap.putAll(this.parseInfo(clusterNodes, metrics));
            }
            if (log.isDebugEnabled()) {
                log.debug("[RedisSingleCollectImpl] fetch redis info");
                valueMap.forEach((k, v) -> log.debug("{} : {}", k, v));
            }
            list.add(valueMap);
        });
        return list;
    }

    private void doMetricsDataList(CollectRep.MetricsData.Builder builder, List<Map<String, String>> valueMapList, Metrics metrics) {
        valueMapList.forEach(e -> this.doMetricsData(builder, (Map<String, String>)e, metrics));
    }

    private void doMetricsData(CollectRep.MetricsData.Builder builder, Map<String, String> valueMap, Metrics metrics) {
        CollectRep.ValueRow.Builder valueRowBuilder = CollectRep.ValueRow.newBuilder();
        metrics.getAliasFields().forEach(it -> {
            if (valueMap.containsKey(it)) {
                String fieldValue = (String)valueMap.get(it);
                valueRowBuilder.addColumn(Objects.requireNonNullElse(fieldValue, "&nbsp;"));
            } else {
                valueRowBuilder.addColumn("&nbsp;");
            }
        });
        builder.addValueRow(valueRowBuilder.build());
    }

    private StatefulRedisConnection<String, String> getSingleConnection(RedisProtocol redisProtocol) throws GeneralSecurityException, IOException {
        String port;
        String[] resolvedArr = this.resolveHostAndPort(redisProtocol);
        String host = resolvedArr[0];
        CacheIdentifier identifier = this.doIdentifier(redisProtocol, host, port = resolvedArr[1]);
        StatefulRedisConnection connection = (StatefulRedisConnection)this.getStatefulConnection(identifier);
        if (Objects.isNull(connection)) {
            RedisClient redisClient = this.buildSingleClient(redisProtocol, host, port);
            connection = redisClient.connect();
            this.connectionCache.addCache((Object)identifier, (AbstractConnection)new RedisConnect((StatefulConnection)connection));
        }
        return connection;
    }

    private Map<String, StatefulRedisClusterConnection<String, String>> getConnectionList(RedisProtocol redisProtocol) throws GeneralSecurityException, IOException {
        StatefulRedisClusterConnection<String, String> connection = this.getClusterConnection(redisProtocol);
        Partitions partitions = connection.getPartitions();
        HashMap<String, StatefulRedisClusterConnection<String, String>> clusterConnectionMap = new HashMap<String, StatefulRedisClusterConnection<String, String>>(partitions.size());
        for (RedisClusterNode partition : partitions) {
            RedisURI uri = partition.getUri();
            redisProtocol.setHost(uri.getHost());
            redisProtocol.setPort(String.valueOf(uri.getPort()));
            StatefulRedisClusterConnection<String, String> clusterConnection = this.getClusterConnection(redisProtocol);
            clusterConnectionMap.put(this.doUri(uri.getHost(), uri.getPort()), clusterConnection);
        }
        return clusterConnectionMap;
    }

    private StatefulRedisClusterConnection<String, String> getClusterConnection(RedisProtocol redisProtocol) throws GeneralSecurityException, IOException {
        String port;
        String[] resolvedArr = this.resolveHostAndPort(redisProtocol);
        String host = resolvedArr[0];
        CacheIdentifier identifier = this.doIdentifier(redisProtocol, host, port = resolvedArr[1]);
        StatefulRedisClusterConnection connection = (StatefulRedisClusterConnection)this.getStatefulConnection(identifier);
        if (connection == null) {
            RedisClusterClient redisClusterClient = this.buildClusterClient(redisProtocol, host, port);
            connection = redisClusterClient.connect();
            this.connectionCache.addCache((Object)identifier, (AbstractConnection)new RedisConnect((StatefulConnection)connection));
        }
        return connection;
    }

    private StatefulConnection<String, String> getStatefulConnection(CacheIdentifier identifier) {
        RedisConnect redisConnect;
        StatefulConnection connection = null;
        Optional cacheOption = this.connectionCache.getCache((Object)identifier, true);
        if (cacheOption.isPresent() && !(connection = (redisConnect = (RedisConnect)cacheOption.get()).getConnection()).isOpen()) {
            try {
                connection.closeAsync();
            }
            catch (Exception e) {
                log.info("The redis connect form cache, close error: {}", (Object)e.getMessage());
            }
            connection = null;
            this.connectionCache.removeCache((Object)identifier);
        }
        return connection;
    }

    private RedisClusterClient buildClusterClient(RedisProtocol redisProtocol, String host, String port) {
        return RedisClusterClient.create((ClientResources)this.defaultClientResources, (RedisURI)this.redisUri(redisProtocol, host, port));
    }

    private RedisClient buildSingleClient(RedisProtocol redisProtocol, String host, String port) {
        return RedisClient.create((ClientResources)this.defaultClientResources, (RedisURI)this.redisUri(redisProtocol, host, port));
    }

    private RedisURI redisUri(RedisProtocol redisProtocol, String host, String port) {
        RedisURI.Builder redisUriBuilder = RedisURI.builder().withHost(host).withPort(Integer.parseInt(port));
        if (StringUtils.hasText((String)redisProtocol.getUsername())) {
            redisUriBuilder.withClientName(redisProtocol.getUsername());
        }
        if (StringUtils.hasText((String)redisProtocol.getPassword())) {
            redisUriBuilder.withPassword(redisProtocol.getPassword().toCharArray());
        }
        Duration timeout = Duration.ofMillis(CollectUtil.getTimeout((String)redisProtocol.getTimeout()));
        redisUriBuilder.withTimeout(timeout);
        return redisUriBuilder.build();
    }

    private String removeCr(String value) {
        return value.replace("\r", "");
    }

    private String doUri(String ip, Integer port) {
        return ip + ":" + port;
    }

    private CacheIdentifier doIdentifier(RedisProtocol redisProtocol, String host, String port) {
        return CacheIdentifier.builder().ip(host).port(port).username(redisProtocol.getUsername()).password(redisProtocol.getPassword()).customArg(redisProtocol.getPattern()).build();
    }

    private Map<String, String> parseInfo(String info, Metrics metrics) {
        int fieldTotalSize = metrics.getFields().size();
        String[] lines = info.split("\n");
        HashMap<String, String> result = new HashMap<String, String>(MapCapUtil.calInitMap((int)fieldTotalSize));
        Arrays.stream(lines).filter(it -> StringUtils.hasText((String)it) && !it.startsWith("#") && it.contains(":")).map(this::removeCr).map(r -> r.split(":")).filter(t -> ((String[])t).length > 1).forEach(it -> result.put(it[0], it[1]));
        if (result.size() < fieldTotalSize) {
            for (Metrics.Field field : metrics.getFields()) {
                if (result.containsKey(field.getField())) continue;
                result.put(field.getField(), "&nbsp;");
            }
        }
        return result;
    }

    private String[] resolveHostAndPort(RedisProtocol redisProtocol) throws GeneralSecurityException, IOException {
        String port;
        String host;
        boolean enableSshTunnel = Optional.ofNullable(redisProtocol.getSshTunnel()).map(ssh -> Boolean.parseBoolean(ssh.getEnable())).orElse(false);
        if (enableSshTunnel) {
            host = "localhost";
            port = String.valueOf(SshTunnelHelper.localPortForward((SshTunnel)redisProtocol.getSshTunnel(), (String)redisProtocol.getHost(), (String)redisProtocol.getPort()));
        } else {
            host = redisProtocol.getHost();
            port = redisProtocol.getPort();
        }
        return new String[]{host, port};
    }

    public String supportProtocol() {
        return "redis";
    }
}

