/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.baremaps.database.type.geometry;

import java.nio.ByteBuffer;
import org.apache.baremaps.database.type.DataType;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Point;

/**
 * A data type for {@link Point} objects.
 */
public class PointDataType implements DataType<Point> {

  private final GeometryFactory geometryFactory;

  /**
   * Constructs a {@code PointDataType} with a default {@code GeometryFactory}.
   */
  public PointDataType() {
    this(new GeometryFactory());
  }

  /**
   * Constructs a {@code PointDataType} with a specified {@code GeometryFactory}.
   *
   * @param geometryFactory the geometry factory
   */
  public PointDataType(GeometryFactory geometryFactory) {
    this.geometryFactory = geometryFactory;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int size(final Point value) {
    return Double.BYTES * 2;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int size(final ByteBuffer buffer, final int position) {
    return Double.BYTES * 2;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(final ByteBuffer buffer, final int position, final Point value) {
    if (value.isEmpty()) {
      buffer.putDouble(position, Double.NaN);
      buffer.putDouble(position + Double.BYTES, Double.NaN);
    } else {
      buffer.putDouble(position, value.getX());
      buffer.putDouble(position + Double.BYTES, value.getY());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Point read(final ByteBuffer buffer, final int position) {
    double x = buffer.getDouble(position);
    double y = buffer.getDouble(position + Double.BYTES);
    if (Double.isNaN(x) || Double.isNaN(y)) {
      return geometryFactory.createPoint();
    } else {
      return geometryFactory.createPoint(new Coordinate(x, y));
    }
  }
}
