package org.apache.torque.generator.configuration;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.torque.generator.configuration.paths.ProjectPaths;
import org.apache.torque.generator.configuration.paths.TorqueGeneratorPaths;

/**
 * Provides InputStreams to read the configuration from a directory.
 */
public class DirectoryConfigurationProvider
        extends AbstractConfigurationProvider
{
    /** The logger. */
    private static Log log
            = LogFactory.getLog(DirectoryConfigurationProvider.class);

    /**
     * The paths needed to interact with the enclosing project, not null.
     */
    private ProjectPaths projectPaths;

    /**
     * The internal directory structure of the Torque generator configuration
     * files, not null.
     */
    private TorqueGeneratorPaths configurationPaths;

    /**
     * Constructor.
     *
     * @param projectPaths the paths needed to interact with the enclosing
     *        project, not null.
     * @param configurationPaths The internal directory structure of the
     *        generator files, not null.
     *
     * @throws NullPointerException if projectPaths or configurationPaths
     *         are null.
     */
    public DirectoryConfigurationProvider(
            ProjectPaths projectPaths,
            TorqueGeneratorPaths configurationPaths)
    {
        super(configurationPaths);
        if (projectPaths == null)
        {
            throw new NullPointerException("projectPaths is null");
        }
        this.projectPaths = projectPaths;
        this.configurationPaths = configurationPaths;
    }

    public String getControlConfigurationLocation()
            throws ConfigurationException
    {
        return getFile(
                configurationPaths.getControlConfigurationFile(),
                configurationPaths.getConfigurationDirectory(),
                "control file").getAbsolutePath();
    }

    private File getFile(
                String name,
                String directory,
                String description)
            throws ConfigurationException
    {
        File file = null;
        try
        {
            File configDir =  new File(
                    projectPaths.getConfigurationPath(),
                    directory);

            file = new File(configDir, name);
            // use canonical file to resolve . and .. directories
            file = file.getCanonicalFile();
        }
        catch (IOException e)
        {
            throw new ConfigurationException("Canonical name for "
                    + description + file
                    + " could not be calculated",
                e);
        }
        return file;
    }

    protected InputStream getInputStream(
                String name,
                String directory,
                String description)
            throws ConfigurationException
    {
        File file = getFile(name, directory, description);

        InputStream inputStream;
        try
        {
            inputStream = new FileInputStream(file);
        }
        catch (FileNotFoundException e)
        {
            throw new ConfigurationException(description + " file "
                    + file.getAbsolutePath()
                    + " not found",
                e);
        }
        BufferedInputStream bis = new BufferedInputStream(inputStream);
        if (log.isDebugEnabled())
        {
            log.debug("Reading " + description + " file: '"
                    + file.getAbsolutePath() + "'");
        }
        return bis;
    }

    public Collection<String> getOutletConfigurationNames()
        throws ConfigurationException
    {
        File outletConfigDir =  new File(
                projectPaths.getConfigurationPath(),
                configurationPaths.getOutletDirectory());

        List<String> result = new ArrayList<String>();
        if (!outletConfigDir.isDirectory())
        {
            throw new ConfigurationException(
                    "OutletsConfigDirectory "
                        + outletConfigDir.getAbsolutePath()
                        + "must be a directory");
        }

        File[] sourceFiles = outletConfigDir.listFiles();
        for (int fileNr = 0; fileNr < sourceFiles.length; ++fileNr)
        {
            if (!sourceFiles[fileNr].isDirectory()
                    && sourceFiles[fileNr].getPath().endsWith("xml"))
            {
                String name = sourceFiles[fileNr].getName();
                result.add(name);
            }
        }
        return result;
    }
}
