/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cayenne.access;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import org.apache.cayenne.CayenneRuntimeException;
import org.apache.cayenne.DataObject;
import org.apache.cayenne.DataRow;
import org.apache.cayenne.ObjectId;
import org.apache.cayenne.Persistent;
import org.apache.cayenne.access.event.SnapshotEvent;
import org.apache.cayenne.configuration.RuntimeProperties;
import org.apache.cayenne.event.EventBridge;
import org.apache.cayenne.event.EventManager;
import org.apache.cayenne.event.EventSubject;
import org.apache.cayenne.graph.GraphManager;
import org.apache.cayenne.util.concurrentlinkedhashmap.ConcurrentLinkedHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataRowStore
implements Serializable {
    private static final Logger logger = LoggerFactory.getLogger(DataRowStore.class);
    public static final long SNAPSHOT_EXPIRATION_DEFAULT = 7200L;
    public static final int SNAPSHOT_CACHE_SIZE_DEFAULT = 10000;
    protected String name;
    private int maxSize;
    protected ConcurrentMap<ObjectId, DataRow> snapshots;
    protected transient EventManager eventManager;
    protected transient EventBridge remoteNotificationsHandler;
    protected transient EventSubject eventSubject;

    public DataRowStore(String name, RuntimeProperties properties, EventManager eventManager) {
        if (name == null) {
            throw new IllegalArgumentException("DataRowStore name can't be null.");
        }
        this.name = name;
        this.eventSubject = this.createSubject();
        this.eventManager = eventManager;
        this.initWithProperties(properties);
    }

    private EventSubject createSubject() {
        return EventSubject.getSubject(this.getClass(), this.name);
    }

    protected void initWithProperties(RuntimeProperties properties) {
        this.maxSize = properties.getInt("cayenne.DataRowStore.snapshot.size", 10000);
        if (logger.isDebugEnabled()) {
            logger.debug("DataRowStore property cayenne.DataRowStore.snapshot.size = " + this.maxSize);
        }
        this.snapshots = new ConcurrentLinkedHashMap.Builder().maximumWeightedCapacity(this.maxSize).build();
    }

    protected void setEventBridge(EventBridge eventBridge) {
        this.remoteNotificationsHandler = eventBridge;
    }

    protected EventBridge getEventBridge() {
        return this.remoteNotificationsHandler;
    }

    void snapshotsUpdatedForObjects(List<Persistent> objects, List<? extends DataRow> snapshots, boolean refresh) {
        int size = objects.size();
        if (size != snapshots.size()) {
            throw new IllegalArgumentException("Counts of objects and corresponding snapshots do not match. Objects count: " + objects.size() + ", snapshots count: " + snapshots.size());
        }
        HashMap<ObjectId, DataRow> modified = null;
        GraphManager eventPostedBy = null;
        for (int i = 0; i < size; ++i) {
            Persistent object = objects.get(i);
            if (object == null || object.getPersistenceState() == 5) continue;
            ObjectId oid = object.getObjectId();
            DataRow cachedSnapshot = (DataRow)this.snapshots.get(oid);
            if (!refresh && cachedSnapshot != null) continue;
            DataRow newSnapshot = snapshots.get(i);
            if (cachedSnapshot != null) {
                if (object instanceof DataObject && cachedSnapshot.equals(newSnapshot)) {
                    ((DataObject)object).setSnapshotVersion(cachedSnapshot.getVersion());
                    continue;
                }
                newSnapshot.setReplacesVersion(cachedSnapshot.getVersion());
            }
            if (modified == null) {
                modified = new HashMap<ObjectId, DataRow>();
                eventPostedBy = object.getObjectContext().getGraphManager();
            }
            modified.put(oid, newSnapshot);
        }
        if (modified != null) {
            this.processSnapshotChanges(eventPostedBy, modified, Collections.emptyList(), Collections.emptyList(), Collections.emptyList());
        }
    }

    public int size() {
        return this.snapshots.size();
    }

    public int maximumSize() {
        return this.maxSize;
    }

    public void shutdown() {
        this.stopListeners();
        this.clear();
    }

    public String getName() {
        return this.name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public EventManager getEventManager() {
        return this.eventManager;
    }

    public void setEventManager(EventManager eventManager) {
        if (eventManager != this.eventManager) {
            this.stopListeners();
            this.eventManager = eventManager;
            this.startListeners();
        }
    }

    public DataRow getCachedSnapshot(ObjectId oid) {
        return (DataRow)this.snapshots.get(oid);
    }

    public EventSubject getSnapshotEventSubject() {
        return this.eventSubject;
    }

    public void clear() {
        this.snapshots.clear();
    }

    public void forgetSnapshot(ObjectId id) {
        this.snapshots.remove(id);
    }

    public void processRemoteEvent(SnapshotEvent event) {
        if (event.getSource() != this.remoteNotificationsHandler) {
            return;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("remote event: " + event);
        }
        Collection<ObjectId> deletedSnapshotIds = event.getDeletedIds();
        Collection<ObjectId> invalidatedSnapshotIds = event.getInvalidatedIds();
        Map<ObjectId, DataRow> diffs = event.getModifiedDiffs();
        Collection<ObjectId> indirectlyModifiedIds = event.getIndirectlyModifiedIds();
        if (deletedSnapshotIds.isEmpty() && invalidatedSnapshotIds.isEmpty() && diffs.isEmpty() && indirectlyModifiedIds.isEmpty()) {
            logger.warn("processRemoteEvent.. bogus call... no changes.");
            return;
        }
        this.processDeletedIDs(deletedSnapshotIds);
        this.processInvalidatedIDs(invalidatedSnapshotIds);
        this.processUpdateDiffs(diffs);
        this.sendUpdateNotification(event.getPostedBy(), diffs, deletedSnapshotIds, invalidatedSnapshotIds, indirectlyModifiedIds);
    }

    public void processSnapshotChanges(Object postedBy, Map<ObjectId, DataRow> updatedSnapshots, Collection<ObjectId> deletedSnapshotIds, Collection<ObjectId> invalidatedSnapshotIds, Collection<ObjectId> indirectlyModifiedIds) {
        if (deletedSnapshotIds.isEmpty() && invalidatedSnapshotIds.isEmpty() && updatedSnapshots.isEmpty() && (indirectlyModifiedIds == null || indirectlyModifiedIds.isEmpty())) {
            logger.warn("postSnapshotsChangeEvent.. bogus call... no changes.");
            return;
        }
        this.processDeletedIDs(deletedSnapshotIds);
        this.processInvalidatedIDs(invalidatedSnapshotIds);
        Map<ObjectId, DataRow> diffs = this.processUpdatedSnapshots(updatedSnapshots);
        this.sendUpdateNotification(postedBy, diffs, deletedSnapshotIds, invalidatedSnapshotIds, indirectlyModifiedIds);
    }

    private void processDeletedIDs(Collection<ObjectId> deletedSnapshotIDs) {
        if (!deletedSnapshotIDs.isEmpty()) {
            for (ObjectId deletedSnapshotID : deletedSnapshotIDs) {
                this.snapshots.remove(deletedSnapshotID);
            }
        }
    }

    private void processInvalidatedIDs(Collection<ObjectId> invalidatedSnapshotIds) {
        if (!invalidatedSnapshotIds.isEmpty()) {
            for (ObjectId invalidatedSnapshotId : invalidatedSnapshotIds) {
                this.snapshots.remove(invalidatedSnapshotId);
            }
        }
    }

    private Map<ObjectId, DataRow> processUpdatedSnapshots(Map<ObjectId, DataRow> updatedSnapshots) {
        HashMap<ObjectId, DataRow> diffs = null;
        if (!updatedSnapshots.isEmpty()) {
            for (Map.Entry<ObjectId, DataRow> entry : updatedSnapshots.entrySet()) {
                DataRow newSnapshot;
                ObjectId key = entry.getKey();
                DataRow oldSnapshot = this.snapshots.put(key, newSnapshot = entry.getValue());
                if (oldSnapshot == null) continue;
                if (oldSnapshot.getVersion() != newSnapshot.getReplacesVersion()) {
                    if (logger.isDebugEnabled()) {
                        logger.debug("snapshot version changed, don't know what to do... Old: " + oldSnapshot + ", New: " + newSnapshot);
                    }
                    this.forgetSnapshot(key);
                    continue;
                }
                DataRow diff = oldSnapshot.createDiff(newSnapshot);
                if (diff == null) continue;
                if (diffs == null) {
                    diffs = new HashMap<ObjectId, DataRow>();
                }
                diffs.put(key, diff);
            }
        }
        return diffs;
    }

    private void processUpdateDiffs(Map<ObjectId, DataRow> diffs) {
        if (!diffs.isEmpty()) {
            for (Map.Entry<ObjectId, DataRow> entry : diffs.entrySet()) {
                ObjectId key = entry.getKey();
                DataRow oldSnapshot = (DataRow)this.snapshots.remove(key);
                if (oldSnapshot == null) continue;
                DataRow newSnapshot = oldSnapshot.applyDiff(entry.getValue());
                this.snapshots.put(key, newSnapshot);
            }
        }
    }

    private void sendUpdateNotification(Object postedBy, Map<ObjectId, DataRow> diffs, Collection<ObjectId> deletedSnapshotIDs, Collection<ObjectId> invalidatedSnapshotIDs, Collection<ObjectId> indirectlyModifiedIds) {
        if (diffs != null && !diffs.isEmpty() || deletedSnapshotIDs != null && !deletedSnapshotIDs.isEmpty() || invalidatedSnapshotIDs != null && !invalidatedSnapshotIDs.isEmpty() || indirectlyModifiedIds != null && !indirectlyModifiedIds.isEmpty()) {
            SnapshotEvent event = new SnapshotEvent(this, postedBy, diffs, deletedSnapshotIDs, invalidatedSnapshotIDs, indirectlyModifiedIds);
            if (logger.isDebugEnabled()) {
                logger.debug("postSnapshotsChangeEvent: " + event);
            }
            this.eventManager.postEvent(event, this.getSnapshotEventSubject());
        }
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        this.eventSubject = this.createSubject();
    }

    void stopListeners() {
        if (this.eventManager != null) {
            this.eventManager.removeListener(this);
        }
        if (this.remoteNotificationsHandler != null) {
            try {
                this.remoteNotificationsHandler.shutdown();
            }
            catch (Exception ex) {
                logger.info("Exception shutting down EventBridge.", ex);
            }
            this.remoteNotificationsHandler = null;
        }
    }

    void startListeners() {
        if (this.eventManager != null && this.remoteNotificationsHandler != null) {
            try {
                this.eventManager.addNonBlockingListener(this, "processRemoteEvent", SnapshotEvent.class, this.getSnapshotEventSubject(), this.remoteNotificationsHandler);
                this.remoteNotificationsHandler.startup(this.eventManager, 3);
            }
            catch (Exception ex) {
                throw new CayenneRuntimeException("Error initializing DataRowStore.", (Throwable)ex, new Object[0]);
            }
        }
    }
}

