/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import bk-shade.com.google.common.annotations.VisibleForTesting;
import bk-shade.com.google.common.util.concurrent.RateLimiter;
import bk-shade.com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.bookkeeper.bookie.CompactableLedgerStorage;
import org.apache.bookkeeper.bookie.EntryLocation;
import org.apache.bookkeeper.bookie.EntryLogMetadata;
import org.apache.bookkeeper.bookie.EntryLogger;
import org.apache.bookkeeper.bookie.GarbageCollector;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.bookie.ScanAndCompareGarbageCollector;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.util.MathUtils;
import org.apache.bookkeeper.util.SafeRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GarbageCollectorThread
extends SafeRunnable {
    private static final Logger LOG = LoggerFactory.getLogger(GarbageCollectorThread.class);
    private static final int SECOND = 1000;
    private Map<Long, EntryLogMetadata> entryLogMetaMap = new ConcurrentHashMap<Long, EntryLogMetadata>();
    ScheduledExecutorService gcExecutor;
    Future<?> scheduledFuture = null;
    final long gcWaitTime;
    boolean enableMinorCompaction = false;
    final double minorCompactionThreshold;
    final long minorCompactionInterval;
    boolean enableMajorCompaction = false;
    final double majorCompactionThreshold;
    final long majorCompactionInterval;
    final boolean isForceGCAllowWhenNoSpace;
    long lastMinorCompactionTime;
    long lastMajorCompactionTime;
    final boolean isThrottleByBytes;
    final int maxOutstandingRequests;
    final int compactionRateByEntries;
    final int compactionRateByBytes;
    final CompactionScannerFactory scannerFactory;
    final EntryLogger entryLogger;
    final CompactableLedgerStorage ledgerStorage;
    final AtomicBoolean compacting = new AtomicBoolean(false);
    volatile boolean running = true;
    long scannedLogId = 0L;
    final AtomicBoolean forceGarbageCollection = new AtomicBoolean(false);
    final AtomicBoolean suspendMajorCompaction = new AtomicBoolean(false);
    final AtomicBoolean suspendMinorCompaction = new AtomicBoolean(false);
    final GarbageCollector garbageCollector;
    final GarbageCollector.GarbageCleaner garbageCleaner;

    public GarbageCollectorThread(ServerConfiguration conf, LedgerManager ledgerManager, final CompactableLedgerStorage ledgerStorage) throws IOException {
        this.gcExecutor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("GarbageCollectorThread-%d").build());
        this.entryLogger = ledgerStorage.getEntryLogger();
        this.ledgerStorage = ledgerStorage;
        this.gcWaitTime = conf.getGcWaitTime();
        this.isThrottleByBytes = conf.getIsThrottleByBytes();
        this.maxOutstandingRequests = conf.getCompactionMaxOutstandingRequests();
        this.compactionRateByEntries = conf.getCompactionRateByEntries();
        this.compactionRateByBytes = conf.getCompactionRateByBytes();
        this.scannerFactory = new CompactionScannerFactory();
        this.garbageCleaner = new GarbageCollector.GarbageCleaner(){

            @Override
            public void clean(long ledgerId) {
                try {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("delete ledger : " + ledgerId);
                    }
                    ledgerStorage.deleteLedger(ledgerId);
                }
                catch (IOException e) {
                    LOG.error("Exception when deleting the ledger index file on the Bookie: ", (Throwable)e);
                }
            }
        };
        this.garbageCollector = new ScanAndCompareGarbageCollector(ledgerManager, ledgerStorage, conf);
        this.minorCompactionThreshold = conf.getMinorCompactionThreshold();
        this.minorCompactionInterval = conf.getMinorCompactionInterval() * 1000L;
        this.majorCompactionThreshold = conf.getMajorCompactionThreshold();
        this.majorCompactionInterval = conf.getMajorCompactionInterval() * 1000L;
        this.isForceGCAllowWhenNoSpace = conf.getIsForceGCAllowWhenNoSpace();
        if (this.minorCompactionInterval > 0L && this.minorCompactionThreshold > 0.0) {
            if (this.minorCompactionThreshold > 1.0) {
                throw new IOException("Invalid minor compaction threshold " + this.minorCompactionThreshold);
            }
            if (this.minorCompactionInterval <= this.gcWaitTime) {
                throw new IOException("Too short minor compaction interval : " + this.minorCompactionInterval);
            }
            this.enableMinorCompaction = true;
        }
        if (this.majorCompactionInterval > 0L && this.majorCompactionThreshold > 0.0) {
            if (this.majorCompactionThreshold > 1.0) {
                throw new IOException("Invalid major compaction threshold " + this.majorCompactionThreshold);
            }
            if (this.majorCompactionInterval <= this.gcWaitTime) {
                throw new IOException("Too short major compaction interval : " + this.majorCompactionInterval);
            }
            this.enableMajorCompaction = true;
        }
        if (this.enableMinorCompaction && this.enableMajorCompaction && (this.minorCompactionInterval >= this.majorCompactionInterval || this.minorCompactionThreshold >= this.majorCompactionThreshold)) {
            throw new IOException("Invalid minor/major compaction settings : minor (" + this.minorCompactionThreshold + ", " + this.minorCompactionInterval + "), major (" + this.majorCompactionThreshold + ", " + this.majorCompactionInterval + ")");
        }
        LOG.info("Minor Compaction : enabled=" + this.enableMinorCompaction + ", threshold=" + this.minorCompactionThreshold + ", interval=" + this.minorCompactionInterval);
        LOG.info("Major Compaction : enabled=" + this.enableMajorCompaction + ", threshold=" + this.majorCompactionThreshold + ", interval=" + this.majorCompactionInterval);
        this.lastMinorCompactionTime = this.lastMajorCompactionTime = MathUtils.now();
    }

    public void enableForceGC() {
        if (this.forceGarbageCollection.compareAndSet(false, true)) {
            LOG.info("Forced garbage collection triggered by thread: {}", (Object)Thread.currentThread().getName());
            this.triggerGC();
        }
    }

    public void disableForceGC() {
        if (this.forceGarbageCollection.compareAndSet(true, false)) {
            LOG.info("{} disabled force garbage collection since bookie has enough space now.", (Object)Thread.currentThread().getName());
        }
    }

    Future<?> triggerGC() {
        return this.gcExecutor.submit(this);
    }

    public void suspendMajorGC() {
        if (this.suspendMajorCompaction.compareAndSet(false, true)) {
            LOG.info("Suspend Major Compaction triggered by thread: {}", (Object)Thread.currentThread().getName());
        }
    }

    public void resumeMajorGC() {
        if (this.suspendMajorCompaction.compareAndSet(true, false)) {
            LOG.info("{} Major Compaction back to normal since bookie has enough space now.", (Object)Thread.currentThread().getName());
        }
    }

    public void suspendMinorGC() {
        if (this.suspendMinorCompaction.compareAndSet(false, true)) {
            LOG.info("Suspend Minor Compaction triggered by thread: {}", (Object)Thread.currentThread().getName());
        }
    }

    public void resumeMinorGC() {
        if (this.suspendMinorCompaction.compareAndSet(true, false)) {
            LOG.info("{} Minor Compaction back to normal since bookie has enough space now.", (Object)Thread.currentThread().getName());
        }
    }

    public void start() {
        if (this.scheduledFuture != null) {
            this.scheduledFuture.cancel(false);
        }
        this.scheduledFuture = this.gcExecutor.scheduleAtFixedRate(this, this.gcWaitTime, this.gcWaitTime, TimeUnit.MILLISECONDS);
    }

    @Override
    public void safeRun() {
        boolean force = this.forceGarbageCollection.get();
        if (force) {
            LOG.info("Garbage collector thread forced to perform GC before expiry of wait time.");
        }
        this.entryLogMetaMap = this.extractMetaFromEntryLogs(this.entryLogMetaMap);
        this.doGcLedgers();
        this.doGcEntryLogs();
        boolean suspendMajor = this.suspendMajorCompaction.get();
        boolean suspendMinor = this.suspendMinorCompaction.get();
        if (suspendMajor) {
            LOG.info("Disk almost full, suspend major compaction to slow down filling disk.");
        }
        if (suspendMinor) {
            LOG.info("Disk full, suspend minor compaction to slow down filling disk.");
        }
        long curTime = MathUtils.now();
        if (this.enableMajorCompaction && !suspendMajor && (force || curTime - this.lastMajorCompactionTime > this.majorCompactionInterval)) {
            LOG.info("Enter major compaction, suspendMajor {}", (Object)suspendMajor);
            this.doCompactEntryLogs(this.majorCompactionThreshold);
            this.lastMinorCompactionTime = this.lastMajorCompactionTime = MathUtils.now();
            this.forceGarbageCollection.set(false);
            return;
        }
        if (this.enableMinorCompaction && !suspendMinor && (force || curTime - this.lastMinorCompactionTime > this.minorCompactionInterval)) {
            LOG.info("Enter minor compaction, suspendMinor {}", (Object)suspendMinor);
            this.doCompactEntryLogs(this.minorCompactionThreshold);
            this.lastMinorCompactionTime = MathUtils.now();
        }
        this.forceGarbageCollection.set(false);
    }

    private void doGcLedgers() {
        this.garbageCollector.gc(this.garbageCleaner);
    }

    private void doGcEntryLogs() {
        for (Map.Entry<Long, EntryLogMetadata> entry : this.entryLogMetaMap.entrySet()) {
            long entryLogId = entry.getKey();
            EntryLogMetadata meta = entry.getValue();
            for (Long entryLogLedger : meta.getLedgersMap().keySet()) {
                try {
                    if (this.ledgerStorage.ledgerExists(entryLogLedger)) continue;
                    meta.removeLedger(entryLogLedger);
                }
                catch (IOException e) {
                    LOG.error("Error reading from ledger storage", (Throwable)e);
                }
            }
            if (!meta.isEmpty()) continue;
            LOG.info("Deleting entryLogId " + entryLogId + " as it has no active ledgers!");
            this.removeEntryLog(entryLogId);
        }
    }

    @VisibleForTesting
    void doCompactEntryLogs(double threshold) {
        EntryLogMetadata meta;
        LOG.info("Do compaction to compact those files lower than " + threshold);
        Comparator<EntryLogMetadata> sizeComparator = new Comparator<EntryLogMetadata>(){

            @Override
            public int compare(EntryLogMetadata m1, EntryLogMetadata m2) {
                long unusedSize2;
                long unusedSize1 = m1.getTotalSize() - m1.getRemainingSize();
                if (unusedSize1 > (unusedSize2 = m2.getTotalSize() - m2.getRemainingSize())) {
                    return -1;
                }
                if (unusedSize1 < unusedSize2) {
                    return 1;
                }
                return 0;
            }
        };
        ArrayList<EntryLogMetadata> logsToCompact = new ArrayList<EntryLogMetadata>();
        logsToCompact.addAll(this.entryLogMetaMap.values());
        Collections.sort(logsToCompact, sizeComparator);
        Iterator iterator = logsToCompact.iterator();
        while (iterator.hasNext() && !((meta = (EntryLogMetadata)iterator.next()).getUsage() >= threshold)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Compacting entry log {} below threshold {}", (Object)meta.getEntryLogId(), (Object)threshold);
            }
            try {
                this.compactEntryLog(this.scannerFactory, meta);
                this.scannerFactory.flush();
                LOG.info("Removing entry log {} after compaction", (Object)meta.getEntryLogId());
                this.removeEntryLog(meta.getEntryLogId());
            }
            catch (LedgerDirsManager.NoWritableLedgerDirException nwlde) {
                LOG.warn("No writable ledger directory available, aborting compaction", (Throwable)nwlde);
                break;
            }
            catch (IOException ioe) {
                LOG.error("Error compacting entry log. Log won't be deleted", (Throwable)ioe);
            }
            if (this.running) continue;
            return;
        }
    }

    public void shutdown() throws InterruptedException {
        this.running = false;
        LOG.info("Shutting down GarbageCollectorThread");
        while (!this.compacting.compareAndSet(false, true)) {
            Thread.sleep(100L);
        }
        this.gcExecutor.shutdownNow();
    }

    private void removeEntryLog(long entryLogId) {
        if (this.entryLogger.removeEntryLog(entryLogId)) {
            this.entryLogMetaMap.remove(entryLogId);
        }
    }

    protected void compactEntryLog(CompactionScannerFactory scannerFactory, EntryLogMetadata entryLogMeta) throws IOException {
        if (!this.compacting.compareAndSet(false, true)) {
            return;
        }
        LOG.info("Compacting entry log : {} - Usage: {} %", (Object)entryLogMeta.getEntryLogId(), (Object)entryLogMeta.getUsage());
        try {
            this.entryLogger.scanEntryLog(entryLogMeta.getEntryLogId(), scannerFactory.newScanner(entryLogMeta));
        }
        finally {
            this.compacting.set(false);
        }
    }

    protected Map<Long, EntryLogMetadata> extractMetaFromEntryLogs(Map<Long, EntryLogMetadata> entryLogMetaMap) {
        long curLogId = this.entryLogger.getLeastUnflushedLogId();
        boolean hasExceptionWhenScan = false;
        for (long entryLogId = this.scannedLogId; entryLogId < curLogId; ++entryLogId) {
            if (entryLogMetaMap.containsKey(entryLogId) || !this.entryLogger.logExists(entryLogId)) continue;
            LOG.info("Extracting entry log meta from entryLogId: {}", (Object)entryLogId);
            try {
                EntryLogMetadata entryLogMeta = this.entryLogger.getEntryLogMetadata(entryLogId);
                entryLogMetaMap.put(entryLogId, entryLogMeta);
            }
            catch (IOException e) {
                hasExceptionWhenScan = true;
                LOG.warn("Premature exception when processing " + entryLogId + " recovery will take care of the problem", (Throwable)e);
            }
            if (hasExceptionWhenScan) continue;
            ++this.scannedLogId;
        }
        return entryLogMetaMap;
    }

    class CompactionScannerFactory {
        List<EntryLocation> offsets = new ArrayList<EntryLocation>();

        CompactionScannerFactory() {
        }

        EntryLogger.EntryLogScanner newScanner(final EntryLogMetadata meta) {
            final Throttler throttler = new Throttler(GarbageCollectorThread.this.isThrottleByBytes, GarbageCollectorThread.this.compactionRateByBytes, GarbageCollectorThread.this.compactionRateByEntries);
            return new EntryLogger.EntryLogScanner(){

                @Override
                public boolean accept(long ledgerId) {
                    return meta.containsLedger(ledgerId);
                }

                @Override
                public void process(long ledgerId, long offset, ByteBuffer entry) throws IOException {
                    throttler.acquire(entry.remaining());
                    if (CompactionScannerFactory.this.offsets.size() > GarbageCollectorThread.this.maxOutstandingRequests) {
                        CompactionScannerFactory.this.flush();
                    }
                    entry.getLong();
                    long entryId = entry.getLong();
                    entry.rewind();
                    long newoffset = GarbageCollectorThread.this.entryLogger.addEntry(ledgerId, entry);
                    CompactionScannerFactory.this.offsets.add(new EntryLocation(ledgerId, entryId, newoffset));
                }
            };
        }

        void flush() throws IOException {
            if (this.offsets.isEmpty()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Skipping entry log flushing, as there are no offset!");
                }
                return;
            }
            try {
                GarbageCollectorThread.this.entryLogger.flush();
                GarbageCollectorThread.this.ledgerStorage.updateEntriesLocations(this.offsets);
            }
            finally {
                this.offsets.clear();
            }
        }
    }

    private static class Throttler {
        final RateLimiter rateLimiter;
        final boolean isThrottleByBytes;
        final int compactionRateByBytes;
        final int compactionRateByEntries;

        Throttler(boolean isThrottleByBytes, int compactionRateByBytes, int compactionRateByEntries) {
            this.isThrottleByBytes = isThrottleByBytes;
            this.compactionRateByBytes = compactionRateByBytes;
            this.compactionRateByEntries = compactionRateByEntries;
            this.rateLimiter = RateLimiter.create(this.isThrottleByBytes ? (double)this.compactionRateByBytes : (double)this.compactionRateByEntries);
        }

        void acquire(int permits) {
            this.rateLimiter.acquire(this.isThrottleByBytes ? permits : 1);
        }
    }
}

