/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.beam.sdk.io.gcp.bigtable.changestreams.dofn;

import com.google.cloud.bigtable.data.v2.models.ChangeStreamMutation;
import com.google.protobuf.ByteString;
import java.io.IOException;
import java.math.BigDecimal;
import org.apache.beam.sdk.annotations.Internal;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.ChangeStreamMetrics;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.action.ActionFactory;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.action.ChangeStreamAction;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.action.ReadChangeStreamPartitionAction;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.dao.ChangeStreamDao;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.dao.DaoFactory;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.dao.MetadataTableDao;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.estimator.BytesThroughputEstimator;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.estimator.NullThroughputEstimator;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.estimator.ThroughputEstimator;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.model.PartitionRecord;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.restriction.ReadChangeStreamPartitionProgressTracker;
import org.apache.beam.sdk.io.gcp.bigtable.changestreams.restriction.StreamProgress;
import org.apache.beam.sdk.transforms.DoFn;
import org.apache.beam.sdk.transforms.DoFn.UnboundedPerElement;
import org.apache.beam.sdk.transforms.splittabledofn.ManualWatermarkEstimator;
import org.apache.beam.sdk.transforms.splittabledofn.RestrictionTracker;
import org.apache.beam.sdk.transforms.splittabledofn.WatermarkEstimators.Manual;
import org.apache.beam.sdk.values.KV;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

// Allows for readChangeStreamPartitionAction setup
@SuppressWarnings("initialization.fields.uninitialized")
@Internal
@UnboundedPerElement
public class ReadChangeStreamPartitionDoFn
    extends DoFn<PartitionRecord, KV<ByteString, ChangeStreamMutation>> {
  private static final long serialVersionUID = 4418739381635104479L;
  private static final BigDecimal MAX_DOUBLE = BigDecimal.valueOf(Double.MAX_VALUE);
  private static final Logger LOG = LoggerFactory.getLogger(ReadChangeStreamPartitionDoFn.class);
  public static final int THROUGHPUT_ESTIMATION_WINDOW_SECONDS = 10;

  private final Duration heartbeatDuration;
  private final DaoFactory daoFactory;
  private final ChangeStreamMetrics metrics;
  private final ActionFactory actionFactory;
  private ThroughputEstimator<KV<ByteString, ChangeStreamMutation>> throughputEstimator;

  private ReadChangeStreamPartitionAction readChangeStreamPartitionAction;

  public ReadChangeStreamPartitionDoFn(
      Duration heartbeatDuration,
      DaoFactory daoFactory,
      ActionFactory actionFactory,
      ChangeStreamMetrics metrics) {
    this.heartbeatDuration = heartbeatDuration;
    this.daoFactory = daoFactory;
    this.metrics = metrics;
    this.actionFactory = actionFactory;
    this.throughputEstimator = new NullThroughputEstimator<>();
  }

  @GetInitialWatermarkEstimatorState
  public Instant getInitialWatermarkEstimatorState(@Element PartitionRecord partitionRecord) {
    return partitionRecord.getParentLowWatermark();
  }

  @NewWatermarkEstimator
  public ManualWatermarkEstimator<Instant> newWatermarkEstimator(
      @WatermarkEstimatorState Instant watermarkEstimatorState) {
    return new Manual(watermarkEstimatorState);
  }

  @GetInitialRestriction
  public StreamProgress initialRestriction() {
    metrics.incPartitionStreamCount();
    return new StreamProgress();
  }

  @NewTracker
  public ReadChangeStreamPartitionProgressTracker restrictionTracker(
      @Restriction StreamProgress restriction) {
    return new ReadChangeStreamPartitionProgressTracker(restriction);
  }

  @GetSize
  public double getSize(@Restriction StreamProgress streamProgress) {
    if (streamProgress == null) {
      return 0d;
    }
    Instant lowWatermark = streamProgress.getEstimatedLowWatermark();
    // This should only be null if:
    // 1) We've failed to lock for the partition in which case we expect 0 throughput
    // 2) We've received a CloseStream in which case we won't process any more data for
    //    this partition
    // 3) RCSP has just started and hasn't completed a checkpoint yet, in which case we can't
    //    estimate throughput yet
    if (lowWatermark == null) {
      return 0;
    }

    Duration watermarkLag = Duration.millis(Instant.now().getMillis() - lowWatermark.getMillis());
    BigDecimal estimatedThroughput = BigDecimal.valueOf(throughputEstimator.get());
    // Return the estimated bytes per second throughput multiplied by the amount of known work
    // outstanding (watermark lag). Cap at max double to avoid overflow.
    final double estimatedSize =
        estimatedThroughput
            .multiply(BigDecimal.valueOf(watermarkLag.getStandardSeconds()))
            .min(MAX_DOUBLE)
            .doubleValue();
    LOG.debug(
        "Estimated size: throughputBytes: {} x watermarkLag {} = {}",
        estimatedThroughput,
        watermarkLag,
        estimatedSize);
    return estimatedSize;
  }

  @Setup
  public void setup() throws IOException {
    MetadataTableDao metadataTableDao = daoFactory.getMetadataTableDao();
    ChangeStreamDao changeStreamDao = daoFactory.getChangeStreamDao();
    ChangeStreamAction changeStreamAction =
        actionFactory.changeStreamAction(this.metrics, this.throughputEstimator);
    readChangeStreamPartitionAction =
        actionFactory.readChangeStreamPartitionAction(
            metadataTableDao, changeStreamDao, metrics, changeStreamAction, heartbeatDuration);
  }

  @ProcessElement
  public ProcessContinuation processElement(
      @Element PartitionRecord partitionRecord,
      RestrictionTracker<StreamProgress, StreamProgress> tracker,
      OutputReceiver<KV<ByteString, ChangeStreamMutation>> receiver,
      ManualWatermarkEstimator<Instant> watermarkEstimator)
      throws InterruptedException, IOException {
    return readChangeStreamPartitionAction.run(
        partitionRecord, tracker, receiver, watermarkEstimator);
  }

  /**
   * Sets the estimator to track throughput for each DoFn instance.
   *
   * @param throughputEstimator an estimator to calculate DoFn instance level throughput
   */
  public void setThroughputEstimator(
      BytesThroughputEstimator<KV<ByteString, ChangeStreamMutation>> throughputEstimator) {
    this.throughputEstimator = throughputEstimator;
  }
}
