/*
 * Copyright (c) 1997 Erez Zadok
 * Copyright (c) 1989 Jan-Simon Pendry
 * Copyright (c) 1989 Imperial College of Science, Technology & Medicine
 * Copyright (c) 1989 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Jan-Simon Pendry at Imperial College, London.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      %W% (Berkeley) %G%
 *
 * $Id: info_ldap.c,v 5.2.2.1 1992/02/09 15:08:29 jsp beta $
 *
 */

/*
 * Get info from LDAP (Lightweight Directory Access Protocol)
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <am_defs.h>
#include <amd.h>

/*
 * MACROS:
 */
#define AMD_LDAP_PREFIX		"ldap:"
#define AMD_LDAP_PREFLEN	5
/* Time to live for an LDAP cached in an mnt_map */
#define AMD_LDAP_TTL		3600
#define AMD_LDAP_HOST		"ldap"

/* Should not be necessary */
#ifndef LDAP_PORT
# define LDAP_PORT		389
#endif /* LDAP_PORT */

#define AMDMAPVALUE		"amdmapValue"
#define AMDMAPTIMESTAMP		"amdmapTimestamp"

/*
 * STATICS:
 */
static char *tsattr[] = {AMDMAPTIMESTAMP, NULL};
static char *valattr[] = {AMDMAPVALUE, NULL};

/*
 * TYPEDEFS:
 */
typedef struct ald_ent ALD;
typedef struct cr_ent CR;
typedef struct he_ent HE;

/*
 * STRUCTURES:
 */
struct ald_ent {
  LDAP *ldap;
  HE *hostent;
  CR *credentials;
  time_t timestamp;
};

struct cr_ent {
  char *who;
  char *pw;
  int method;
};

struct he_ent {
  char *host;
  int port;
  struct he_ent *next;
};


/*
 * FORWARD DECLARATIONS:
 */
static int amu_ldap_rebind(ALD * a);
static int get_ldap_timestamp(LDAP * ld, char *map, time_t *ts);


/*
 * FUNCTIONS:
 */

static void
he_free(HE * h)
{
  free(h->host);
  if (h->next != NULL)
    he_free(h->next);
  free(h);
}

static HE *
string2he(char *s)
{
  char *c, *p;
  HE *new, *old = NULL;

  if (s == NULL)
    return (NULL);
  for (p = s; p; p = strchr(p, ',')) {
    if (old != NULL) {
      new = (HE *) malloc(sizeof(HE));
      old->next = new;
      old = new;
    } else {
      old = (HE *) malloc(sizeof(HE));
      old->next = NULL;
    }
    c = strchr(p, ':');
    if (c) {			/* Host and port */
      *c++ = '\0';
      old->host = strdup(p);
      old->port = atoi(c);
    } else
      old->host = strdup(p);

  }
  return (old);
}


static void
cr_free(CR * c)
{
  free(c->who);
  free(c->pw);
  free(c);
}


static void
ald_free(ALD * a)
{
  he_free(a->hostent);
  cr_free(a->credentials);
  ldap_unbind(a->ldap);
  free(a);
}


int
amu_ldap_init(mnt_map *m, char *map, time_t *ts)
{
  ALD *aldh;
  CR *creds;

  if (strncmp(map, AMD_LDAP_PREFIX, AMD_LDAP_PREFLEN) == 0)
    return (ENOENT);
#ifdef DEBUG
  else
    dlog("%s seems to be ldap\n", map);
#endif /* DEBUG */

  aldh = (ALD *) malloc(sizeof(ALD));
  creds = (CR *) malloc(sizeof(CR));

  aldh->hostent = string2he(gopt.ldap_hostports);
  if (aldh->hostent == NULL) {
#ifdef DEBUG
    dlog("Unable to decode %s for ldap map %s", gopt.ldap_hostports, map);
#endif /* DEBUG */
    return (ENOENT);
  }
  creds->who = "";
  creds->pw = "";
  creds->method = LDAP_AUTH_SIMPLE;
  aldh->credentials = creds;
  aldh->timestamp = 0;

  if (amu_ldap_rebind(aldh)) {
    ald_free(aldh);
    return (ENOENT);
  }
  if (get_ldap_timestamp(aldh->ldap, map, ts)) {
    ald_free(aldh);
    return (ENOENT);
  }
  m->map_data = (void *) aldh;
  return (0);
}


static int
amu_ldap_rebind(ALD * a)
{
  LDAP *ld;
  HE *h;
  CR *c = a->credentials;
  time_t now = clocktime();

  if (a->ldap != NULL) {
    if ((a->timestamp - now) > AMD_LDAP_TTL) {
      ldap_unbind(a->ldap);
      a->timestamp = now;
    } else
      return (0);
  }
  for (h = a->hostent; h != NULL; h = h->next) {
    if ((ld = ldap_open(h->host, h->port)) == NULL) {
#ifdef DEBUG
      dlog("Unable to ldap_open to %s:%d\n", h->host, h->port);
#endif /* DEBUG */
      continue;
    }
    if (ldap_bind_s(ld, c->who, c->pw, c->method) != LDAP_SUCCESS) {
#ifdef DEBUG
      dlog("Unable to ldap_bind to %s:%d as %s\n", h->host, h->port, c->who);
#endif /* DEBUG */
      continue;
    }
    a->ldap = ld;
    a->timestamp = now;
    return (0);
  }
#ifdef DEBUG
  dlog("Unable to rebind\n");
#endif /* DEBUG */

  return (ENOENT);
}


static int
get_ldap_timestamp(LDAP * ld, char *map, time_t *ts)
{
  struct timeval tv;
  char *p, **vals;
  char filter[MAXPATHLEN];
  int err;
  LDAPMessage *res, *entry;

  p = map + AMD_LDAP_PREFLEN;

  tv.tv_sec = 2;
  tv.tv_usec = 0;
  sprintf(filter, "(&(objectClass=amdmap)(amdmapName=%s))", p);
  err = ldap_search_st(ld,
		       gopt.ldap_base,
		       LDAP_SCOPE_SUBTREE,
		       filter,
		       tsattr,
		       0,
		       &tv,
		       &res);

  if (err != LDAP_SUCCESS) {
    *ts = 0;
#ifdef DEBUG
    dlog("LDAP timestamp search failed: %s\n", ldap_err2string(ld->ld_errno));
#endif /* DEBUG */
    return (ENOENT);
  }
  entry = ldap_first_entry(ld, res);
  vals = ldap_get_values(ld, entry, AMDMAPTIMESTAMP);
  if (vals[0])
    *ts = atoi(vals[0]);
  if (*ts == 0) {
#ifdef DEBUG
    dlog("Unable to decode ldap timestamp %s for map %s\n", vals[0], map);
#endif /* DEBUG */
    err = ENOENT;
  } else {
    *ts = 0;
#ifdef DEBUG
    dlog("Unable to find ldap timestamp for map %s\n", map);
#endif /* DEBUG */
    err = ENOENT;
  }
  ldap_value_free(vals);
  ldap_msgfree(res);
  ldap_msgfree(entry);

  return (err);
}


int
amu_ldap_search(mnt_map *m, char *map, char *key, char **pval, time_t *ts)
{
  char *p, **vals, filter[MAXPATHLEN];
  struct timeval tv;
  int err;
  LDAPMessage *entry, *res;
  ALD *a = (ALD *) (m->map_data);

  tv.tv_sec = 2;
  tv.tv_usec = 0;
  if (a == NULL) {
#ifdef DEBUG
    dlog("LDAP panic: unable to find map data\n");
#endif /* DEBUG */
    return (ENOENT);
  }
  if (amu_ldap_rebind(a))	/* Check that's the handle is still valid */
    return (ENOENT);

  p = map + AMD_LDAP_PREFLEN;
  sprintf(filter, "(&(objectClass=amdmap)(amdmapName=%s)(amdmapKey=%s))", p, key);

 err = ldap_search_st(a->ldap,
		      gopt.ldap_base,
		      LDAP_SCOPE_SUBTREE,
		      filter,
		      valattr,
		      0,
		      &tv,
		      &res);
  if (err != LDAP_SUCCESS) {
#ifdef DEBUG
    dlog("LDAP search failed: %s\n", ldap_err2string(a->ldap->ld_errno));
#endif /* DEBUG */
    return (ENOENT);
  }

  entry = ldap_first_entry(a->ldap, res);
  vals = ldap_get_values(a->ldap, entry, AMDMAPVALUE);
  if (vals[0]) {
    *pval = strdup(vals[0]);
    err = 0;
  } else
    err = ENOENT;
  ldap_msgfree(res);
  ldap_msgfree(entry);
  ldap_value_free(vals);

  return (err);
}


int
amu_ldap_mtime(mnt_map *m, char *map, time_t *ts)
{
  ALD *aldh = (ALD *) (m->map_data);

  if (aldh == NULL) {
#ifdef DEBUG
    dlog("LDAP panic: unable to find map data\n");
#endif /* DEBUG */
    return (ENOENT);
  }
  if (amu_ldap_rebind(aldh)) {
    return (ENOENT);
  }
  if (get_ldap_timestamp(aldh->ldap, map, ts)) {
    return (ENOENT);
  }
  return (0);
}
